/*
    filterlist.cpp - Show System, Country, Editor filter list.

    Copyright (c) 2005      by Michaël Larouche       <michael.larouche@kdemail.net>

    *************************************************************************
    *                                                                       *
    * This program is free software; you can redistribute it and/or modify  *
    * it under the terms of the GNU General Public License as published by  *
    * the Free Software Foundation; either version 2 of the License, or     *
    * (at your option) any later version.                                   *
    *                                                                       *
    *************************************************************************
*/
#include "filterlist.h"

// Qt includes
#include <qlayout.h>
#include <qlabel.h>
#include <qstringlist.h>
#include <qtimer.h>

// KDE includes
#include <kdebug.h>
#include <klocale.h>

#include "filterlistwidget.h"

// Kamefu includes
#include "collectionaccess.h"
#include "romquerybuilder.h"

class FilterList::Private
{
public:
	Private()
	 : systemFilter(0), countryFilter(0), editorFilter(0)
	{}

	FilterListWidget *systemFilter;
	FilterListWidget *countryFilter;
	FilterListWidget *editorFilter;
	
	QString selectedSystem;
	QString selectedCountry;
	QString selectedEditor;

	QString latestQuery;
};

FilterList::FilterList(QWidget *parent, const char *name)
	: QWidget(parent, name), d(new Private)
{
	QHBoxLayout *layout = new QHBoxLayout(this);
	
	d->systemFilter = new FilterListWidget(i18n("System"), this);
	d->countryFilter = new FilterListWidget(i18n("Country"), this);
	d->editorFilter = new FilterListWidget(i18n("Developer"), this);
	d->selectedSystem = QString::null;
	d->selectedCountry = QString::null;
	d->selectedEditor = QString::null;

	layout->addWidget(d->systemFilter);
	layout->addWidget( new QLabel("<b><font size=\"+2\">></font></b>", this) );
	layout->addWidget(d->countryFilter);
	layout->addWidget( new QLabel("<b><font size=\"+2\">></font></b>", this) );
	layout->addWidget(d->editorFilter);

	connectSignals();
}


FilterList::~FilterList()
{
	delete d;
}

void FilterList::connectSignals()
{
	connect(d->systemFilter, SIGNAL(selectionChanged(const QString&)), this, SLOT(systemChanged(const QString &)));
	connect(d->countryFilter, SIGNAL(selectionChanged(const QString&)), this, SLOT(countryChanged(const QString &)));
	connect(d->editorFilter, SIGNAL(selectionChanged(const QString&)), this, SLOT(editorChanged(const QString &)));
}

void FilterList::disconnectSignals()
{
	disconnect(d->systemFilter, SIGNAL(selectionChanged(const QString&)), this, SLOT(systemChanged(const QString &)));
	disconnect(d->countryFilter, SIGNAL(selectionChanged(const QString&)), this, SLOT(countryChanged(const QString &)));
	disconnect(d->editorFilter, SIGNAL(selectionChanged(const QString&)), this, SLOT(editorChanged(const QString &)));
}

void FilterList::fillFilterLists()
{
	// Disable signal magic to emit the filter query
	disconnectSignals();

	// Backup current selected items.
	d->selectedSystem = d->systemFilter->selection();
	d->selectedCountry = d->countryFilter->selection();
	d->selectedCountry = d->editorFilter->selection();

	// Clear all the filters.
	d->systemFilter->clear();
	d->countryFilter->clear();
	d->editorFilter->clear();

	// Retrieve content of system, country and editor and fill the listboxes
	QStringList systemList, countryList, editorList;
	systemList = Kamefu::CollectionAccess::self()->retrieveTableEntries("system");
	countryList = Kamefu::CollectionAccess::self()->retrieveTableEntries("country");
	editorList = Kamefu::CollectionAccess::self()->retrieveTableEntries("editor");

	QStringList::ConstIterator systemListIt, systemListItEnd, countryListIt, countryListItEnd, editorListIt, editorListItEnd;
	
	systemListItEnd = systemList.constEnd();
	for(systemListIt = systemList.constBegin(); systemListIt != systemListItEnd; ++systemListIt)
	{
		QString tempSystem = *systemListIt;
		d->systemFilter->insertItem(tempSystem);
	}

	countryListItEnd = countryList.constEnd();
	for(countryListIt = countryList.constBegin(); countryListIt != countryListItEnd; ++countryListIt)
	{
		QString tempCountry = *countryListIt;
		d->countryFilter->insertItem(tempCountry);
	}
	editorListItEnd = editorList.constEnd();
	for(editorListIt = editorList.constBegin(); editorListIt != editorListItEnd; ++editorListIt)
	{
		QString tempEditor = *editorListIt;
		d->editorFilter->insertItem(tempEditor);
	}

	// Sort the filter
	d->systemFilter->sort();
	d->editorFilter->sort();
	d->countryFilter->sort();

	// Restore the selected items
	d->systemFilter->select( d->selectedSystem );
	d->countryFilter->select( d->selectedCountry );
	d->editorFilter->select( d->selectedEditor );

	// Make sure the filter are correctly reseted
	systemChanged( d->systemFilter->selection() );
	countryChanged( d->countryFilter->selection() );
	// Emit the new ROM query too.
	editorChanged( d->editorFilter->selection() );

	// Re-enable signal magic to emit the filter query.
	connectSignals();
}

void FilterList::systemChanged(const QString &system)
{
	d->selectedSystem = system;
	fillCountries(system);
}

void FilterList::countryChanged(const QString &country)
{
	d->selectedCountry = country;
	fillEditors(country);
}

void FilterList::editorChanged(const QString &editor)
{
	d->selectedEditor = editor;
	filterRoms();
}

void FilterList::fillCountries(const QString &system)
{
	QString selectedCountry = d->countryFilter->selection();
	d->countryFilter->clear();
	QStringList countryList;

	if( system.isEmpty() )
	{
		countryList = Kamefu::CollectionAccess::self()->retrieveTableEntries( "country" );
	}
	else
	{
		countryList = Kamefu::CollectionAccess::self()->retrieveTableEntriesFiltered( "country", QString("country_id IN (SELECT country_id FROM rom WHERE system_id=(SELECT system_id FROM system WHERE system_name = '%1'))").arg(system) );
	}
	
	QStringList::ConstIterator countryListIt, countryListItEnd;
	
	countryListItEnd = countryList.constEnd();
	for(countryListIt = countryList.constBegin(); countryListIt != countryListItEnd; ++countryListIt)
	{
		QString tempCountry = *countryListIt;
		d->countryFilter->insertItem(tempCountry);
	}
	
	d->countryFilter->sort();
	d->countryFilter->select(selectedCountry);
}

void FilterList::fillEditors(const QString &country)
{
	QString selectedEditor = d->editorFilter->selection();
	d->editorFilter->clear();
	QStringList editorList;
	
	QString systemFilter = QString::null;
	if( !d->selectedSystem.isEmpty() )
	{
		systemFilter = QString("editor_id IN (SELECT editor_id FROM rom WHERE system_id=(SELECT system_id FROM system WHERE system_name = '%1'))").arg(d->selectedSystem);
	}

	if( country.isEmpty() && systemFilter.isEmpty() )
	{
		editorList = Kamefu::CollectionAccess::self()->retrieveTableEntries( "editor" );
	}
	else
	{
		QString filter = systemFilter;
		
		if( !country.isEmpty() )
		{
			if( !filter.isEmpty() )
			{
				filter += " and ";
			}
			filter += QString("editor_id IN (SELECT editor_id FROM rom WHERE country_id=(SELECT country_id FROM country WHERE country_name = '%1'))").arg(country);
		}
		editorList = Kamefu::CollectionAccess::self()->retrieveTableEntriesFiltered( "editor", filter );
	}
	
	QStringList::ConstIterator editorListIt, editorListItEnd;
	
	editorListItEnd = editorList.constEnd();
	for(editorListIt = editorList.constBegin(); editorListIt != editorListItEnd; ++editorListIt)
	{
		QString tempEditor = *editorListIt;
		d->editorFilter->insertItem(tempEditor);
	}
	
	d->editorFilter->sort();
	d->editorFilter->select(selectedEditor);
}

void FilterList::filterRoms()
{
	Kamefu::RomQueryBuilder queryBuilder;
	if( !d->selectedSystem.isEmpty() )
	{
		queryBuilder.addFilter("system", d->selectedSystem);
	}
	if( !d->selectedCountry.isEmpty() )
	{
		queryBuilder.addFilter("country", d->selectedCountry);
	}
	if( !d->selectedEditor.isEmpty() )
	{
		queryBuilder.addFilter("editor", d->selectedEditor);
	}

	QString currentQuery = queryBuilder.sqlQuery();
	if( d->latestQuery.isEmpty() || d->latestQuery != currentQuery )
	{
		kdDebug() << k_funcinfo << "Emitting new ROM query..." << endl;
		d->latestQuery = currentQuery;
		emit refreshRoms(currentQuery);
	}
}

#include "filterlist.moc"
