/*
 * encodings.cc
 * This file is part of katoob
 *
 * Copyright (C) 2006 Mohammed Sameer
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <iostream>
#include <cassert>
#include "encodings.hh"

#if 0
Encoding arabic[] = {
  {"ISO 8859-6 (Arabic)", "ISO_8859-6"},
  {"WINDOWS-1256 (Arabic)", "WINDOWS-1256"},
};

Encoding baltic[] = {
  {"ISO_8859-4 (Baltic)", "ISO_8859-4"},
  {"ISO_8859-13 (Baltic)", "ISO_8859-13"},
  {"WINDOWS-1257 (Baltic)", "WINDOWS-1257"},
};

Encoding central_european[] = {
  {"ISO 8859-2 (Central European)", "ISO_8859-2"},
  {"WINDOWS-1250 (Central European)", "WINDOWS-1250"},
};

Encoding cyrillic[] = {
  {"ISO_8859-5 (Cyrillic)", "ISO_8859-5"},
  {"WINDOWS-1251 (Cyrillic)", "WINDOWS-1251"},
};

Encoding greek[] = {
  {"ISO 8859-7 (Greek)", "ISO_8859-7"},
  {"WINDOWS-1253 (Greek)", "WINDOWS-1253"},
};

Encoding hebrew[] = {
  {"ISO 8859-8-i (Hebrew - logical ordering)", "ISO_8859-8"},
  {"WINDOWS-1255 (Hebrew)", "WINDOWS-1255"},
};

Encoding japanese[] = {
  {"EUC-JP (Japanese)", "EUC-JP"},
  {"SHIFT_JIS (Japanese)", "SHIFT_JIS"},
  {"ISO-2022-JP (Japanese)", "ISO-2022-JP"},
};

Encoding romanian[] = {
  {"ISO_8859-16 (Romanian)", "ISO-8859-16"},
};

Encoding turkish[] = {
  {"ISO 8859-9 (Turkish)", "ISO_8859-9"},
  {"WINDOWS-1254 (Turkish)", "WINDOWS-1254"},
};

Encoding western[] = {
  {"ISO 8859-1 (Western European)", "ISO_8859-1"},
  {"ISO_8859-15 (Western, New)", "ISO_8859-15"},
  {"WINDOWS-1252 (Western)", "WINDOWS-1252"},
};

Encoding other[] = {
  {"ISO_8859-3 (South European)", "ISO_8859-3"},
  {"ISO_8859-10 (Nordic)", "ISO_8859-10"},
  {"ISO_8859-11", "ISO_8859-11"},
  {"ISO_8859-12", "ISO_8859-12"},
  {"ISO_8859-14", "ISO_8859-14"},
  {"WINDOWS-1258 (Vietnamese)", "WINDOWS-1258"},
  /*  {"PLAIN TEXT (No Unicode Control Characters)", "PLAIN TEXT", NULL, NULL, TRUE},*/
  {"UTF-8", "UTF-8"},
};

struct Language languages[] = {
  {"Arabic", arabic},
  {"Baltic", baltic},
  {"Central European", central_european},
  {"Cyrillic", cyrillic},
  {"Greek", greek},
  {"Hebrew", hebrew},
  {"Japanese", japanese},
  {"Romanian", romanian},
  {"Turkish", turkish},
  {"Western", western},
  {"Other", other},
};

#endif

Encodings::Encodings()
{
  _default_save = 28;
  _default_open = 1;

  Language *lang;
  Encoding *enc;

  /* Arabic */
  lang = new Language;
  lang->name = "Arabic";
  _languages.push_back(lang);

  enc = new Encoding;
  enc->name = "ISO 8859-6 (Arabic)";
  enc->encoding = "ISO_8859-6";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  enc = new Encoding;
  enc->name = "WINDOWS-1256 (Arabic)";
  enc->encoding = "WINDOWS-1256";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  /* Baltic */
  lang = new Language;
  lang->name = "Baltic";
  _languages.push_back(lang);

  enc = new Encoding;
  enc->name = "ISO_8859-4 (Baltic)";
  enc->encoding = "ISO_8859-4";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  enc = new Encoding;
  enc->name = "ISO_8859-13 (Baltic)";
  enc->encoding = "ISO_8859-13";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  enc = new Encoding;
  enc->name = "WINDOWS-1257 (Baltic)";
  enc->encoding = "WINDOWS-1257";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  /* Central European */
  lang = new Language;
  lang->name = "Central European";
  _languages.push_back(lang);

  enc = new Encoding;
  enc->name = "ISO 8859-2 (Central European)";
  enc->encoding = "ISO_8859-2";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  enc = new Encoding;
  enc->name = "WINDOWS-1250 (Central European)";
  enc->encoding = "WINDOWS-1250";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  /* Cyrillic */
  lang = new Language;
  lang->name = "Cyrillic";
  _languages.push_back(lang);

  enc = new Encoding;
  enc->name = "ISO_8859-5 (Cyrillic)";
  enc->encoding = "ISO_8859-5";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  enc = new Encoding;
  enc->name = "WINDOWS-1251 (Cyrillic)";
  enc->encoding = "WINDOWS-1251";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  /* Greek */
  lang = new Language;
  lang->name = "Greek";
  _languages.push_back(lang);

  enc = new Encoding;
  enc->name = "ISO 8859-7 (Greek)";
  enc->encoding = "ISO_8859-7";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  enc = new Encoding;
  enc->name = "WINDOWS-1253 (Greek)";
  enc->encoding = "WINDOWS-1253";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  /* Hebrew */
  lang = new Language;
  lang->name = "Hebrew";
  _languages.push_back(lang);

  enc = new Encoding;
  enc->name = "ISO 8859-8-i (Hebrew - logical ordering)";
  enc->encoding = "ISO_8859-8";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  enc = new Encoding;
  enc->name = "WINDOWS-1255 (Hebrew)";
  enc->encoding = "WINDOWS-1255";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  /* Japanese */
  lang = new Language;
  lang->name = "Japanese";
  _languages.push_back(lang);

  enc = new Encoding;
  enc->name = "EUC-JP (Japanese)";
  enc->encoding = "EUC-JP";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  enc = new Encoding;
  enc->name = "SHIFT_JIS (Japanese)";
  enc->encoding = "SHIFT_JIS";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  enc = new Encoding;
  enc->name = "ISO-2022-JP (Japanese)";
  enc->encoding = "ISO-2022-JP";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  /* Romanian */
  lang = new Language;
  lang->name = "Romanian";
  _languages.push_back(lang);

  enc = new Encoding;
  enc->name = "ISO_8859-16 (Romanian)";
  enc->encoding = "ISO-8859-16";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  /* Turkish */
  lang = new Language;
  lang->name = "Turkish";
  _languages.push_back(lang);

  enc = new Encoding;
  enc->name = "ISO 8859-9 (Turkish)";
  enc->encoding = "ISO_8859-9";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  enc = new Encoding;
  enc->name = "WINDOWS-1254 (Turkish)";
  enc->encoding = "WINDOWS-1254";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  /* Western */
  lang = new Language;
  lang->name = "Western";
  _languages.push_back(lang);

  enc = new Encoding;
  enc->name = "ISO 8859-1 (Western European)";
  enc->encoding = "ISO_8859-1";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  enc = new Encoding;
  enc->name = "ISO_8859-15 (Western, New)";
  enc->encoding = "ISO_8859-15";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  enc = new Encoding;
  enc->name = "WINDOWS-1252 (Western)";
  enc->encoding = "WINDOWS-1252";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  /* Other */
  lang = new Language;
  lang->name = "Other";
  _languages.push_back(lang);

  enc = new Encoding;
  enc->name = "ISO_8859-3 (South European)";
  enc->encoding = "ISO_8859-3";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  enc = new Encoding;
  enc->name = "ISO_8859-10 (Nordic)";
  enc->encoding = "ISO_8859-10";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  enc = new Encoding;
  enc->name = "ISO_8859-11";
  enc->encoding = "ISO_8859-11";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  enc = new Encoding;
  enc->name = "ISO_8859-12";
  enc->encoding = "ISO_8859-12";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  enc = new Encoding;
  enc->name = "ISO_8859-14";
  enc->encoding = "ISO_8859-14";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  enc = new Encoding;
  enc->name = "WINDOWS-1258 (Vietnamese)";
  enc->encoding = "WINDOWS-1258";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  enc = new Encoding;
  enc->name = "UTF-8";
  enc->encoding = "UTF-8";
  lang->children.push_back(enc);
  _encodings.push_back(enc);

  /*
  while (tmp->name.c_str())
    {
      Encoding *tmp2 = tmp->children;
      while (tmp2->name.c_str())
	{
	  encodings.push_back(tmp2);
	  ++tmp2;
	}
      ++tmp;
    }
  */
}

Encodings::~Encodings()
{
}

const std::string& Encodings::get(unsigned int x)
{
  assert(x <= _encodings.size());
  return _encodings[x]->encoding;
}

int Encodings::get(std::string& enc)
{
  for (unsigned int x = 0; x < _encodings.size(); x++)
    if (_encodings[x]->name == enc)
      return x;
  return -1;
}

const std::string& Encodings::name(unsigned int x)
{
  assert(x <= _encodings.size());
  return _encodings[x]->name;
}

bool Encodings::convert(std::string& text, std::string& res, unsigned int from, unsigned int to, std::string& err)
{
  assert(from <= _encodings.size());
  assert(to <= _encodings.size());
  assert(from != to);
  try {
    res = Glib::convert (text, get(to), get(from));
    return true;
  }
  catch (Glib::ConvertError& _err)
    {
      err = _err.what();
      return false;
    }
}

int Encodings::convert(std::string& in, std::string& out, int enc)
{
  if (utf8(in))
    return utf8();
  else
    {
      std::string err;
      if (convert(in, out, enc, utf8(), err))
	{
	  if (!utf8(out))
	    {
	      out = err;
	      return -1;
	    }
	  return enc;
	}
      return -1;
    }
}

bool Encodings::utf8(std::string& text)
{
  return Glib::ustring(text).validate();
}
