/*
 * mdi.cc
 * This file is part of katoob
 *
 * Copyright (C) 2006 Mohammed Sameer
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <iostream>
#include "mdi.hh"
#include "dialogs.hh"
#include "filedialog.hh"
#include "macros.h"
#ifdef ENABLE_PRINT
#include "printdialog.hh"
#endif
#include "searchdialog.hh"
//#include "replacedialog.hh"
#ifdef HAVE_SPELL
#include "spelldialog.hh"
#endif
#include "dict.hh"
#include "utils.hh"
#include <cassert>
#include "execdialog.hh"
#include "pipe.hh"
#include "tempfile.hh"
#include "openlocationdialog.hh"
#include "thread.hh"

MDI::MDI(Conf& conf, Encodings& enc) : _conf(conf), _encodings(enc)
{
    signal_switch_page().connect(sigc::mem_fun(*this, &MDI::on_switch_page_cb));
}

MDI::~MDI()
{

}

void MDI::document(std::string& file, int enc)
{
  if (enc == -1)
    enc = _encodings.default_open();

  int x = children.size();
  Document *doc;
  if (file == "-")
    doc = new Document(_conf, _encodings, ++x, enc);
  else
    doc = new Document(_conf, _encodings, enc, file);

  if (doc->ok())
    {
      add(doc);
      std::string str = Glib::path_get_basename(file);
      _conf.open_dir(str);

      // Ensure absolute path for the file.
      if (Glib::path_is_absolute(file))
	_conf.recent_prepend(file);
      else {
	std::vector<std::string> elems;
	elems.push_back(Glib::get_current_dir());
	elems.push_back(file);
	file = Glib::build_path(Utils::get_dir_separator(), elems);
	elems.clear();
	_conf.recent_prepend(file);
      }
      recent_regenerate.emit();
    }
  else
    {
      delete doc;
      if (children.size() == 0)
	signal_reset_gui.emit(-1);
    }
}

void MDI::document()
{
  int x = children.size();
  Document *doc = new Document(_conf, _encodings, ++x);
  add(doc);
}

void MDI::add(Document *doc, bool signals)
{
  // This should be first so when we get the switch_page signal we'll find something.
  children.push_back(doc);

  if (children.size() == 1)
    signal_reset_gui.emit(0);

  if (signals)
    connect_signals(doc);

  // This should be first otherwise the documents menu will try to manipulate
  // a menu that has not been added.
  signal_document_added.emit(doc->readonly(), doc->modified(), doc->get_title());

  append_page (*doc, doc->get_label());
  set_menu_label_text(*doc, doc->get_title());

  set_current_page (-1);

  // This is here because we can't put it in connect_signals() because the document
  // is appended after connect_signals()
  // We will connect to the label directly bypassing the Document object.
  dynamic_cast<Label *>(get_tab_label(*children[get_current_page()]))->signal_close_clicked.connect(sigc::bind<Document *>(sigc::mem_fun(this, &MDI::signal_document_label_close_clicked_cb), doc));
}

Document *MDI::active()
{
  int x = get_current_page();
  if (x != -1)
    return children[x];
  return NULL;
}

void MDI::open_cb()
{
  std::string title = _("Please select a file for editing.");
  FileDialog dlg(title, FILE_OPEN, -1, _conf, _encodings);
  if (dlg.run())
    {
      int enc = dlg.encoding();
      std::vector<std::string> files = dlg.get();
      for (unsigned int x = 0; x < files.size(); x++)
	document(files[x], enc);
    }
}

void MDI::open_location_cb()
{
  OpenLocationDialog dialog(_conf, _encodings);

  Document *doc = active();
  // disable the check if we have no active.
  if (!doc)
    dialog.disable_insert_to_active();

  bool to_active = false;

  std::string out;
  if(dialog.run(out))
    {
      if (doc)
	to_active = dialog.insert_to_active();
      // get encoding,
      int enc = dialog.get_encoding();
      if (!to_active)
	{
	  document();
	  doc = active();
	}
	// We convert it to utf8.
      std::string out2;
      if (_encodings.convert(out, out2, enc) != -1)
	{
	  // Insert.
	  doc->insert(out2.size() == 0 ? out : out2);
	  // TODO: Set a title for the document.
	  // We also need the document to know that it's a remote document.
	}
      else {
	katoob_error(out2);
      }
    }
  else {
    if (out.size() > 0)
      katoob_error(out);
  }
}

void MDI::insert_file_cb()
{
  // FIXME: Implement me.
  Document *doc = active();
  if (!doc)
    return;

  if (doc->readonly())
    {
      std::string error(_("This is a read only document."));
      katoob_error(error);
      return;
    }

  std::string title = _("Please select a file.");
  FileDialog dlg(title, FILE_OPEN, -1, _conf, _encodings);
  if (dlg.run())
    {
      int enc = dlg.encoding();
      std::vector<std::string> files = dlg.get();
      std::string contents, contents2;

      for (unsigned int x = 0; x < files.size(); x++)
	{
	  if (Utils::katoob_read(files[x], contents))
	    {
	      int tenc = _encodings.convert(contents, contents2, enc);
	      if (tenc == -1) {
		std::string str(Utils::substitute(_("Couldn't detect the encoding of %s"), files[x]));
		katoob_error(str);
	      }
	      else {
		doc->insert(tenc == _encodings.utf8() ? contents : contents2);
	      }
	    }
	  else {
	    katoob_error(contents);
	  }
	}
    }
}

void MDI::save_cb()
{
  save();
}

bool MDI::save()
{
  Document *doc = active();
  if (!doc)
    return true;
  return save(doc);
}

bool MDI::save(Document *doc)
{
  assert(doc != NULL);

  if (doc->readonly())
    {
      std::string er(_("This document can not be saved. It's a read only file."));
      katoob_error(er);
      return true;
    }

  if (doc->has_file())
      return doc->save();

  return save(true);
}

void MDI::save_as_cb()
{
  Document *doc = active();
  if (doc)
    save(true);
}

void MDI::save_copy_cb()
{
  Document *doc = active();
  if (doc)
    save(false);
}

bool MDI::save(bool replace)
{
  Document *doc = active();
  if (!doc)
    return true;

  std::string str = _("Please select a file for saving.");
  FileDialog dlg(str, FILE_SAVE, doc->get_encoding(), _conf, _encodings);
  if (dlg.run())
    {
      int enc = dlg.encoding();
      std::vector<std::string> files = dlg.get();
      std::string str = Glib::path_get_basename(files[0]);

      if (Glib::file_test (files[0], Glib::FILE_TEST_EXISTS))
	{
	  std::string message(Utils::substitute(_("Are you sure you want to overwrite the file %s ?"), files[0]));
	  if (katoob_simple_question(message))
	    {
	      if (doc->save(files[0], enc, replace))
		{
		  if (replace) {
		    _conf.recent_prepend(files[0]);
		    recent_regenerate.emit();
		  }
		  _conf.save_dir(str);
		  return true;
		}
	      else
		return false;
	    }
	  return false;
	}
      else
	if (doc->save(files[0], enc, replace))
	  {
	    if (replace) {
	      _conf.recent_prepend(files[0]);
	      recent_regenerate.emit();
	    }
	    _conf.save_dir(str);
	    return true;
	  }
	else
	  return false;
    }
  return false;
}

#ifdef ENABLE_PRINT
void MDI::print_cb()
{
  Document *doc = active();
  if (!doc)
    return;

  PrintDialog dialog(_conf);

  std::string err;
  if (!dialog.ok(err))
    {
      katoob_error(err);
      return;
    }
  dialog.has_selection(doc->has_selection());
  dialog.lines(doc->line_count());

  if (!dialog.run())
    return;

  dialog.start_process();

  std::vector<std::string> lines;
  if (dialog.all())
    {
      doc->get_lines(lines);
    }
  else if (dialog.selection())
    {
      doc->get_selection(lines);
    }
  else {
    int x, y;
    bool l = dialog.range(x, y);
    assert(x < y);
    assert(l = true);
    if (!l)
      {
	std::string err = _("You must select something to print.");
	katoob_error(err);
	return;
      }
    doc->get_lines(lines, x, y);
  }

  for (unsigned l = 0; l < lines.size(); l++)
    dialog.add_line(lines[l]);

  if (!dialog.end_process(err))
    {
      katoob_error(err);
      return;
    }

  dialog.manipulate();
}
#endif

void MDI::close_cb()
{
  Document *doc = active();
  if (doc)
    close();

  if (children.size() == 0)
    signal_reset_gui.emit(-1);
}

bool MDI::close(int n)
{
  Document *doc = (n == -1 ? active() : children[n]);

  if (!doc)
    return true;

  if (doc->modified())
    {
      std::string message;
      if (doc->has_file())
	message = Utils::substitute(_("The file %s is not saved, Save first ?"), doc->file());
      else
	message = Utils::substitute(_("\"%s\" is not saved, Save first?"), doc->get_title());
      switch(katoob_question(message))
	{
	case Gtk::RESPONSE_YES:
	  if (save())
	    break;
	  else
	    return false;
	case Gtk::RESPONSE_NO:
	  break;
	case Gtk::RESPONSE_CANCEL:
	default:
	  return false;
	}
    }

  std::vector<Document *>::iterator iter = children.begin();

  int x = (n == -1 ? get_current_page() : n);

  iter += x;
  children.erase(iter);
  remove_page(x);

  // Let's add it to our closed documents vector.
  // Is it enabled ?
  if (_conf.get("undo_closed", true))
    {
      closed_children.push_back(doc);
      // Now let's see how many items do we have.
      unsigned x = _conf.get("undo_closedno", 5);
      if (x != 0)
	{
	  while (closed_children.size() > x)
	    {
	      Document *_doc = closed_children[0];
	      closed_children.erase(closed_children.begin());
	      signal_closed_document_erased.emit();
	      delete _doc;
	    }
	}
      // emit the signal so the menu can rebuild.
      signal_closed_document_added.emit(doc->get_title());
    }
  else {
    delete doc;
  }
  signal_document_removed.emit(x);
  return true;
}

void MDI::recent_cb(std::string& rfile)
{
  unsigned x = children.size();
  document(rfile, -1);
  if (children.size() != x)
    {
      _conf.recent_prepend(rfile);
      recent_regenerate.emit();
    }
}

void MDI::undo_cb()
{
  Document *doc = active();
  if (doc)
    return doc->undo();
}

void MDI::redo_cb()
{
  Document *doc = active();
  if (doc)
    return doc->redo();
}

void MDI::cut_cb()
{
  Document *doc = active();
  if (doc)
    doc->cut();
}

void MDI::copy_cb()
{
  Document *doc = active();
  if (doc)
    doc->copy();
}

void MDI::paste_cb()
{
  Document *doc = active();
  if (doc)
    doc->paste();
}

void MDI::erase_cb()
{
  Document *doc = active();
  if (doc)
    doc->erase();
}

void MDI::revert_cb()
{
  Document *doc = active();
  if (!doc)
    return;

  if (!doc->has_file())
    return;

  std::string message(Utils::substitute(_("This will close the current file \"%s\". Discard any changes done and reload the saved copy of the file. Are you sure you want to continue ?"), doc->file()));

  if (katoob_simple_question(message))
    {
      message.clear();
      if (!doc->revert(message))
	katoob_error(message);
    }
}

void MDI::select_all_cb()
{
  Document *doc = active();
  if (doc)
    doc->select_all();
}

void MDI::close_all_cb()
{
  close_all();
}

void MDI::save_all_cb()
{
  for (unsigned x = 0; x < children.size(); x++)
    save(children[x]);
}

void MDI::line_numbers(bool _active)
{
  Document *doc = active();
  if (doc)
    doc->line_numbers(_active);
}

void MDI::wrap_text(bool _active)
{
  Document *doc = active();
  if (doc)
    doc->wrap_text(_active);
}

bool MDI::close_all()
{
  while (true)
    {
      if (!active())
	return true;
      if (!close())
	return false;
    }
}

void MDI::goto_line_cb()
{
  Document *doc = active();
  if (!doc)
    return;

  goto_line_cb2(katoob_goto_dialog());
}

void MDI::goto_line_cb2(int line)
{
  Document *doc = active();
  if (!doc)
    return;

  if (line >= 1)
    doc->scroll_to(line);
}

void MDI::find(std::string& str)
{
  Document *doc = active();
  if (!doc)
    return;

  doc->set_search_text(str);
  find_next_cb();
}

void MDI::find_cb()
{
  Document *doc = active();
  if (!doc)
    return;

  SearchDialog dialog;

  dialog.whole_word(doc->get_search_whole_word());
  dialog.match_case(doc->get_search_match_case());
  dialog.wrap(doc->get_search_wrap());
  dialog.backwards(doc->get_search_backwards());
  dialog.beginning(doc->get_search_from_beginning());
  dialog.set_text(doc->get_search_text());

  if (!dialog.run())
    return;

  if (dialog.get_text().size() == 0)
    {
      std::string str(_("You must inter a word to search for."));
      katoob_error(str);
      return;
    }

  doc->set_search_whole_word(dialog.whole_word());
  doc->set_search_match_case(dialog.match_case());
  doc->set_search_wrap(dialog.wrap());
  doc->set_search_backwards(dialog.backwards());
  doc->set_search_from_beginning(dialog.beginning());
  doc->set_search_text(dialog.get_text());

  if (!doc->search())
    {
      std::string str(_("No search results found."));
      katoob_error(str);
    }
}

void MDI::find_next_cb()
{
  Document *doc = active();
  if (!doc)
    return;

  std::string str = doc->get_search_text();
  if (str.size() == 0)
    {
      find_cb();
      return;
    }

  if (!doc->search_next())
    {
      std::string str(_("No search results found."));
      katoob_error(str);
    }
}

void MDI::replace_cb()
{
  Document *doc = active();
  if (!doc)
    return;

  ReplaceDialog dialog;

  dialog.whole_word(doc->get_search_whole_word());
  dialog.match_case(doc->get_search_match_case());
  dialog.wrap(doc->get_search_wrap());
  dialog.backwards(doc->get_search_backwards());
  dialog.beginning(doc->get_search_from_beginning());
  dialog.set_text(doc->get_search_text());
  dialog.set_replace(doc->get_replace_text());

  // Connect the signals.
  dialog.signal_find.connect(sigc::bind<ReplaceDialog *>(sigc::mem_fun(*this, &MDI::replace_dialog_signal_find_cb), &dialog));
  dialog.signal_replace.connect(sigc::bind<ReplaceDialog *>(sigc::mem_fun(*this, &MDI::replace_dialog_signal_replace_cb), &dialog));
  dialog.signal_replace_all.connect(sigc::bind<ReplaceDialog *>(sigc::mem_fun(*this, &MDI::replace_dialog_signal_replace_all_cb), &dialog));
  dialog.run();

  doc->set_search_whole_word(dialog.whole_word());
  doc->set_search_match_case(dialog.match_case());
  doc->set_search_wrap(dialog.wrap());
  doc->set_search_backwards(dialog.backwards());
  doc->set_search_from_beginning(dialog.beginning());
  doc->set_search_text(dialog.get_text());
  doc->set_replace_text(dialog.get_replace());
}

bool MDI::replace_dialog_signal_find_cb(ReplaceDialog *dialog)
{
  Document *doc = active();
  assert(doc != NULL);

  doc->set_search_whole_word(dialog->whole_word());
  doc->set_search_match_case(dialog->match_case());
  doc->set_search_wrap(dialog->wrap());
  doc->set_search_backwards(dialog->backwards());
  doc->set_search_from_beginning(dialog->beginning());
  doc->set_search_text(dialog->get_text());

  if (!doc->search())
    {
      std::string str(_("No search results found."));
      katoob_error(str);
      return false;
    }
  dialog->beginning(doc->get_search_from_beginning());
  return true;
}

void MDI::replace_dialog_signal_replace_cb(ReplaceDialog *dialog)
{
  Document *doc = active();
  assert(doc != NULL);
  doc->set_replace_text(dialog->get_replace());
  doc->replace();
}

void MDI::replace_dialog_signal_replace_all_cb(ReplaceDialog *dialog)
{
  Document *doc = active();
  assert(doc != NULL);
  int x = 0;

  doc->set_search_whole_word(dialog->whole_word());
  doc->set_search_match_case(dialog->match_case());
  doc->set_search_backwards(dialog->backwards());
  doc->set_search_from_beginning(dialog->beginning());
  doc->set_search_text(dialog->get_text());
  doc->set_replace_text(dialog->get_replace());

  // We don't wrap otherwise we'll get an infinite loop.
  doc->set_search_wrap(false);

  while (doc->search())
    {
      ++x;
      doc->replace();
    }
  std::string msg(Utils::substitute(_("Replaced %d occurences."), x));
  katoob_info(msg);
}

void MDI::execute_cb()
{
  Document *doc = active();
  if (!doc)
    return;

  ExecDialog dlg(_conf);

  if (dlg.run())
    {
      bool new_buffer = dlg.get_new_buffer();
      std::string command = dlg.get_command();
      _conf.exec_cmd_prepend(command);

      unsigned found = 0;
      for (unsigned x = 0; x < command.size(); x++)
	if ((command[x] == '%') && (command[x+1] == 's'))
	  ++found;

      if (found > 1)
	{
	  std::string message(_("You can use only one '%s'"));
	  katoob_error(message);
	  return;
	}

      std::string error;
      std::string cmd;
      std::string text;
      TempFile tf;
      if (found == 1)
	{
	  if (!tf.ok(error))
	    {
	      katoob_error(error);
	      return;
	    }

	  // Write the buffer contents.
	  text = doc->get_text();
	  if (!tf.write(text, error))
	    {
	      katoob_error(error);
	      return;
	    }
	  text.clear();
	  cmd = Utils::substitute(command, tf.get_name());
	}
      else
	cmd = command;

      Pipe pipe;
      if (!pipe.exec(cmd, text, error))
	{
	  katoob_error(error);
	  return;
	}
      if(!new_buffer)
	doc->set_text(text);
      else {
	// We create a new document then we insert.
	document();
	Document *d = active();
	d->set_text(text);
      }
    }
}

#ifdef HAVE_SPELL
void MDI::auto_spell(bool s)
{
  Document *doc = active();
  if (!doc)
    return;
  doc->auto_spell(s);
}

void MDI::spell()
{
  Document *doc = active();
  if ((!doc) || (doc->readonly()) || (doc->is_empty()))
    return;

  // TODO: Check that the document can really do the spell checking.

  SpellDialog dialog(doc);
  dialog.run();
}

bool MDI::set_dictionary(std::string& old, std::string& new_dict)
{
  std::string error;
  Document *doc = active();
  assert(doc != NULL);
  old = doc->get_dictionary();
  if (!doc->set_dictionary(new_dict, error))
    {
      katoob_error(error);
      return false;
    }
  return true;
}

#endif

void MDI::on_switch_page_cb(GtkNotebookPage *p, guint n)
{
  Document *doc = children[n];
  doc->emit_signals();
  signal_doc_activated.emit(n);
}

void MDI::activate(int x)
{
  set_current_page(x);
}

void MDI::connect_signals(Document *doc)
{
  // NOTE: Connect new signals whenever the Document class gets new!
  doc->signal_modified.connect(sigc::mem_fun(this, &MDI::signal_document_modified_cb));
  doc->signal_can_undo.connect(sigc::mem_fun(this, &MDI::signal_document_can_undo_cb));
  doc->signal_can_redo.connect(sigc::mem_fun(this, &MDI::signal_document_can_redo_cb));
  doc->signal_readonly.connect(sigc::mem_fun(this, &MDI::signal_document_readonly_cb));
  doc->signal_file_changed.connect(sigc::mem_fun(this, &MDI::signal_document_file_changed_cb));
  doc->signal_cursor_moved.connect(sigc::mem_fun(this, &MDI::signal_document_cursor_moved_cb));
  doc->signal_encoding_changed.connect(sigc::mem_fun(this, &MDI::signal_document_encoding_changed_cb));
  doc->signal_overwrite_toggled.connect(sigc::mem_fun(this, &MDI::signal_document_overwrite_toggled_cb));
  doc->signal_title_changed.connect(sigc::mem_fun(this, &MDI::signal_document_title_changed_cb));
#ifdef HAVE_SPELL
  doc->signal_spell_enabled.connect(sigc::mem_fun(this, &MDI::signal_document_spell_enabled_cb));
  doc->signal_dictionary_changed.connect(sigc::mem_fun(this, &MDI::signal_dictionary_changed_cb));
#endif
  doc->signal_wrap_text.connect(sigc::mem_fun(this, &MDI::signal_document_wrap_text_cb));
  doc->signal_line_numbers.connect(sigc::mem_fun(this, &MDI::signal_document_line_numbers_cb));
  doc->signal_dict_lookup.connect(sigc::mem_fun(this, &MDI::signal_document_dict_lookup_cb));
#ifdef ENABLE_HIGHLIGHT
  doc->signal_highlight.connect(sigc::mem_fun(this, &MDI::signal_document_highlight_cb));
#endif
}

bool MDI::set_encoding(int n, int& o)
{
  std::string err;

  Document *doc = active();

  if (!doc)
    return true;

  o = doc->get_encoding();
  if (!doc->has_file())
    {
      doc->set_encoding(n, false, err);
      return true;
    }

  if (!doc->set_encoding(n, true, err))
    {
      katoob_error(err);
      return false;
    }
  return true;
}

void MDI::signal_document_dict_lookup_cb(std::string word)
{
  std::string result, error;
  try {
    Dispatcher *dispatcher = new Dispatcher;
    dispatcher->in = word;

    sigc::slot<void> slot = sigc::bind<Conf, Dispatcher *>(sigc::ptr_fun(katoob_dict_define_word), _conf, dispatcher);
    Thread thread(slot, dispatcher);

    std::string error;
    if (thread.ok(error))
      {
	if (dispatcher->err)
	  katoob_error(dispatcher->error);
	else
	  katoob_info(dispatcher->out);
      }
    else {
      katoob_error(error);
    }
    delete dispatcher;
  }
  catch (Glib::FileError &er)
    {
#if ((defined(_WIN32) && !defined(NDEBUG)) || (!defined(_WIN32)))
      std::cout << er.what() << std::endl;
#endif
    }
}

void MDI::signal_document_title_changed_cb(std::string t)
{
  signal_document_title_changed.emit(t, get_current_page());
  set_menu_label_text(*active(), t);
}

void MDI::signal_document_readonly_cb(bool ro)
{
  signal_document_readonly.emit(get_current_page(), ro);
  // NOTE: No idea why won't this work when I set it from the document itself.
  dynamic_cast<Label *>(get_tab_label(*children[get_current_page()]))->readonly(ro);
}

void MDI::signal_document_modified_cb(bool m)
{
  signal_document_modified.emit(get_current_page(), m);
  // NOTE: No idea why won't this work when I set it from the document itself.
  dynamic_cast<Label *>(get_tab_label(*children[get_current_page()]))->modified(m);
}

void MDI::signal_document_label_close_clicked_cb(Document *doc)
{
  for (unsigned x = 0; x < children.size(); x++)
    if (children[x] == doc)
      {
	close(x);
	if (children.size() == 0)
	  signal_reset_gui.emit(-1);
	return;
      }
}

void MDI::closed_document_activated_cb(int x)
{
  Document *doc = closed_children[x];
  std::vector<Document *>::iterator iter = closed_children.begin();
  iter += x;
  closed_children.erase(iter);

  add(doc, false);
}

void MDI::reset_gui()
{
  if (_conf.get("tabsmenu", true))
    popup_enable();
  else
    popup_disable();

  set_show_tabs(_conf.get("showtabs", true));
  set_scrollable(_conf.get("scrolltabs", true));
  switch(_conf.get("tabspos", TABS_POS_TOP))
    {
    case TABS_POS_BOTTOM:
      set_tab_pos(Gtk::POS_BOTTOM);
      break;
    case TABS_POS_RIGHT:
      set_tab_pos(Gtk::POS_RIGHT);
      break;
    case TABS_POS_LEFT:
      set_tab_pos(Gtk::POS_LEFT);
      break;
    default:
      set_tab_pos(Gtk::POS_TOP);
      break;
    }

  for (unsigned x = 0; x < children.size(); x++)
    children[x]->reset_gui();

  for (unsigned x = 0; x < closed_children.size(); x++)
    closed_children[x]->reset_gui();

  int total = get_n_pages();
  for (int x = 0; x < total; x++)
    {
      bool m = children[x]->modified();
      bool r = children[x]->readonly();

      if (m)
	dynamic_cast<Label *>(get_tab_label(*children[x]))->modified(children[x]->modified(), true);

      if (r)
	dynamic_cast<Label *>(get_tab_label(*children[x]))->readonly(children[x]->readonly(), true);
      if ((!r) && (!m))
	dynamic_cast<Label *>(get_tab_label(*children[x]))->normal();
    }
}

void MDI::import_cb(Import im)
{
  SimpleFileDialog dialog(im.name, FILE_OPEN, _conf);
  if (!dialog.run())
    return;
  std::string file = dialog.get();
  std::string out;
  if (!im.func(file, out))
    {
      katoob_error(out);
      return;
    }
  // is it utf8 ?
  if (_encodings.utf8(out))
    {
      // create.
      document();
      Document *doc = active();
      doc->set_text(out);
    }
  else {
    std::string res;
    std::string error;
    if (_encodings.convert(out, res, _encodings.default_open(), _encodings.utf8(), error))
      {
	if (_encodings.utf8(res))
	  {
	    // create.
	    document();
	    Document *doc = active();
	    doc->set_text(res);
	  }
	else
	  {
	    std::string message(Utils::substitute(_("Couldn't detect the encoding of %s"), file));
	    katoob_error(message);
	  }
      }
    else {
      katoob_error(error);
    }
  }
}

void MDI::export_cb(Export ex)
{
  Document *doc = active();
  if (!doc)
    return;

  SimpleFileDialog dialog(ex.name, FILE_SAVE, _conf);
  if (!dialog.run())
    return;

  std::string file = dialog.get();

  if (Glib::file_test (file, Glib::FILE_TEST_EXISTS))
    {
      std::string message(Utils::substitute(_("Are you sure you want to overwrite the file %s ?"), file));
      if (!katoob_simple_question(message))
	return;
    }
  std::string error;
  std::string out;
  if (ex.lines) {
    std::vector<Glib::ustring> lines;
    doc->get_lines(lines);
    for (unsigned x = 0; x < lines.size(); x++) {
      std::string tout;

      if (!ex.func(lines[x], tout, error))
	{
	  katoob_error(error);
	  return;
	}
      out += tout;
      // TODO: We need to check whether the document has a \n at the end or not.
      out += '\n';
    }
  }
  else {
    Glib::ustring text = doc->get_text();
    if (!ex.func(text, out, error))
      {
	katoob_error(error);
	return;
      }
    text.clear();
  }
  // Write.
  if (!Utils::katoob_write(_conf, file, out, error))
    katoob_error(error);
}

void MDI::signal_extra_button_clicked_cb(std::string str)
{
  Document *doc = active();
  if (!doc)
    return;
  doc->insert(str);
}

#ifdef ENABLE_HIGHLIGHT
void MDI::highlighter_clicked_cb(int x) {
  Document *doc = active();
  if (doc)
    doc->set_highlight(x);
}
#endif
