/*
 * spell.cc
 * This file is part of katoob
 *
 * Copyright (C) 2006 Mohammed Sameer
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include "spell.hh"
#include <cassert>

Spell::Spell()
{
  speller = NULL;

  config = new_aspell_config();
  err = new_aspell_speller(config);

  aspell_config_replace(config, "encoding", "utf-8");

  //  aspell_config_replace(config, "language-tag", lang);

  if (aspell_error_number(err) == 0)
    speller = to_aspell_speller(err);
}

Spell::~Spell()
{
  delete_aspell_config(config);
  delete_aspell_speller(speller);
  err = NULL;
}

bool Spell::ok(std::string& error)
{
  if (speller)
    return true;
  else
    error = aspell_error_message(err);
  return false;
}

bool Spell::set_lang(std::string& lang, std::string& error)
{
  //  assert(speller != NULL);
  aspell_config_replace (config, "language-tag", lang.c_str());

  // NOTE: We don't free error. We cast it to our speller object
  // And we delete the Speller object instead.
  err = new_aspell_speller (config);

  if (aspell_error_number (err) == 0)
    {
      delete_aspell_speller(speller);
      speller = to_aspell_speller(err);
      return true;
    }
  else {
    error = aspell_error_message(err);
    return false;
  }
}

bool Spell::check(std::string& word)
{
  assert(speller != NULL);
  if (aspell_speller_check (speller, word.c_str(), word.size()))
    return true;
  return false;
}

void Spell::suggest(std::string& word, std::vector<std::string>& sugg)
{
  assert(speller != NULL);

  const AspellWordList *suggs = aspell_speller_suggest(speller, word.c_str(), word.size());
  AspellStringEnumeration *elements = aspell_word_list_elements(suggs);

  const char *wd;
  while ( (wd = aspell_string_enumeration_next(elements)) != NULL)
    {
      sugg.push_back(wd);
    }
  delete_aspell_string_enumeration(elements);
}

void Spell::replace(std::string& a, std::string& b)
{
  assert(speller != NULL);
  aspell_speller_store_replacement(speller, a.c_str(), a.size(), b.c_str(), b.size());
}

void Spell::to_session(std::string& a)
{
  aspell_speller_add_to_session(speller, a.c_str(), a.size());
}

void Spell::to_personal(std::string& s)
{
  assert(speller != NULL);
  aspell_speller_add_to_personal(speller, s.c_str(), s.size());

  // NOTE:I wanted to move this to destructor but it
  // looks like all the implementation is doing this here

  aspell_speller_save_all_word_lists(speller);
}

void katoob_spell_list_available(std::vector<std::string>& dicts)
{
  AspellConfig *config = NULL;
  AspellDictInfoList *dlist = NULL;
  AspellDictInfoEnumeration *enums = NULL;
  const AspellDictInfo *entry = NULL;

  config = new_aspell_config ();
  dlist = get_aspell_dict_info_list (config);
  delete_aspell_config (config);
  enums = aspell_dict_info_list_elements (dlist);

  while ((entry = aspell_dict_info_enumeration_next (enums)) != 0)
    dicts.push_back(entry->name);

  delete_aspell_dict_info_enumeration(enums);
}
