/***************************************************************************
*   Copyright (C) 2004-2006 by Thomas Fischer                             *
*   fischer@unix-ag.uni-kl.de                                             *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
***************************************************************************/
#include <qstring.h>
#include <qregexp.h>

#include <encoderlatex.h>

namespace BibTeX
{

    EncoderLaTeX *encoderLaTeX = NULL;

    static const struct EncoderLaTeXCharMapping
    {
        const char *regexp;
        unsigned int unicode;
        const char *latex;
    }
    charmappingdatalatex[] = {
                                 {"\\\\#", 0x0023, "\\#"},
                                 /*{"\\\\&", 0x0026, "\\&"},*/
                                 {"!`", 0x00A1, "!`"},
                                 {"\"<", 0x00AB, "\"<"},
                                 {"\">", 0x00BB, "\">"},
                                 {"?`", 0x00BF, "?`"},
                                 {"\\{?\\\\`\\{?A\\}?\\}?", 0x00C0, "{\\`A}"},
                                 {"\\{?\\\\'\\{?A\\}?\\}?", 0x00C1, "{\\'A}"},
                                 {"\\{?\\\\\\^\\{?A\\}?\\}?", 0x00C21, "{\\^A}"},
                                 {"\\{?\\\\~\\{?A\\}?\\}?", 0x00C3, "{\\~A}"},
                                 {"\\{?\\\\\"\\{?A\\}?\\}?", 0x00C4, "{\\\"A}"},
                                 {"\\{?\\\\AA(\\{\\})?\\}?", 0x00C5, "{\\r{A}}"},
                                 {"\\{?\\\\r\\{?A\\}?\\}?", 0x00C5, "{\\r{A}}"},
                                 {"\\{?\\\\AE\\}?(\\{\\})?", 0x00C6, "{\\AE}"},
                                 {"\\{?\\\\c\\{?C\\}?\\}?", 0x00C7, "{\\c{C}}"},
                                 {"\\{?\\\\`\\{?E\\}?\\}?", 0x00C8, "{\\`E}"},
                                 {"\\{?\\\\'\\{?E\\}?\\}?", 0x00C9, "{\\'E}"},
                                 {"\\{?\\\\\\^\\{?E\\}?\\}?", 0x00CA, "{\\^E}"},
                                 {"\\{?\\\\\"\\{?E\\}?\\}?", 0x00CB, "{\\\"E}"},
                                 {"\\{?\\\\'\\{?I\\}?\\}?", 0x00CD, "{\\'I}"},
                                 {"\\{?\\\\\\^\\{?I\\}?\\}?", 0x00CE, "{\\^I}"},
                                 {"\\{?\\\\~\\{?N\\}?\\}?", 0x00D1, "{\\~N}"},
                                 {"\\{?\\\\`\\{?O\\}?\\}?", 0x00D2, "{\\`O}"},
                                 {"\\{?\\\\'\\{?O\\}?\\}?", 0x00D3, "{\\'O}"},
                                 {"\\{?\\\\\\^\\{?O\\}?\\}?", 0x00D4, "{\\^O}"},
                                 {"\\{?\\\\\"\\{?O\\}?\\}?", 0x00D6, "{\\\"O}"},
                                 {"\\{?\\\\{?O\\}?\\}?", 0x00D8, "{\\O}"},
                                 {"\\{?\\\\`\\{?U\\}?\\}?", 0x00D9, "{\\`U}"},
                                 {"\\{?\\\\'\\{?U\\}?\\}?", 0x00DA, "{\\'U}"},
                                 {"\\{?\\\\\\^\\{?U\\}?\\}?", 0x00DB, "{\\^U}"},
                                 {"\\{?\\\\\"\\{?U\\}?\\}?", 0x00DC, "{\\\"U}"},
                                 {"\\{?\\\\'\\{?Y\\}?\\}?", 0x00DD, "{\\'Y}"},
                                 {"\\{?\\\\\"\\{?s\\}?\\}?", 0x00DF, "{\\ss}"},
                                 {"\\{?\\\\ss\\}?(\\{\\})?", 0x00DF, "{\\ss}"},
                                 {"\\{?\\\\`\\{?a\\}?\\}?", 0x00E0, "{\\`a}"},
                                 {"\\{?\\\\'\\{?a\\}?\\}?", 0x00E1, "{\\'a}"},
                                 {"\\{?\\\\\\^\\{?a\\}?\\}?", 0x00E2, "{\\^a}"},
                                 {"\\{?\\\\~\\{?a\\}?\\}?", 0x00E3, "{\\~a}"},
                                 {"\\{?\\\\\"\\{?a\\}?\\}?", 0x00E4, "{\\\"a}"},
                                 {"\\{?\\\\aa(\\{\\})?\\}?", 0x00E5, "{\\r{a}}"},
                                 {"\\{?\\\\r\\{?a\\}?\\}?", 0x00E5, "{\\r{a}}"},
                                 {"\\{?\\\\ae\\}?(\\{\\})?", 0x00E6, "{\\ae}"},
                                 {"\\{?\\\\c\\{?c\\}?\\}?", 0x00E7, "{\\c{c}}"},
                                 {"\\{?\\\\`\\{?e\\}?\\}?", 0x00E8, "{\\`e}"},
                                 {"\\{?\\\\'\\{?e\\}?\\}?", 0x00E9, "{\\'e}"},
                                 {"\\{?\\\\\\^\\{?e\\}?\\}?", 0x00EA, "{\\^e}"},
                                 {"\\{?\\\\\"\\{?e\\}?\\}?", 0x00EB, "{\\\"e}"},
                                 {"\\{?\\\\'\\{?\\\\?i\\}?\\}?", 0x00ED, "{\\'i}"},
                                 {"\\{?\\\\\\^\\{?i\\}?\\}?", 0x00EE, "{\\^i}"},
                                 {"\\{?\\\\~\\{?n\\}?\\}?", 0x00F1, "{\\~n}"},
                                 {"\\{?\\\\`\\{?o\\}?\\}?", 0x00F2, "{\\`o}"},
                                 {"\\{?\\\\'\\{?o\\}?\\}?", 0x00F3, "{\\'o}"},
                                 {"\\{?\\\\\\^\\{?o\\}?\\}?", 0x00F4, "{\\^o}"},
                                 {"\\{?\\\\\"\\{?o\\}?\\}?", 0x00F6, "{\\\"o}"},
                                 {"\\{?\\\\u\\{?G\\}?\\}?", 0x011E, "{\\u{G}}"},
                                 {"\\{?\\\\u\\{?g\\}?\\}?", 0x011F, "{\\u{g}}"},
                                 // \oe has to be before \o
                                 {"\\{?\\\\OE\\}?(\\{\\})?", 0x0152, "{\\OE}"},
                                 {"\\{?\\\\oe\\}?(\\{\\})?", 0x0153, "{\\oe}"},
                                 {"\\{?\\\\\\{?o\\}?\\}?", 0x00F8, "{\\o}"},
                                 {"\\{?\\\\`\\{?u\\}?\\}?", 0x00F9, "{\\`u}"},
                                 {"\\{?\\\\'\\{?u\\}?\\}?", 0x00FA, "{\\'u}"},
                                 {"\\{?\\\\\\^\\{?u\\}?\\}?", 0x00FB, "{\\^u}"},
                                 {"\\{?\\\\\"\\{?u\\}?\\}?", 0x00FC, "{\\\"u}"},
                                 {"\\{?\\\\'\\{?y\\}?\\}?", 0x00FD, "{\\'y}"},
                                 {"\\{?\\\\u\\{?A\\}?\\}?", 0x0102, "{\\u{A}}"},
                                 {"\\{?\\\\u\\{?a\\}?\\}?", 0x0103, "{\\u{a}}"},
                                 {"\\{?\\\\'\\{?C\\}?\\}?", 0x0106, "{\\'C}"},
                                 {"\\{?\\\\'\\{?c\\}?\\}?", 0x0107, "{\\'c}"},
                                 {"\\{?\\\\v\\{?C\\}?\\}?", 0x010C, "{\\v{C}}"},
                                 {"\\{?\\\\v\\{?c\\}?\\}?", 0x010D, "{\\v{c}}"},
                                 {"\\{?\\\\L\\}?(\\{\\})?", 0x0141, "{\\L}"},
                                 {"\\{?\\\\l\\}?(\\{\\})?", 0x0142, "{\\l}"},
                                 {"\\{?\\\\'\\{?N\\}?\\}?", 0x0143, "{\\'N}"},
                                 {"\\{?\\\\'\\{?n\\}?\\}?", 0x0144, "{\\'n}"},
                                 {"\\{?\\\\H\\{?O\\}?\\}?", 0x0150, "{\\H{O}}"},
                                 {"\\{?\\\\H\\{?o\\}?\\}?", 0x0151, "{\\H{o}}"},
                                 {"\\{?\\\\c\\{?S\\}?\\}?", 0x015E, "{\\c{S}}"},
                                 {"\\{?\\\\c\\{?s\\}?\\}?", 0x015F, "{\\c{s}}"},
                                 {"\\{?\\\\v\\{?S\\}?\\}?", 0x0160, "{\\v{S}}"},
                                 {"\\{?\\\\v\\{?s\\}?\\}?", 0x0161, "{\\v{s}}"},
                                 {"\\{?\\\\r\\{?U\\}?\\}?", 0x016E, "{\\r{U}}"},
                                 {"\\{?\\\\r\\{?u\\}?\\}?", 0x016F, "{\\r{u}}"},
                                 {"\\{?\\\\v\\{?A\\}?\\}?", 0x01CD, "{\\v{A}}"},
                                 {"\\{?\\\\v\\{?a\\}?\\}?", 0x01CE, "{\\v{a}}"},
                                 {"--", 0x2013, "--"},
                                 {"\\{?\\\\glqq\\{?\\}?\\}?", 0x201E, "{\\glqq}"},
                                 {"\\{?\\\\grqq\\{?\\}?\\}?", 0x201C, "{\\grqq}"}
                             };
    static const int charmappingdatalatexcount = sizeof( charmappingdatalatex ) / sizeof( charmappingdatalatex[ 0 ] ) ;

    EncoderLaTeX::EncoderLaTeX() : Encoder()
    {
        buildCharMapping();
    }

    EncoderLaTeX::~EncoderLaTeX()
    {
        // nothing
    }

    QString EncoderLaTeX::decode( const QString & text )
    {
        // split text into math and non-math regions
        QStringList intermediate = QStringList::split( '$', text, TRUE );

        for ( QStringList::Iterator it = intermediate.begin(); it != intermediate.end(); it++ )
        {
            for ( QValueList<CharMappingItem>::ConstIterator cmit = m_charMapping.begin(); cmit != m_charMapping.end(); ++cmit )
                ( *it ).replace( ( *cmit ).regExp, ( *cmit ).unicode );

            // skip math regions
            it++;
            if ( it == intermediate.end() )
                break;
        }

        QString result = intermediate.join( "$" );
        // result.replace( "\\\\", "\\" );

        return result;
    }

    QString EncoderLaTeX::encode( const QString & text )
    {
        QString result = text;
        bool beginningQuotationNext = TRUE;

        // result.replace( "\\", "\\\\" );

        for ( QValueList<CharMappingItem>::ConstIterator it = m_charMapping.begin(); it != m_charMapping.end(); ++it )
            result.replace( ( *it ).unicode, ( *it ).latex );

        for ( unsigned int i = 0; i < result.length(); i++ )
            if ( result.at( i ) == '"' && ( i == 0 || result.at( i - 1 ) != '\\' ) )
            {
                if ( beginningQuotationNext )
                    result.replace( i, 1, "``" );
                else
                    result.replace( i, 1, "''" );
                beginningQuotationNext = !beginningQuotationNext;
            }

        return result;
    }

    QString EncoderLaTeX::decodeSpecialized( const QString & text, const EntryField::FieldType fieldType )
    {
        QString result = decode( text );

        switch ( fieldType )
        {
        case EntryField::ftPages:
            result.replace( "--", QChar( 0x2013 ) );
            result.replace( "-", QChar( 0x2013 ) );
            break;
        default:
            break;
        }

        return result;
    }

    QString EncoderLaTeX::encodeSpecialized( const QString & text, const EntryField::FieldType fieldType )
    {
        QString result = encode( text );

        switch ( fieldType )
        {
        case EntryField::ftPages:
            result.replace( QChar( 0x2013 ), "--" );
            break;
        default:
            break;
        }

        return result;
    }

    void EncoderLaTeX::buildCharMapping()
    {
        for ( int i = 0; i < charmappingdatalatexcount; i++ )
        {
            CharMappingItem charMappingItem;
            charMappingItem.regExp = QRegExp( charmappingdatalatex[ i ].regexp );
            charMappingItem.unicode = QChar( charmappingdatalatex[ i ].unicode );
            charMappingItem.latex = QString( charmappingdatalatex[ i ].latex );
            m_charMapping.append( charMappingItem );

        }
    }

    EncoderLaTeX* EncoderLaTeX::currentEncoderLaTeX()
    {
        if ( encoderLaTeX == NULL )
            encoderLaTeX = new EncoderLaTeX();

        return encoderLaTeX;
    }

    void EncoderLaTeX::deleteCurrentEncoderLaTeX()
    {
        if ( encoderLaTeX != NULL )
        {
            delete encoderLaTeX;
            encoderLaTeX = NULL;
        }
    }

}
