/***************************************************************************
*   Copyright (C) 2004-2006 by Thomas Fischer                             *
*   fischer@unix-ag.uni-kl.de                                             *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
***************************************************************************/
#include <qwidget.h>
#include <qdom.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qspinbox.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qcombobox.h>
#include <qapplication.h>
#include <qbuffer.h>
#include <qhttp.h>
#include <qstringlist.h>
#include <qcstring.h>
#include <qregexp.h>

#include <kdialog.h>
#include <kurllabel.h>
#include <kurl.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kdebug.h>
#include <kapplication.h>
#include <kiconloader.h>

#include <entryfield.h>
#include <value.h>
#include "webquerypubmed.h"

namespace KBibTeX
{

    WebQueryPubMed::WebQueryPubMed( QWidget* parent, const char* name ) : WebQuery( parent, name )
    {
        wizard = new PubMedWizard( title(), parent, name );
    }

    WebQueryPubMed::~WebQueryPubMed()
    {
        // nothing
    }

    BibTeX::File* WebQueryPubMed::query( )
    {
        if ( wizard->exec() == QDialog::Accepted )
            return wizard->result();
        else
            return NULL;
    }

    QString WebQueryPubMed::title()
    {
        return i18n( "NCBI (PubMed)" );
    }

    int WebQueryPubMed::uniqueId()
    {
        return 0x013ac4b8;
    }

    PubMedWizard::PubMedWizard ( const QString &caption, QWidget* parent, const char* name ) : KDialogBase( parent, name, TRUE, "Import", Ok | Cancel, Ok, TRUE ), buffer( NULL ), bibtexFile( NULL )
    {
        setCaption( caption );
        setupGUI();

        http = new QHttp ( "eutils.ncbi.nlm.nih.gov" );
    }

    PubMedWizard::~PubMedWizard()
    {
        delete http;
    }

    BibTeX::File *PubMedWizard::result()
    {
        return bibtexFile;
    }

    void PubMedWizard::accept ()
    {
        bibtexFile = new BibTeX::File();
        for ( QListViewItemIterator it( m_listViewResults, QListViewItemIterator::Selected ); it.current();++it )
        {
            ResultsListViewItem *item = dynamic_cast<ResultsListViewItem*>( *it );
            if ( item != NULL )
                bibtexFile->appendElement( BibTeX::File::cloneElement( item->entry() ) );
        }

        KDialogBase::accept();
    }

    int PubMedWizard::exec ()
    {
        buffer = new QBuffer( QByteArray( 0x1000000 ) );
        int result = QDialog::exec();
        delete buffer;

        return result;
    }

    void PubMedWizard::startSearch()
    {
        setEnabled( FALSE );
        QApplication::setOverrideCursor( Qt::waitCursor );
        m_listViewResults->clear();

        connect( http, SIGNAL( done( bool ) ), this, SLOT( databaseQueryDone( bool ) ) );
        buffer->open( IO_ReadWrite );
        KURL url = KURL( QString( "http://eutils.ncbi.nlm.nih.gov/entrez/eutils/esearch.fcgi?db=pubmed&term=%1&retmax=%2&tool=KBibTeX&email=kbibtex@unix-ag.uni-kl.de" ).arg( m_lineEditQuery->text() ).arg( m_spinBoxMaxHits->text() ) );
        kdDebug() << "url: " << url.prettyURL() << endl;
        QHttpRequestHeader header( "GET", url.prettyURL(), 1, 0 );
        http->request( header, 0, buffer );
    }

    void PubMedWizard::databaseQueryDone( bool error )
    {
        disconnect( http, SIGNAL( done( bool ) ), this, SLOT( databaseQueryDone( bool ) ) );
        buffer->close();

        if ( error )
        {
            setEnabled( TRUE );
            QApplication::restoreOverrideCursor();
            KMessageBox::error( this, QString( i18n( "Could not query NCBI's database:\n%1" ) ).arg( http->errorString() ) );
            return ;
        }

        //         buffer->open( IO_ReadOnly );
        //         QTextStream ts( buffer );
        //         ts.setEncoding( QTextStream::UnicodeUTF8 );
        //         QString dump = ts.read();
        //         kdDebug() << "Fetched XML data BEGIN" << endl << dump << endl << "Fetched XML data END" << endl;
        //         buffer->close();

        QValueList<int> intList;
        buffer->open( IO_ReadOnly );
        QXmlInputSource input( buffer );
        QXmlSimpleReader reader;
        StructureParserQuery handler( &intList );
        reader.setContentHandler( &handler );
        reader.parse( &input );
        buffer->close();

        QString ids;
        QValueList<int>::iterator it = intList.begin();
        if ( it != intList.end() )
        {
            ids.append( QString::number( *it ) );
            ++it;
            for ( ; it != intList.end(); ++it )
            {
                ids.append( "," );
                ids.append( QString::number( *it ) );
            }
        }

        KURL url = KURL( QString( "http://eutils.ncbi.nlm.nih.gov/entrez/eutils/efetch.fcgi?db=pubmed&retmode=xml&id=%1&tool=KBibTeX&email=kbibtex@unix-ag.uni-kl.de" ).arg( ids ) );
        kdDebug() << "url: " << url.prettyURL() << endl;

        connect( http, SIGNAL( done( bool ) ), this, SLOT( fetchingSummaryDone( bool ) ) );
        buffer->open( IO_ReadWrite );
        QHttpRequestHeader header( "GET", url.prettyURL(), 1, 0 );
        http->request( header, 0, buffer );

    }

    void PubMedWizard::fetchingSummaryDone( bool error )
    {
        disconnect( http, SIGNAL( done( bool ) ), this, SLOT( fetchingSummaryDone( bool ) ) );
        buffer->close();

        if ( error )
        {
            setEnabled( TRUE );
            QApplication::restoreOverrideCursor();
            KMessageBox::error( this, QString( i18n( "Could not query NCBI's database:\n%1" ) ).arg( http->errorString() ) );
            return ;
        }

        //         buffer->open( IO_ReadOnly );
        //         QTextStream ts( buffer );
        //         ts.setEncoding( QTextStream::UnicodeUTF8 );
        //         QString dump = ts.read();
        //         kdDebug() << "Fetched XML data BEGIN" << endl << dump << endl << "Fetched XML data END" << endl;
        //         buffer->close();

        QDomDocument doc( "efetch'ed" );
        doc.setContent( buffer );
        QDomElement docElem = doc.documentElement();
        ResultParser resultParser( m_listViewResults, doc.documentElement() );

        setEnabled( TRUE );
        QApplication::restoreOverrideCursor();
        m_listViewResults->setFocus();
    }

    void PubMedWizard::listViewResultsSelectionChanged( )
    {
        QListViewItemIterator it( m_listViewResults, QListViewItemIterator::Selected );

        enableButtonOK( it.current() != NULL );
    }

    void PubMedWizard::setupGUI()
    {
        QWidget * mainWidget = new QWidget( this );
        mainWidget->setMinimumSize( 640, 256 );
        QGridLayout *gridLayout = new QGridLayout( mainWidget, 3, 5, 0, KDialog::spacingHint() );
        gridLayout->setColStretch( 1, 10 );
        gridLayout->setRowStretch( 1, 10 );
        QLabel *label = new QLabel( i18n( "&Query string:" ), mainWidget );
        gridLayout->addWidget ( label, 0, 0 );
        m_lineEditQuery = new QLineEdit( mainWidget );
        gridLayout->addWidget ( m_lineEditQuery, 0, 1 );
        label->setBuddy( m_lineEditQuery );
        label = new QLabel( i18n( "&Number of results:" ), mainWidget );
        gridLayout->addWidget ( label, 0, 2 );
        m_spinBoxMaxHits = new QSpinBox( 5, 5000, 1, mainWidget );
        m_spinBoxMaxHits->setValue( 50 );
        gridLayout->addWidget ( m_spinBoxMaxHits, 0, 3 );
        label->setBuddy( m_spinBoxMaxHits );
        m_pushButtonSearch = new QPushButton( i18n( "&Search" ), mainWidget );
        gridLayout->addWidget ( m_pushButtonSearch, 0, 4 );
        m_pushButtonSearch->setIconSet( QIconSet( SmallIcon( "find" ) ) );

        m_listViewResults = new QListView( mainWidget );
        m_listViewResults->addColumn( i18n( "Year" ), 64 );
        m_listViewResults->addColumn( i18n( "Author" ), 128 );
        m_listViewResults->addColumn( i18n( "Title" ), 512 );
        m_listViewResults->setAllColumnsShowFocus( TRUE );
        m_listViewResults->setSelectionMode( QListView::Extended );
        gridLayout->addMultiCellWidget ( m_listViewResults, 1, 1, 0, 4 );

        KURLLabel *disclaimerLabel = new KURLLabel( mainWidget );
        disclaimerLabel->setText( i18n( "NCBI's Disclaimer and Copyright" ) );
        disclaimerLabel->setURL( "http://eutils.ncbi.nlm.nih.gov/About/disclaimer.html" );
        gridLayout->addMultiCellWidget ( disclaimerLabel, 2, 2, 0, 4 );

        m_lineEditQuery->setFocus();
        setMainWidget( mainWidget );
        setButtonOK( KGuiItem( i18n( "&Import" ), "import", i18n( "Import selected items" ) ) );
        enableButtonOK( FALSE );
        setPlainCaption( i18n( "Import from NCBI (PubMed)" ) );

        connect( disclaimerLabel, SIGNAL( leftClickedURL( const QString& ) ), kapp, SLOT( invokeBrowser( const QString& ) ) );
        connect( m_listViewResults, SIGNAL( selectionChanged ( ) ), this, SLOT( listViewResultsSelectionChanged( ) ) );
        connect( m_listViewResults, SIGNAL( clicked ( QListViewItem* ) ), this, SLOT( listViewResultsSelectionChanged( ) ) );
        connect( m_pushButtonSearch, SIGNAL( clicked() ), this, SLOT( startSearch() ) );
        connect( m_lineEditQuery, SIGNAL( returnPressed() ), this, SLOT( startSearch() ) );
    }

    StructureParserQuery::StructureParserQuery( QListView * listView ) : QXmlDefaultHandler(), m_listView( listView )
    {
        m_listView->clear();
    }

    StructureParserQuery::StructureParserQuery( QValueList<int> *intList ) : QXmlDefaultHandler(), m_listView( NULL ), m_intList( intList )
    {
        m_intList->clear();
    }

    StructureParserQuery::~StructureParserQuery( )
    {
        // nothing
    }

    bool StructureParserQuery::startElement( const QString & /*namespaceURI*/, const QString & /*localName*/, const QString & qName, const QXmlAttributes & /*atts*/ )
    {
        concatString = QString();

        if ( qName == "ResultItem" && ( lastDbName == "books" || lastDbName == "pubmed" || lastDbName == "journals" ) )
        {
            bool ok;
            int hits = lastCount.toInt( &ok );
            if ( ok && hits > 0 )
                new DatabasesListViewItem( m_listView, lastDbName, lastMenuName, hits );
        }

        return TRUE;
    }

    bool StructureParserQuery::endElement( const QString & /*namespaceURI*/, const QString & /*localName*/, const QString & qName )
    {
        if ( qName == "DbName" )
            lastDbName = concatString;
        else if ( qName == "MenuName" )
            lastMenuName = concatString;
        else if ( qName == "Count" )
            lastCount = concatString;
        else if ( qName == "Id" )
        {
            bool ok;
            int id = concatString.toInt( &ok );
            if ( ok && id > 0 && m_intList != NULL )
                m_intList->append( id );
        }

        return TRUE;
    }

    bool StructureParserQuery::characters ( const QString & ch )
    {
        concatString.append( ch );
        return TRUE;
    }

    ResultParser::ResultParser( QListView *listView, const QDomElement& rootElement )
    {
        if ( rootElement.tagName() == "PubmedArticleSet" )
            for ( QDomNode n = rootElement.firstChild(); !n.isNull(); n = n.nextSibling() )
            {
                QDomElement e = n.toElement();
                if ( !e.isNull() && e.tagName() == "PubmedArticle" )
                {
                    BibTeX::Entry * entry = new BibTeX::Entry( BibTeX::Entry::etMisc, "PubMed" );
                    parsePubmedArticle( e, entry );
                    new ResultsListViewItem( listView, entry );
                }
            }
    }

    ResultParser::~ResultParser()
    {
        // nothing
    }

    void ResultParser::parsePubmedArticle( const QDomElement& element, BibTeX::Entry *entry )
    {
        for ( QDomNode n = element.firstChild(); !n.isNull(); n = n.nextSibling() )
        {
            QDomElement e = n.toElement();
            if ( !e.isNull() && e.tagName() == "MedlineCitation" )
                parseMedlineCitation( e, entry );
        }
    }

    void ResultParser::parseMedlineCitation( const QDomElement& element, BibTeX::Entry *entry )
    {
        for ( QDomNode n = element.firstChild(); !n.isNull(); n = n.nextSibling() )
        {
            QDomElement e = n.toElement();
            if ( !e.isNull() )
            {
                if ( e.tagName() == "PMID" )
                    entry->setId( QString( "PubMed_%1" ).arg( e.text() ) );
                else if ( e.tagName() == "Article" )
                    parseArticle( e, entry );
                else if ( e.tagName() == "MedlineJournalInfo" )
                {
                    for ( QDomNode n2 = e.firstChild(); !n2.isNull(); n2 = n2.nextSibling() )
                    {
                        QDomElement e2 = n2.toElement();
                        if ( e2.tagName() == "MedlineTA" )
                        {
                            BibTeX::EntryField * field = entry->getField( BibTeX::EntryField::ftJournal );
                            if ( field == NULL )
                            {
                                field = new BibTeX::EntryField( BibTeX::EntryField::ftJournal );
                                entry->addField( field );
                            }
                            BibTeX::Value *value = new BibTeX::Value ();;
                            value->add( new BibTeX::ValueItem( e2.text(), FALSE ) );
                            field->setValue( value );
                        }
                    }
                }
            }
        }

    }

    void ResultParser::parseArticle( const QDomElement& element, BibTeX::Entry *entry )
    {
        entry->setEntryType( BibTeX::Entry::etArticle );

        for ( QDomNode n = element.firstChild(); !n.isNull(); n = n.nextSibling() )
        {
            QDomElement e = n.toElement();

            if ( e.tagName() == "Journal" )
                parseJournal( e, entry );
            else if ( e.tagName() == "ArticleTitle" )
            {
                BibTeX::EntryField * field = entry->getField( BibTeX::EntryField::ftTitle );
                if ( field == NULL )
                {
                    field = new BibTeX::EntryField( BibTeX::EntryField::ftTitle );
                    entry->addField( field );
                }
                BibTeX::Value *value = new BibTeX::Value ();;
                value->add( new BibTeX::ValueItem( e.text(), FALSE ) );
                field->setValue( value );
            }
            else if ( e.tagName() == "Pagination" )
            {
                QDomElement medlinePgn = e.firstChild().toElement(); // may fail?
                if ( !medlinePgn.text().isEmpty() )
                {
                    BibTeX::EntryField * field = entry->getField( BibTeX::EntryField::ftPages );
                    if ( field == NULL )
                    {
                        field = new BibTeX::EntryField( BibTeX::EntryField::ftPages );
                        entry->addField( field );
                    }
                    BibTeX::Value *value = new BibTeX::Value ();;
                    value->add( new BibTeX::ValueItem( medlinePgn.text(), FALSE ) );
                    field->setValue( value );
                }
            }
            // TF: I AM NOT SURE ABOUT COPYRIGHTS, SO FETCHING ABSTRACTS IS DISABLED FOR NOW
            /*            else if ( e.tagName() == "Abstract" )
                        {
                            QDomElement abstractText = e.firstChild().toElement();
                            BibTeX::EntryField * field = entry->getField( BibTeX::EntryField::ftAbstract );
                            if ( field == NULL )
                            {
                                field = new BibTeX::EntryField( BibTeX::EntryField::ftAbstract );
                                entry->addField( field );
                            }
                            BibTeX::Value *value = new BibTeX::Value ();;
                            value->add( new BibTeX::ValueItem( abstractText.text(), FALSE ) );
                            field->setValue( value );
                        }*/
            else if ( e.tagName() == "Affiliation" )
            {
                BibTeX::EntryField * field = entry->getField( "affiliation" );
                if ( field == NULL )
                {
                    field = new BibTeX::EntryField( "affiliation" );
                    entry->addField( field );
                }
                BibTeX::Value *value = new BibTeX::Value ();;
                value->add( new BibTeX::ValueItem( e.text(), FALSE ) );
                field->setValue( value );
            }
            else if ( e.tagName() == "AuthorList" )
                parseAuthorList( e, entry );
        }
    }

    void ResultParser::parseJournal( const QDomElement& element, BibTeX::Entry *entry )
    {
        for ( QDomNode n = element.firstChild(); !n.isNull(); n = n.nextSibling() )
        {
            QDomElement e = n.toElement();

            if ( e.tagName() == "ISSN" )
            {
                BibTeX::EntryField * field = entry->getField( BibTeX::EntryField::ftISSN );
                if ( field == NULL )
                {
                    field = new BibTeX::EntryField( BibTeX::EntryField::ftISSN );
                    entry->addField( field );
                }
                BibTeX::Value *value = new BibTeX::Value ();;
                value->add( new BibTeX::ValueItem( e.text(), FALSE ) );
                field->setValue( value );
            }
            else if ( e.tagName() == "JournalIssue" )
                parseJournalIssue( e, entry );
            else if ( e.tagName() == "Title" )
            {
                BibTeX::EntryField * field = entry->getField( BibTeX::EntryField::ftJournal );
                if ( field == NULL )
                {
                    field = new BibTeX::EntryField( BibTeX::EntryField::ftJournal );
                    entry->addField( field );
                }
                BibTeX::Value *value = new BibTeX::Value ();;
                value->add( new BibTeX::ValueItem( e.text(), FALSE ) );
                field->setValue( value );
            }
        }
    }

    void ResultParser::parseJournalIssue( const QDomElement& element, BibTeX::Entry *entry )
    {
        for ( QDomNode n = element.firstChild(); !n.isNull(); n = n.nextSibling() )
        {
            QDomElement e = n.toElement();

            if ( e.tagName() == "Volume" )
            {
                BibTeX::EntryField * field = entry->getField( BibTeX::EntryField::ftVolume );
                if ( field == NULL )
                {
                    field = new BibTeX::EntryField( BibTeX::EntryField::ftVolume );
                    entry->addField( field );
                }
                BibTeX::Value *value = new BibTeX::Value ();;
                value->add( new BibTeX::ValueItem( e.text(), FALSE ) );
                field->setValue( value );
            }
            else if ( e.tagName() == "Issue" )
            {
                BibTeX::EntryField * field = entry->getField( BibTeX::EntryField::ftNumber );
                if ( field == NULL )
                {
                    field = new BibTeX::EntryField( BibTeX::EntryField::ftNumber );
                    entry->addField( field );
                }
                BibTeX::Value *value = new BibTeX::Value ();;
                value->add( new BibTeX::ValueItem( e.text(), FALSE ) );
                field->setValue( value );
            }
            else if ( e.tagName() == "PubDate" )
                parsePubDate( e, entry );
        }
    }

    void ResultParser::parsePubDate( const QDomElement& element, BibTeX::Entry *entry )
    {
        for ( QDomNode n = element.firstChild(); !n.isNull(); n = n.nextSibling() )
        {
            QDomElement e = n.toElement();

            if ( e.tagName() == "Year" )
            {
                BibTeX::EntryField * field = entry->getField( BibTeX::EntryField::ftYear );
                if ( field == NULL )
                {
                    field = new BibTeX::EntryField( BibTeX::EntryField::ftYear );
                    entry->addField( field );
                }
                BibTeX::Value *value = new BibTeX::Value ();;
                value->add( new BibTeX::ValueItem( e.text(), TRUE ) );
                field->setValue( value );
            }
            else if ( e.tagName() == "Month" )
            {
                QString month = e.text().lower();
                BibTeX::EntryField * field = entry->getField( BibTeX::EntryField::ftMonth );
                if ( field == NULL )
                {
                    field = new BibTeX::EntryField( BibTeX::EntryField::ftMonth );
                    entry->addField( field );
                }
                BibTeX::Value *value = new BibTeX::Value ();;
                value->add( new BibTeX::ValueItem( month, TRUE ) );
                field->setValue( value );
            }
            else if ( e.tagName() == "MedlineDate" )
            {
                QStringList frags = QStringList::split( QRegExp( "\\s+" ), e.text() );
                for ( QStringList::Iterator it = frags.begin(); it != frags.end(); ++it )
                {
                    bool ok;
                    int num = ( *it ).toInt( &ok );
                    if ( ok && num > 1000 && num < 3000 )
                    {
                        BibTeX::EntryField * field = entry->getField( BibTeX::EntryField::ftYear );
                        if ( field == NULL )
                        {
                            field = new BibTeX::EntryField( BibTeX::EntryField::ftYear );
                            entry->addField( field );
                        }
                        BibTeX::Value *value = new BibTeX::Value ();;
                        value->add( new BibTeX::ValueItem( QString::number( num ), TRUE ) );
                        field->setValue( value );
                    }
                    else if ( !ok && ( *it ).length() == 3 )
                    {
                        BibTeX::EntryField * field = entry->getField( BibTeX::EntryField::ftMonth );
                        if ( field == NULL )
                        {
                            field = new BibTeX::EntryField( BibTeX::EntryField::ftMonth );
                            entry->addField( field );
                        }
                        BibTeX::Value *value = new BibTeX::Value ();;
                        value->add( new BibTeX::ValueItem( ( *it ).lower(), TRUE ) );
                        field->setValue( value );
                    }
                }
            }
        }
    }

    void ResultParser::parseAuthorList( const QDomElement& element, BibTeX::Entry *entry )
    {
        if ( element.attribute( "CompleteYN", "Y" ) == "Y" )
        {
            QStringList authorList;
            for ( QDomNode n = element.firstChild(); !n.isNull(); n = n.nextSibling() )
            {
                QDomElement e = n.toElement();
                if ( e.tagName() == "Author" && e.attribute( "ValidYN", "Y" ) == "Y" )
                {
                    QString lastName, firstName;
                    for ( QDomNode n2 = e.firstChild(); !n2.isNull(); n2 = n2.nextSibling() )
                    {
                        QDomElement e2 = n2.toElement();
                        if ( e2.tagName() == "LastName" )
                            lastName = e2.text();
                        else if ( e2.tagName() == "FirstName" || e2.tagName() == "ForeName" )
                            firstName = e2.text();
                    }
                    QString name = lastName;
                    if ( !firstName.isEmpty() )
                        name.prepend( " " ).prepend( firstName );
                    authorList.append( name );
                }
            }

            BibTeX::EntryField * field = entry->getField( BibTeX::EntryField::ftAuthor );
            if ( field == NULL )
            {
                field = new BibTeX::EntryField( BibTeX::EntryField::ftAuthor );
                entry->addField( field );
            }
            BibTeX::ValuePersons *value = new BibTeX::ValuePersons ();;
            value->add( new BibTeX::ValueItem( authorList.join( " and " ), FALSE ) );
            field->setValue( value );
        }
    }

    DatabasesListViewItem::DatabasesListViewItem ( QListView * parent, const QString & dbName, const QString & dbTitle, int numHits ) : QListViewItem( parent ), m_dbName( dbName )
    {
        setText( 0, QString::number( numHits ) );
        setText( 1, dbTitle );
    }

    QString DatabasesListViewItem::dbName()
    {
        return m_dbName;
    }

    ResultsListViewItem::ResultsListViewItem ( QListView * parent, BibTeX::Entry * entry ) : QListViewItem( parent ), m_entry( entry )
    {
        BibTeX::EntryField * field = entry->getField( BibTeX::EntryField::ftTitle );
        if ( field != NULL && field->value() != NULL )
            setText( 2, field ->value() ->plainString() );
        field = entry->getField( BibTeX::EntryField::ftAuthor );
        if ( field != NULL && field->value() != NULL )
        {
            BibTeX::ValuePersons * persons = dynamic_cast<BibTeX::ValuePersons*>( field->value() );
            if ( persons != NULL )
            {
                QValueList<BibTeX::Person*> list = persons->persons();
                QStringList authors;
                for ( QValueList<BibTeX::Person*>::ConstIterator it = list.begin();it != list.end(); ++it )
                    authors.append( ( *it ) ->text() );
                setText( 1, authors.join( " and " ) );
            }
            else setText( 1, field ->value() ->plainString() );
        }
        field = entry->getField( BibTeX::EntryField::ftYear );
        if ( field != NULL && field->value() != NULL )
            setText( 0, field ->value() ->plainString() );
    }

    ResultsListViewItem::~ResultsListViewItem()
    {
        if ( m_entry != NULL )
            delete m_entry;
    }

    BibTeX::Entry* ResultsListViewItem::entry()
    {
        return m_entry;
    }
}

