/***************************************************************************
 *   Copyright (C) 2006 by Rohan McGovern                                  *
 *   rohan.pm@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef DBUSITEM_H
#define DBUSITEM_H

#include <qstring.h>
#include <qlistview.h>

#include "dbusexceptions.h"

class QWidget;
class QDomElement;

/**
 * Any item which appears in the list view.
 */
class DBusItem : public QListViewItem {
public:
    DBusItem(
      QListViewItem * parent,
      QString const & text1,
      QString const & text2
    );

    DBusItem(
      QListView * parent,
      QString const & text1,
      QString const & text2
    );

    /**
     * Return a widget to display in bottom segment of window.
     * Widget is not shown until show() is called.
     */
    virtual QWidget * widget( QWidget * parent ) const = 0;

    /**
     * Get service of this item.  Always applicable.
     */
    QString service() const;

    /**
     * Get full path of object of this item, or null if not applicable
     * (i.e., for services).
     */
    QString object() const;

    /**
     * Get interface of this item, or null if not applicable
     * (i.e., for services and objects).
     */
    QString interface() const;

    /**
     * Check XML element @a elem is named @a name.
     * Does nothing on success, throws std::logic_error on failure.
     * It is a logic error because the creator of a new DBUS item is
     * supposed to check the tag is correct before creating it.
     * @param elem XML element to check
     * @param name tagName() @a elem is expected to return
     */
    static void ensureElementIsNamed(
      QDomElement const & elem,
      QString const & name
    ) throw( std::logic_error );

    /**
     * Check XML element @a elem for existence of a specified attribute,
     * @a attr.
     * Does nothing on success, throws QDBusXmlError on failure.
     * @param elem XML element to check
     * @param attr attribute to check for existence of
     */
    static void ensureElementHasAttribute(
      QDomElement const & elem,
      QString const & attr
    ) throw( QDBusXmlError );

    virtual ~DBusItem();

protected:

    /**
     * Figure out service containing this item.
     */
    QString discoverService() const;

    /**
     * Figure out object containing this item.
     */
    QString discoverObject() const;

    /**
     * Figure out interface containing this item.
     */
    QString discoverInterface() const;

    /**
     * Add labels describing what, if anything, went wrong while
     * navigating this item.
     */
    void addErrorInfo( QWidget * widget ) const;

    QString m_service;
    QString m_object;
    QString m_interface;

    /** Whether or not all went OK when enumerating this item's
     *  immediate children */
    bool m_ok;

    /// if !m_ok, contains error message
    QString m_error;

private:
    /// Not implemented.
    DBusItem( DBusItem const & other );

    /// Not implemented.
    const DBusItem & operator=( DBusItem const & other );
};

#endif
