/***************************************************************************
 *                                                                         *
 *   copyright (C) 2004 by Michael Buesch                                  *
 *   email: mbuesch@freenet.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation.                         *
 *                                                                         *
 ***************************************************************************/

#include "masterkey/smartkey/smartkey.h"
#include "masterkey/keydata.h"
#include "pwmexception.h"
#include "pwm.h"

#include <kmessagebox.h>
#include <klocale.h>
#include <kfiledialog.h>

#include <qfile.h>
#include <qcstring.h>


SmartKey::SmartKey(QWidget *_parent, PwM *_mainwnd)
 : parent (_parent)
 , mainwnd (_mainwnd)
{
	backend = new SmartKey_backend(this);
}

SmartKey::~SmartKey()
{
	delete backend;
}

bool SmartKey::requestKey(QByteArray *key)
{
	SmartKey_backend::Status status;
	QByteArray rawKey;
	status = backend->readCard(&rawKey);
	if (status != SmartKey_backend::status_ok) {
		displayStatusMsg(status);
		return false;
	}
	KeyData kd;
	if (!kd.setStream(rawKey)) {
		displayCorruptStreamMsg();
		return false;
	}
	if (kd.getType() != KeyData::type_smartcard)
		printWarn("Key Data Stream not a smartcard stream.");
	kd.getKey(key);
	return true;
}

bool SmartKey::genKey(QByteArray *key)
{
	bool err;
	SmartKey_backend::Status status;
	int e = KMessageBox::questionYesNo(parent,
					   i18n("Do you really want to generate "
						"a new SmartKey?\n"
						"All data on the SmartCard will be lost."),
					   i18n("Generate SmartKey?"));
	if (e != KMessageBox::Yes)
		return false;
	KeyData kd;
	kd.generate(KeyData::type_smartcard);
	QByteArray rawKey;
	err = kd.getStream(&rawKey);
	PWM_ASSERT(err);
	status = backend->writeCard(rawKey);
	if (status != SmartKey_backend::status_ok) {
		displayStatusMsg(status);
		return false;
	}
	if (key)
		kd.getKey(key);
	statusBarMsg(i18n("Successfully generated a new SmartKey on the SmartCard."));
	return true;
}

bool SmartKey::eraseCard()
{
	SmartKey_backend::Status status;
	int e = KMessageBox::questionYesNo(parent,
					   i18n("Do you really want to erase "
						"the SmartCard?\n"
						"All data on the SmartCard will be lost."),
					   i18n("Erase SmartCard?"));
	if (e != KMessageBox::Yes)
		return false;
	status = backend->eraseCard();
	if (status != SmartKey_backend::status_ok) {
		displayStatusMsg(status);
		return false;
	}
	statusBarMsg(i18n("Successfully erased the SmartCard."));
	return true;
}

bool SmartKey::exportToKeyFile()
{
	SmartKey_backend::Status status;
	QByteArray rawKey;
	status = backend->readCard(&rawKey);
	if (status != SmartKey_backend::status_ok) {
		displayStatusMsg(status);
		return false;
	}
	KeyData kd;
	if (!kd.setStream(rawKey)) {
		displayCorruptStreamMsg();
		return false;
	}

	QString path;
	path = KFileDialog::getSaveFileName(QString(),
					    i18n("*.pwk|PwManager Key File"),
					    parent);
	if (path.isEmpty())
		return false;
	if (QFile::exists(path)) {
		int e = KMessageBox::questionYesNo(parent,
						   i18n("The file \"%1\" does already exist.\n"
							"Do you want to overwrite it?").arg(path),
						   i18n("Overwrite File?"));
		if (e != KMessageBox::Yes)
			return false;
		if (!QFile::remove(path)) {
			KMessageBox::error(parent,
					   i18n("Could not delete file \"%1\"").arg(path),
					   i18n("Could Not Delete File"));
			return false;
		}
	}

	QFile fd(path);
	if (!fd.open(IO_ReadWrite)) {
		KMessageBox::error(parent,
				   i18n("Could not open the target file \"%1\"")
				   .arg(path),
				   i18n("Could Not Open File"));
		return false;
	}
	// Generate a new datastream.
	kd.getStream(&rawKey);
	Q_LONG written = fd.writeBlock(rawKey);
	if (static_cast<Q_ULONG>(written) != rawKey.size()) {
		KMessageBox::error(parent,
				   i18n("Could not write to the target file \"%1\"")
				   .arg(path),
				   i18n("Could Not Write File"));
		return false;
	}
	statusBarMsg(i18n("Successfully exported SmartKey to KeyFile."));
	return true;
}

bool SmartKey::importFromKeyFile()
{
	QString path;
	path = KFileDialog::getOpenFileName(QString(),
					    i18n("*.pwk|PwManager Key File\n"
						 "*|All files"),
					    parent);
	if (path.isEmpty())
		return false;
	QFile fd(path);
	if (!fd.open(IO_ReadOnly)) {
		KMessageBox::error(parent,
				   i18n("Could not open source file \"%1\"")
				   .arg(path),
				   i18n("Could Not Open File"));
		return false;
	}
	if (fd.size() > 1024*1024) {
		KMessageBox::error(parent,
				   i18n("This file is not a PwManager Key File."),
				   i18n("Invalid File"));
		return false;
	}
	QByteArray rawKey;
	rawKey = fd.readAll();
	KeyData kd;
	if (!kd.setStream(rawKey)) {
		KMessageBox::error(parent,
				   i18n("This file is not a PwManager Key File."),
				   i18n("Invalid File"));
		return false;
	}
	// Generate a new datastream.
	kd.getStream(&rawKey);
	int e = KMessageBox::questionYesNo(parent,
					   i18n("Do you really want to write the SmartCard?\n"
						"All data on the inserted SmartCard will be lost."),
					   i18n("Write SmartCard?"));
	if (e != KMessageBox::Yes)
		return false;
	SmartKey_backend::Status status;
	status = backend->writeCard(rawKey);
	if (status != SmartKey_backend::status_ok) {
		displayStatusMsg(status);
		return false;
	}
	statusBarMsg(i18n("Successfully imported KeyFile to SmartKey."));
	return true;
}

void SmartKey::displayKeyId()
{
	SmartKey_backend::Status status;
	QByteArray rawKey;
	status = backend->readCard(&rawKey);
	if (status != SmartKey_backend::status_ok) {
		displayStatusMsg(status);
		return;
	}
	KeyData kd;
	if (!kd.setStream(rawKey)) {
		displayCorruptStreamMsg();
		return;
	}
	QByteArray idData;
	kd.getId(&idData);
	QString id(PwMGlobal::humanReadableHex(idData));
	KMessageBox::information(parent,
				 i18n("The ID Number of the SmartKey is:\n%1").arg(id),
				 i18n("SmartKey ID"));
}

void SmartKey::displayStatusMsg(SmartKey_backend::Status status)
{
	switch (status) {
	case SmartKey_backend::status_errNoMem:
		KMessageBox::error(parent,
				   i18n("Not enough memory."),
				   i18n("Not Enough Memory"));
		break;
	case SmartKey_backend::status_errOpen:
		KMessageBox::error(parent,
				   i18n("Could not open the SmartCard.\n"
					"Is your SmartCard Terminal correctly configured?"),
				   i18n("SmartCard Open Failure"));
		break;
	case SmartKey_backend::status_errTimeout:
		KMessageBox::error(parent,
				   i18n("SmartCard open Timeout.\n"
					"Did you insert a SmartCard into the Terminal?"),
				   i18n("SmartCard Open Timeout"));
		break;
	case SmartKey_backend::status_errRead:
		KMessageBox::error(parent,
				   i18n("Could not read from the SmartCard.\n"
					"Is the inserted SmartCard a Memory Card?"),
				   i18n("SmartCard Read Failure"));
		break;
	case SmartKey_backend::status_errWrite:
		KMessageBox::error(parent,
				   i18n("Could not write to the SmartCard.\n"
					"Is the inserted SmartCard a Memory Card?"),
				   i18n("SmartCard Write Failure"));
		break;
	case SmartKey_backend::status_err2small:
		KMessageBox::error(parent,
				   i18n("SmartCard memory too small.\n"
					"The SmartCard memory size is %1 kBit (%2 byte).")
					.arg(QString::number(MIN_SMARTCARD_SIZE * 8 / 1024))
					.arg(QString::number(MIN_SMARTCARD_SIZE)),
				   i18n("SmartCard Size Error"));
		break;
	case SmartKey_backend::status_cancel:
		break; // The user cancelled the operation. No error message.
	case SmartKey_backend::status_none:
		PWM_ASSERT(false);
	case SmartKey_backend::status_running:
		PWM_ASSERT(false);
	case SmartKey_backend::status_ok:
		PWM_ASSERT(false);
	}
}

void SmartKey::displayCorruptStreamMsg()
{
	KMessageBox::error(parent,
			   i18n("The data stream read from the "
				"SmartCard is corrupt.\n"
				"Consider generating a new card."),
			   i18n("Corrupt SmartKey"));
}

void SmartKey::statusBarMsg(const QString &msg)
{
	if (!mainwnd)
		return;
	mainwnd->showStatMsg(msg);
}

#include "smartkey.moc"
