/***************************************************************************
 *                                                                         *
 *   copyright (C) 2004 by Michael Buesch                                  *
 *   email: mbuesch@freenet.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation.                         *
 *                                                                         *
 ***************************************************************************/

#include "serializer.h"
#include "configuration.h"
#include "pwmexception.h"


/* enable/disable serializer debugging (0/1) */
#define SERIALIZER_DEBUG	0
/* use the old xml tags for writing (0/1) */
#define USE_OLD_TAGS		0
/* write a CDATA section (0/1) */
#define WRITE_CDATA_SEC		0

/* This is compatibility stuff.
 * The names of the entries have changed and here are the
 * new and old ones
 */
#define ROOT_MAGIC_OLD		"PwM-xml-dat"
#define VER_STR_OLD		"ver"
#define COMPAT_VER_OLD		"0x02"
#define CAT_ROOT_OLD		"categories"
#define CAT_PREFIX_OLD		"cat_"
#define CAT_NAME_OLD		"name"
#define ENTRY_PREFIX_OLD	"entry_"
#define ENTRY_DESC_OLD		"desc"
#define ENTRY_NAME_OLD		"name"
#define ENTRY_PW_OLD		"pw"
#define ENTRY_COMMENT_OLD	"comment"
#define ENTRY_URL_OLD		"url"
#define ENTRY_LAUNCHER_OLD	"launcher"
#define ENTRY_LVP_OLD		"listViewPos"
#define ENTRY_BIN_OLD		"b"

#define ROOT_MAGIC_NEW		"P"
#define VER_STR_NEW		"v"
#define COMPAT_VER_NEW		"2"
#define CAT_ROOT_NEW		"c"
#define CAT_PREFIX_NEW		"c"
#define CAT_NAME_NEW		"n"
#define ENTRY_PREFIX_NEW	"e"
#define ENTRY_DESC_NEW		"d"
#define ENTRY_NAME_NEW		"n"
#define ENTRY_PW_NEW		"p"
#define ENTRY_COMMENT_NEW	"c"
#define ENTRY_URL_NEW		"u"
#define ENTRY_LAUNCHER_NEW	"l"
#define ENTRY_LVP_NEW		"v"
#define ENTRY_BIN_NEW		"b"

#if USE_OLD_TAGS != 0
# define ROOT_MAGIC_WR		ROOT_MAGIC_OLD
# define VER_STR_WR		VER_STR_OLD
# define COMPAT_VER_WR		COMPAT_VER_OLD
# define CAT_ROOT_WR		CAT_ROOT_OLD
# define CAT_PREFIX_WR		CAT_PREFIX_OLD
# define CAT_NAME_WR		CAT_NAME_OLD
# define ENTRY_PREFIX_WR	ENTRY_PREFIX_OLD
# define ENTRY_DESC_WR		ENTRY_DESC_OLD
# define ENTRY_NAME_WR		ENTRY_NAME_OLD
# define ENTRY_PW_WR		ENTRY_PW_OLD
# define ENTRY_COMMENT_WR	ENTRY_COMMENT_OLD
# define ENTRY_URL_WR		ENTRY_URL_OLD
# define ENTRY_LAUNCHER_WR	ENTRY_LAUNCHER_OLD
# define ENTRY_LVP_WR		ENTRY_LVP_OLD
# define ENTRY_BIN_WR		ENTRY_BIN_OLD
#else
# define ROOT_MAGIC_WR		ROOT_MAGIC_NEW
# define VER_STR_WR		VER_STR_NEW
# define COMPAT_VER_WR		COMPAT_VER_NEW
# define CAT_ROOT_WR		CAT_ROOT_NEW
# define CAT_PREFIX_WR		CAT_PREFIX_NEW
# define CAT_NAME_WR		CAT_NAME_NEW
# define ENTRY_PREFIX_WR	ENTRY_PREFIX_NEW
# define ENTRY_DESC_WR		ENTRY_DESC_NEW
# define ENTRY_NAME_WR		ENTRY_NAME_NEW
# define ENTRY_PW_WR		ENTRY_PW_NEW
# define ENTRY_COMMENT_WR	ENTRY_COMMENT_NEW
# define ENTRY_URL_WR		ENTRY_URL_NEW
# define ENTRY_LAUNCHER_WR	ENTRY_LAUNCHER_NEW
# define ENTRY_LVP_WR		ENTRY_LVP_NEW
# define ENTRY_BIN_WR		ENTRY_BIN_NEW
#endif


Serializer::Serializer()
 : base64Data (false)
{
	defaultLockStat = true;
	domDoc = new QDomDocument;
}

Serializer::Serializer(const QCString &buffer)
 : base64Data (false)
{
	defaultLockStat = true;
	domDoc = new QDomDocument;
	if (!parseXml(buffer)) {
		delete domDoc;
		throw PwMException(PwMException::EX_PARSE);
	}
}

Serializer::~Serializer()
{
	delete_ifnot_null(domDoc);
}

void Serializer::clear()
{
	delete_ifnot_null(domDoc);
	domDoc = new QDomDocument;
}

bool Serializer::parseXml(const QCString &buffer)
{
	PWM_ASSERT(domDoc);
	if (!domDoc->setContent(buffer, true))
		return false;
	if (!checkValid())
		return false;
	return true;
}

QCString Serializer::getXml()
{
	PWM_ASSERT(domDoc);

#if defined(PWM_DEBUG) && SERIALIZER_DEBUG != 0
	QCString tmp(domDoc->toCString(8));
	printDebug("<BEGIN Serializer::getXml() dump>\n");
	cout << tmp << endl;
	printDebug("<END Serializer::getXml() dump>");
#endif // DEBUG

	QCString ret(domDoc->toCString(0));
	ret.replace('\n', "");
	return ret;
}

bool Serializer::serialize(const vector<PwMCategoryItem> &dta)
{
	base64Data = conf()->confGlobBase64Storage();
	PWM_ASSERT(domDoc);
	QDomElement root(genNewRoot());
	QDomElement catNode(domDoc->createElement(CAT_ROOT_WR));
	root.appendChild(catNode);
	if (!addCategories(&catNode, dta))
		return false;
	return true;
}

bool Serializer::deSerialize(vector<PwMCategoryItem> *dta)
{
	PWM_ASSERT(domDoc);
	PWM_ASSERT(dta);
	QDomElement root(domDoc->documentElement());
	QDomNode n;

	dta->clear();
	for (n = root.firstChild(); !n.isNull(); n = n.nextSibling()) {
		// find <categories> ... </categories>
		//      <c>          ... </c>
		if (n.nodeName() == CAT_ROOT_NEW ||
		    n.nodeName() == CAT_ROOT_OLD) {
			if (!readCategories(n, dta)) {
				return false;
			}

			/* NOTE: We can stop processing here, as we
			 *       don't have more nodes in root, yet.
			 */
			return true;
		}
	}
	return false;
}

bool Serializer::readCategories(const QDomNode &n,
				vector<PwMCategoryItem> *dta)
{
	QDomNodeList nl(n.childNodes());
	QDomNode cur;
	QString name;
	unsigned int numCat = nl.count(), i;
	PwMCategoryItem curCat;
	vector<PwMDataItem> curEntr;

	if (!numCat) {
		printDebug("Serializer::readCategories(): empty");
		return false;
	}
	for (i = 0; i < numCat; ++i) {
		cur = nl.item(i);
		if (cur.nodeName().left(1) == CAT_PREFIX_NEW ||
		    cur.nodeName().left(4) == CAT_PREFIX_OLD) {
			name = cur.toElement().attribute(CAT_NAME_NEW);
			if (name == QString::null)
				name = cur.toElement().attribute(CAT_NAME_OLD);
			PWM_ASSERT(name != QString::null);
			PWM_ASSERT(name != "");
			curCat.clear();
			curCat.name = name.latin1();
			if (!readEntries(cur, &curEntr)) {
				dta->clear();
				return false;
			}
			curCat.d = curEntr;
			dta->push_back(curCat);
		} else {
			printDebug("Serializer::readCategories(): uh? not a category?");
		}
	}
	return true;
}

bool Serializer::readEntries(const QDomNode &n,
			     vector<PwMDataItem> *dta)
{
	QDomNodeList nl(n.childNodes());
	QDomNode cur;
	unsigned int numEntr = nl.count(), i;
	PwMDataItem curEntr;

	dta->clear();
	for (i = 0; i < numEntr; ++i) {
		cur = nl.item(i);
		if (cur.nodeName().left(1) == ENTRY_PREFIX_NEW ||
		    cur.nodeName().left(6) == ENTRY_PREFIX_OLD) {
			if (!extractEntry(cur, &curEntr)) {
				return false;
			}
			dta->push_back(curEntr);
		} else {
			printDebug("Serializer::readEntries(): hm? not an entry?");
		}
	}
	return true;
}

bool Serializer::extractEntry(const QDomNode &n,
			      PwMDataItem *dta)
{
	QDomNodeList nl(n.childNodes());
	QDomNode cur, cdata;
	unsigned int cnt = nl.count(), i;
	QString name, text;

	if (!cnt) {
		printDebug("Serializer::extractEntry(): empty");
		return false;
	}
	dta->clear();

	base64Data = false;
	if (n.hasAttributes()) {
		QDomAttr attr(QDomNode(n).toElement().attributeNode("b64"));
		if (!attr.isNull() && attr.value() == "1")
			base64Data = true;
	}

	for (i = 0; i < cnt; ++i) {
		cur = nl.item(i);
		name = cur.nodeName();
		cdata = cur.firstChild();
		if (likely(cdata.isCDATASection())) {
			text = cdata.toCDATASection().data();
		} else if (cur.isElement()) {
			text = cur.toElement().text();
		} else {
			printDebug("Serializer::extractEntry(): neither CDATA nor element.");
			return false;
		}
		if (name == ENTRY_DESC_NEW ||
		    name == ENTRY_DESC_OLD) {
			dta->desc = unescapeEntryData(text).latin1();
		} else if (name == ENTRY_NAME_NEW ||
			   name == ENTRY_NAME_OLD) {
			dta->name = unescapeEntryData(text).latin1();
		} else if (name == ENTRY_PW_NEW ||
			   name == ENTRY_PW_OLD) {
			dta->pw = unescapeEntryData(text).latin1();
		} else if (name == ENTRY_COMMENT_NEW ||
			   name == ENTRY_COMMENT_OLD) {
			dta->comment = unescapeEntryData(text).latin1();
		} else if (name == ENTRY_URL_NEW ||
			   name == ENTRY_URL_OLD) {
			dta->url = unescapeEntryData(text).latin1();
		} else if (name == ENTRY_LAUNCHER_NEW ||
			   name == ENTRY_LAUNCHER_OLD) {
			dta->launcher = unescapeEntryData(text).latin1();
		} else if (name == ENTRY_LVP_NEW ||
			   name == ENTRY_LVP_OLD) {
			dta->listViewPos = strtol(text.latin1(), 0, 10);
		} else if (name == ENTRY_BIN_NEW ||
			   name == ENTRY_BIN_OLD) {
			if (text == "0") {
				dta->binary = false;
			} else {
				dta->binary = true;
			}
		} else {
			printDebug(string("Serializer::extractEntry(): invalid: ")
				   + name.latin1());
		}
	}
	dta->lockStat = defaultLockStat;
	return true;
}

bool Serializer::checkValid()
{
	PWM_ASSERT(domDoc);
	QDomElement root(domDoc->documentElement());
	if (root.nodeName() != ROOT_MAGIC_NEW &&
	    root.nodeName() != ROOT_MAGIC_OLD) {
		printDebug("Serializer: wrong magic");
		return false;
	}
	if (root.attribute(VER_STR_NEW) != COMPAT_VER_NEW &&
	    root.attribute(VER_STR_OLD) != COMPAT_VER_OLD) {
		printDebug("Serializer: wrong version");
		return false;
	}
	return true;
}

QDomElement Serializer::genNewRoot()
{
	PWM_ASSERT(domDoc);
	QDomElement root(domDoc->createElement(ROOT_MAGIC_WR));
	root.setAttribute(VER_STR_WR, COMPAT_VER_WR);
	domDoc->appendChild(root);
	return root;
}

bool Serializer::addCategories(QDomElement *e,
			       const vector<PwMCategoryItem> &dta)
{
	unsigned int numCat = dta.size(), i;
	QString curId, curName;
	QDomElement curCat;

	for (i = 0; i < numCat; ++i) {
		curId = CAT_PREFIX_WR;
		curId += tostr(i).c_str();
		curName = dta[i].name.c_str();
		curCat = domDoc->createElement(curId);
		curCat.setAttribute(CAT_NAME_WR, curName);
		if (!addEntries(&curCat, dta[i].d)) {
			return false;
		}
		e->appendChild(curCat);
	}
	return true;
}

bool Serializer::addEntries(QDomElement *e,
			    const vector<PwMDataItem> &dta)
{
	unsigned int numEntr = dta.size(), i;
	QString curId;
	QDomElement curEntr;

	for (i = 0; i < numEntr; ++i) {
		curId = ENTRY_PREFIX_WR;
		curId += tostr(i).c_str();
		curEntr = domDoc->createElement(curId);
		if (base64Data)
			curEntr.setAttribute("b64", "1");
		if (!writeEntry(&curEntr, dta[i])) {
			return false;
		}
		e->appendChild(curEntr);
	}
	return true;
}

bool Serializer::writeEntry(QDomElement *e,
			    const PwMDataItem &_dta)
{
#if WRITE_CDATA_SEC != 0
# define new_text(x)	domDoc->createCDATASection(x)
	QDomCDATASection curText;
#else
# define new_text(x)	domDoc->createTextNode(x)
	QDomText curText;
#endif

	QDomText plainText;
	QDomElement tag;

	// begin -- This is for compatibility with the old serializer
	PwMDataItem dta = _dta;
	if (!dta.desc.size())
		dta.desc = " ";
	if (!dta.name.size())
		dta.name = " ";
	if (!dta.pw.size())
		dta.pw = " ";
	if (!dta.comment.size())
		dta.comment = " ";
	if (!dta.url.size())
		dta.url = " ";
	if (!dta.launcher.size())
		dta.launcher = " ";
	// end -- This is for compatibility with the old serializer

	tag = domDoc->createElement(ENTRY_DESC_WR);
	curText = new_text(escapeEntryData(dta.desc.c_str()));
	tag.appendChild(curText);
	e->appendChild(tag);

	tag = domDoc->createElement(ENTRY_NAME_WR);
	curText = new_text(escapeEntryData(dta.name.c_str()));
	tag.appendChild(curText);
	e->appendChild(tag);

	tag = domDoc->createElement(ENTRY_PW_WR);
	curText = new_text(escapeEntryData(dta.pw.c_str()));
	tag.appendChild(curText);
	e->appendChild(tag);

	tag = domDoc->createElement(ENTRY_COMMENT_WR);
	curText = new_text(escapeEntryData(dta.comment.c_str()));
	tag.appendChild(curText);
	e->appendChild(tag);

	tag = domDoc->createElement(ENTRY_URL_WR);
	curText = new_text(escapeEntryData(dta.url.c_str()));
	tag.appendChild(curText);
	e->appendChild(tag);

	tag = domDoc->createElement(ENTRY_LAUNCHER_WR);
	curText = new_text(escapeEntryData(dta.launcher.c_str()));
	tag.appendChild(curText);
	e->appendChild(tag);

	tag = domDoc->createElement(ENTRY_LVP_WR);
	plainText = domDoc->createTextNode(tostr(dta.listViewPos).c_str());
	tag.appendChild(plainText);
	e->appendChild(tag);

	tag = domDoc->createElement(ENTRY_BIN_WR);
	if (dta.binary)
		plainText = domDoc->createTextNode("1");
	else
		plainText = domDoc->createTextNode("0");
	tag.appendChild(plainText);
	e->appendChild(tag);

#undef new_text
	return true;
}

/** b64encode function from QCA. */
static QByteArray b64encode(const QByteArray &s)
{
	int i;
	int len = s.size();
	static char tbl[] =
		"ABCDEFGH"
		"IJKLMNOP"
		"QRSTUVWX"
		"YZabcdef"
		"ghijklmn"
		"opqrstuv"
		"wxyz0123"
		"456789+/"
		"=";
	int a, b, c;

	QByteArray p((len + 2) / 3 * 4);
	int at = 0;
	for(i = 0; i < len; i += 3)
	{
		a = ((unsigned char)s[i] & 3) << 4;
		if(i + 1 < len)
		{
			a += (unsigned char)s[i + 1] >> 4;
			b = ((unsigned char)s[i + 1] & 0xf) << 2;
			if(i + 2 < len)
			{
				b += (unsigned char)s[i + 2] >> 6;
				c = (unsigned char)s[i + 2] & 0x3f;
			}
			else
				c = 64;
		}
		else
			b = c = 64;

		p[at++] = tbl[(unsigned char)s[i] >> 2];
		p[at++] = tbl[a];
		p[at++] = tbl[b];
		p[at++] = tbl[c];
	}
	return p;
}

/** b64decode function from QCA. */
static QByteArray b64decode(const QByteArray &s, bool *ok)
{
	// -1 specifies invalid
	// 64 specifies eof
	// everything else specifies data

	static char tbl[] =
	{
		-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
		-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
		-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,62,-1,-1,-1,63,
		52,53,54,55,56,57,58,59,60,61,-1,-1,-1,64,-1,-1,
		-1, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12,13,14,
		15,16,17,18,19,20,21,22,23,24,25,-1,-1,-1,-1,-1,
		-1,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,
		41,42,43,44,45,46,47,48,49,50,51,-1,-1,-1,-1,-1,
		-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
		-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
		-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
		-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
		-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
		-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
		-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
		-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
	};

	// return value
	QByteArray p;
	*ok = true;

	// this should be a multiple of 4
	int len = s.size();
	if(len % 4)
	{
		*ok = false;
		return p;
	}

	p.resize(len / 4 * 3);

	int i;
	int at = 0;

	int a, b, c, d;
	c = d = 0;

	for(i = 0; i < len; i += 4)
	{
		a = tbl[(int)s[i]];
		b = tbl[(int)s[i + 1]];
		c = tbl[(int)s[i + 2]];
		d = tbl[(int)s[i + 3]];
		if((a == 64 || b == 64) || (a < 0 || b < 0 || c < 0 || d < 0))
		{
			p.resize(0);
			*ok = false;
			return p;
		}
		p[at++] = ((a & 0x3F) << 2) | ((b >> 4) & 0x03);
		p[at++] = ((b & 0x0F) << 4) | ((c >> 2) & 0x0F);
		p[at++] = ((c & 0x03) << 6) | ((d >> 0) & 0x3F);
	}

	if(c & 64)
		p.resize(at - 2);
	else if(d & 64)
		p.resize(at - 1);

	return p;
}

/** Ugly function to base64 decode a QString. */
static QString fromB64(const QString &data)
{
	QByteArray d;
	d.duplicate(data.ascii(), data.length());
	QByteArray tmp;
	bool ok;
	tmp = b64decode(d, &ok);
	if (!ok)
		return QString();
	QString ret;
	ret.setAscii(tmp.data(), tmp.size());
	return ret;
}

/** Ugly function to base64 encode a QString. */
static QString toB64(const QString &data)
{
	QByteArray d;
	d.duplicate(data.ascii(), data.length());
	QByteArray tmp;
	tmp = b64encode(d);
	QString ret;
	ret.setAscii(tmp.data(), tmp.size());
	return ret;
}

QString Serializer::escapeEntryData(QString dta)
{
	if (base64Data) {
		/* This is NOT backward compatible and has to be
		 * enabled explicitely by hand in the PwManager config file.
		 */
		return toB64(dta);
	}
	// This is the backward compatible default.
	dta.replace('\n', "$>--endl--<$");
	dta.replace("]]>", "||>");
	return dta;
}

QString Serializer::unescapeEntryData(QString dta)
{
	if (base64Data) {
		dta = fromB64(dta);
	} else {
		dta.replace("$>--endl--<$", "\n");
		dta.replace("||>", "]]>");
	}
	if (dta == " ")
		dta = ""; // for backward compatibility.
	return dta;
}
