/*
   Copyright (C) 2000, 2001, 2002 Dawit Alemayehu <adawit@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 2 as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include <unistd.h>
#include <sys/types.h>

#include <qtimer.h>
#include <qapplication.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <khbox.h>
#include <kicon.h>
#include <q3iconview.h>
#include <k3listview.h>

#include <kdebug.h>
#include <klocale.h>
#include <kcomponentdata.h>
#include <kactionmenu.h>

#include <kaction.h>
#include <kmenu.h>
#include <kmessagebox.h>
#include <kiconloader.h>

#include <kdirlister.h>
#include <k3listviewsearchline.h>
#include <k3iconviewsearchline.h>
#include <konq_dirpart.h>
#include <konq_propsview.h>
#include <kstaticdeleter.h>
#include <kgenericfactory.h>
#include <kparts/browserextension.h>
#include <kactioncollection.h>
#include "dirfilterplugin.h"

SessionManager* SessionManager::m_self = 0;
static KStaticDeleter<SessionManager> dirfiltersd;

SessionManager *SessionManager::self ()
{
  if (!m_self)
    m_self = dirfiltersd.setObject(m_self, new SessionManager);

  return m_self;
}

SessionManager::SessionManager()
{
  m_bSettingsLoaded = false;
  loadSettings ();
}

SessionManager::~SessionManager()
{
  saveSettings();
  m_self = 0;
}

QString SessionManager::generateKey (const KUrl& url)
{
  QString key;

  key = url.protocol ();
  key += ':';

  if (!url.host ().isEmpty ())
  {
    key += url.host ();
    key += ':';
  }

  key += url.path ();
  key += ':';
  key += QString::number (m_pid);

  return key;
}

QStringList SessionManager::restore (const KUrl& url)
{
  QString key = generateKey (url);

  if (m_filters.contains(key))
    return m_filters[key];
  else
    return QStringList ();
}

void SessionManager::save (const KUrl& url, const QStringList& filters)
{
  QString key = generateKey(url);
  m_filters[key] = filters;
}

void SessionManager::saveSettings()
{
  KConfig cfg ("dirfilterrc", KConfig::NoGlobals);
  KConfigGroup group = cfg.group ("General");

  group.writeEntry ("ShowCount", showCount);
  group.writeEntry ("UseMultipleFilters", useMultipleFilters);
  cfg.sync();
}

void SessionManager::loadSettings()
{
  if (m_bSettingsLoaded)
    return;

  KConfig cfg ("dirfilterrc", KConfig::NoGlobals);
  KConfigGroup group = cfg.group ("General");

  showCount = group.readEntry ("ShowCount", false);
  useMultipleFilters = group.readEntry ("UseMultipleFilters", true);
  m_pid = getpid ();
  m_bSettingsLoaded = true;
}



DirFilterPlugin::DirFilterPlugin (QObject* parent,
                                  const QStringList &)
                :KParts::Plugin (parent), m_pFilterMenu(0)
{
  m_part = qobject_cast<KonqDirPart*>(parent);

  if ( !m_part || !m_part->scrollWidget() )
    return;

  m_pFilterMenu = new KActionMenu (KIcon("search-filter"),i18n("View F&ilter"),
                                   actionCollection());
  actionCollection()->addAction("filterdir", m_pFilterMenu);
  m_pFilterMenu->setDelayed (false);
  m_pFilterMenu->setWhatsThis(i18n("Allow to filter the currently displayed items by filetype."));

  connect (m_pFilterMenu->menu(), SIGNAL (aboutToShow()),
           SLOT (slotShowPopup()));

  connect (m_part, SIGNAL (itemRemoved(const KFileItem*)),
           SLOT( slotItemRemoved (const KFileItem*)));
  connect (m_part, SIGNAL (itemsAdded(const KFileItemList&)),
           SLOT (slotItemsAdded(const KFileItemList&)));
  connect (m_part, SIGNAL (itemsFilteredByMime(const KFileItemList&)),
           SLOT (slotItemsAdded(const KFileItemList&)));
  connect (m_part, SIGNAL(aboutToOpenURL()), SLOT(slotOpenURL()));

  // add a searchline filter for konqis icons/list views
  KHBox *hbox = new KHBox(m_part->widget());
  hbox->hide();

  QAction *clear = actionCollection()->addAction("clear_filter");
  clear->setText(i18n("Clear Filter Field"));
  clear->setIcon(KIcon(QApplication::isRightToLeft() ? "clear-left" : "locationbar-erase"));
  clear->setWhatsThis(i18n("Clear filter field<p>Clears the content of the filter field."));

  QWidget *search = 0;
  // TODO port to the non-q3 classes
  if ( m_part->scrollWidget()->inherits("Q3IconView") ) // TODO qobject_cast
  {
    search = new K3IconViewSearchLine(hbox);
    static_cast<K3IconViewSearchLine*>(search)->setIconView(static_cast<Q3IconView*>(m_part->scrollWidget()));
  }
  else if ( m_part->scrollWidget()->inherits("K3ListView") )
  {
    search = new K3ListViewSearchLine(hbox);
    static_cast<K3ListViewSearchLine*>(search)->setListView(static_cast<K3ListView*>(m_part->scrollWidget()));
  }

  if ( search )
  {
    search->setWhatsThis( i18n("Enter here a text which an item in the view must contain anywhere to be shown."));
    connect(clear, SIGNAL(activated()), search, SLOT(clear()));
  }

  QAction *filterAction = actionCollection()->addAction("toolbar_filter_field");
  filterAction->setText(i18n("Filter Field"));
  qobject_cast<KAction*>(filterAction)->setDefaultWidget( hbox );
  qobject_cast<KAction*>(filterAction)->setShortcutConfigurable(false);
}

DirFilterPlugin::~DirFilterPlugin()
{
  delete m_pFilterMenu;
}

void DirFilterPlugin::slotOpenURL ()
{
  KUrl url = m_part->url();

  //kDebug(90190) << "DirFilterPlugin: New URL    : " << url.url() << endl;
  //kDebug(90190) << "DirFilterPlugin: Current URL: " << m_pURL.url() << endl;

  if (m_pURL != url)
  {
    m_pURL = url;
    m_pMimeInfo.clear();
    m_part->setMimeFilter (SessionManager::self()->restore(url));
  }
}

void DirFilterPlugin::slotShowPopup()
{
  if (!m_part)
  {
    m_pFilterMenu->setEnabled (false);
    return;
  }

  int id = 0;
  uint enableReset = 0;

  QString label;
  QStringList inodes;

  m_pFilterMenu->menu()->clear();
  m_pFilterMenu->menu()->addTitle (i18n("Only Show Items of Type"));

  for (MimeInfoIterator it = m_pMimeInfo.begin(); it != m_pMimeInfo.end() ; ++it)
  {
    if (it.key().startsWith("inode"))
    {
      inodes << it.key();
      continue;
    }

    if (!SessionManager::self()->showCount)
      label = it.data().mimeComment;
    else
    {
      label = it.data().mimeComment;
      label += "  (";
      label += QString::number (it.data().filenames.size ());
      label += ')';
    }

    m_pMimeInfo[it.key()].id = m_pFilterMenu->menu()->insertItem (
                               KIcon(it.data().iconName), label,
                               this, SLOT(slotItemSelected(int)), 0, ++id);

    if (it.data().useAsFilter)
    {
      m_pFilterMenu->menu()->setItemChecked (id, true);
      enableReset++;
    }
  }

  // Add all the items that have mime-type of "inode/*" here...
  if (!inodes.isEmpty())
  {
    m_pFilterMenu->menu()->insertSeparator ();

    for (QStringList::Iterator it = inodes.begin(); it != inodes.end(); ++it)
    {
      if (!SessionManager::self()->showCount)
        label = m_pMimeInfo[(*it)].mimeComment;
      else
      {
        label = m_pMimeInfo[(*it)].mimeComment;
        label += "  (";
        label += QString::number (m_pMimeInfo[(*it)].filenames.size ());
        label += ')';
      }

      m_pMimeInfo[(*it)].id = m_pFilterMenu->menu()->insertItem (
                              KIcon(m_pMimeInfo[(*it)].iconName), label,
                              this, SLOT(slotItemSelected(int)), 0, ++id);

      if (m_pMimeInfo[(*it)].useAsFilter)
      {
        m_pFilterMenu->menu()->setItemChecked (id, true);
        enableReset ++;
      }
    }
  }

  m_pFilterMenu->menu()->addSeparator ();
  id = m_pFilterMenu->menu()->insertItem (i18n("Use Multiple Filters"),
                                               this, SLOT(slotMultipleFilters()));
  m_pFilterMenu->menu()->setItemEnabled (id, enableReset <= 1);
  m_pFilterMenu->menu()->setItemChecked (id, SessionManager::self()->useMultipleFilters);

  id = m_pFilterMenu->menu()->insertItem (i18n("Show Count"), this,
                                               SLOT(slotShowCount()));
  m_pFilterMenu->menu()->setItemChecked (id, SessionManager::self()->showCount);

  id = m_pFilterMenu->menu()->insertItem (i18n("Reset"), this,
                                               SLOT(slotReset()));

  m_pFilterMenu->menu()->setItemEnabled (id, enableReset);
}

void DirFilterPlugin::slotItemSelected (int id)
{
  if (!m_part)
    return;

  MimeInfoIterator it = m_pMimeInfo.begin();
  while (it != m_pMimeInfo.end () && id != it.data().id)
    it++;

  if (it != m_pMimeInfo.end())
  {
    QStringList filters;

    if (it.data().useAsFilter)
    {
      it.data().useAsFilter = false;
      filters = m_part->mimeFilter ();
      if (filters.remove (it.key()))
        m_part->setMimeFilter (filters);
    }
    else
    {
      m_pMimeInfo[it.key()].useAsFilter = true;

      if (SessionManager::self()->useMultipleFilters)
      {
        filters = m_part->mimeFilter ();
        filters << it.key();
      }
      else
      {
        filters << it.key();

        MimeInfoIterator item = m_pMimeInfo.begin();
        while ( item != m_pMimeInfo.end() )
        {
          if ( item != it )
            item.data().useAsFilter = false;
          item++;
        }
      }

      m_part->setMimeFilter (filters);
    }

    KUrl url = m_part->url();
    m_part->openUrl (url);
    SessionManager::self()->save (url, filters);
  }
}

void DirFilterPlugin::slotItemsAdded (const KFileItemList& list)
{
 if (list.count() == 0 || !m_part || !m_part->nameFilter().isEmpty())
  {
    if (m_part) m_pFilterMenu->setEnabled (m_part->nameFilter().isEmpty());
    return;
  }

  KUrl url = m_part->url();
 
  // Make sure the filter menu is enabled once a named
  // filter is removed.
  if (!m_pFilterMenu->isEnabled())
    m_pFilterMenu->setEnabled (true);

  KFileItemList::const_iterator kit = list.begin();
  const KFileItemList::const_iterator kend = list.end();
  for (; kit != kend; ++kit )
  {
    QString name = (*kit)->name();
    QString mimeType = (*kit)->mimetype();

    if (mimeType.isEmpty ())
      continue;

    if (!m_pMimeInfo.contains (mimeType))
    {
      QStringList filters = m_part->mimeFilter ();
      m_pMimeInfo[mimeType].useAsFilter = (!filters.isEmpty () &&
                                           filters.contains (mimeType));
      m_pMimeInfo[mimeType].mimeComment = (*kit)->mimeComment();
      m_pMimeInfo[mimeType].iconName = (*kit)->iconName();
      m_pMimeInfo[mimeType].filenames.insert(name, false);
    }
    else
    {
      m_pMimeInfo[mimeType].filenames.insert(name, false);
    }
  }
}

void DirFilterPlugin::slotItemRemoved (const KFileItem* item)
{
  if (!item || !m_part)
    return;

  QString mimeType = item->mimetype().trimmed();

  if (m_pMimeInfo.contains (mimeType))
  {
    MimeInfo info = m_pMimeInfo [mimeType];

    if (info.filenames.size () > 1)
      m_pMimeInfo [mimeType].filenames.remove (item->name ());
    else
    {
      if (info.useAsFilter)
      {
        QStringList filters = m_part->mimeFilter ();
        filters.remove (mimeType);
        m_part->setMimeFilter (filters);
        SessionManager::self()->save (m_part->url(), filters);
        QTimer::singleShot( 0, this, SLOT(slotTimeout()) );
      }
      m_pMimeInfo.remove (mimeType);
    }
  }
}

void DirFilterPlugin::slotReset()
{
  if (!m_part)
    return;

  MimeInfoIterator it = m_pMimeInfo.begin();
  for (; it != m_pMimeInfo.end(); ++it)
    it.data().useAsFilter = false;

  QStringList filters;
  KUrl url = m_part->url();

  m_part->setMimeFilter (filters);
  m_part->openUrl (url);
  SessionManager::self()->save (m_part->url(), filters);
}

void DirFilterPlugin::slotShowCount()
{
  if (SessionManager::self()->showCount)
    SessionManager::self()->showCount = false;
  else
    SessionManager::self()->showCount = true;
}

void DirFilterPlugin::slotMultipleFilters()
{
  if (SessionManager::self()->useMultipleFilters)
    SessionManager::self()->useMultipleFilters = false;
  else
    SessionManager::self()->useMultipleFilters = true;
}

void DirFilterPlugin::slotTimeout()
{
  if (m_part)
    m_part->openUrl (m_part->url());
}

typedef KGenericFactory<DirFilterPlugin> DirFilterFactory;
K_EXPORT_COMPONENT_FACTORY (libdirfilterplugin, DirFilterFactory("dirfilterplugin"))

#include "dirfilterplugin.moc"
