/***************************************************************************
                          observinglist.cpp  -  K Desktop Planetarium
                             -------------------
    begin                : 29 Nov 2004
    copyright            : (C) 2004 by Jeff Woods, Jason Harris
    email                : jcwoods@bellsouth.net, jharris@30doradus.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdio.h>

#include <QFile>
#include <QDir>
//#include <qlabel.h>
//#include <qlayout.h>
//#include <qstringlist.h>
//#include <q3widgetstack.h>
#include <QFrame>
#include <QTextStream>

#include <k3listview.h>
#include <kpushbutton.h>
#include <kstatusbar.h>
#include <ktextedit.h>
#include <kinputdialog.h>
#include <kicon.h>
#include <kio/netaccess.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include <ktemporaryfile.h>
#include <klineedit.h>

#include "observinglist.h"
#include "obslistwizard.h"
#include "kstars.h"
#include "kstarsdata.h"
#include "skyobject.h"
#include "skymap.h"
#include "detaildialog.h"
#include "tools/altvstime.h"

#include "indimenu.h"
#include "indielement.h"
#include "indiproperty.h"
#include "indidevice.h"
#include "devicemanager.h"
#include "indistd.h"

ObservingListUI::ObservingListUI( QWidget *p ) : QFrame( p ) {
  setupUi( this );
}

ObservingList::ObservingList( KStars *_ks )
    : KDialog( (QWidget*)_ks ),
      ks( _ks ), LogObject(0), m_CurrentObject(0),
      noNameStars(0), isModified(false), bIsLarge(true)
{
	ui = new ObservingListUI( this );
	setMainWidget( ui );
        setCaption( i18n( "Observing List" ) );
        setButtons( KDialog::Close );

	//Set up the QTableWidget
	ui->FullTable->setColumnCount( 5 );
	ui->FullTable->setHorizontalHeaderLabels(
			QStringList() << i18n( "Name" ) << i18nc( "Right Ascension", "RA" ) << i18nc( "Declination", "Dec" )
			<< i18nc( "Magnitude", "Mag" ) << i18n( "Type" ) );

	//make sure widget stack starts with FullTable shown
	ui->TableStack->setCurrentWidget( ui->FullPage );

	//Connections
	connect( this, SIGNAL( closeClicked() ), this, SLOT( slotClose() ) );
	connect( ui->FullTable, SIGNAL( itemSelectionChanged() ),
			this, SLOT( slotNewSelection() ) );
	connect( ui->TinyTable, SIGNAL( itemSelectionChanged() ),
			this, SLOT( slotNewSelection() ) );
	connect( ui->FullTable, SIGNAL( itemDoubleClicked( QTableWidgetItem* ) ),
			this, SLOT( slotCenterObject() ) );
	connect( ui->TinyTable, SIGNAL( itemDoubleClicked( QListWidgetItem* ) ),
			this, SLOT( slotCenterObject() ) );
	connect( ui->RemoveButton, SIGNAL( clicked() ),
			this, SLOT( slotRemoveObjects() ) );
	connect( ui->CenterButton, SIGNAL( clicked() ),
			this, SLOT( slotCenterObject() ) );
	connect( ui->ScopeButton, SIGNAL( clicked() ),
			this, SLOT( slotSlewToObject() ) );
	connect( ui->DetailsButton, SIGNAL( clicked() ),
			this, SLOT( slotDetails() ) );
	connect( ui->AVTButton, SIGNAL( clicked() ),
			this, SLOT( slotAVT() ) );

	connect( ui->OpenButton, SIGNAL( clicked() ),
			this, SLOT( slotOpenList() ) );
	connect( ui->SaveButton, SIGNAL( clicked() ),
			this, SLOT( slotSaveList() ) );
	connect( ui->SaveAsButton, SIGNAL( clicked() ),
			this, SLOT( slotSaveListAs() ) );
	connect( ui->WizardButton, SIGNAL( clicked() ),
			this, SLOT( slotWizard() ) );
	connect( ui->MiniButton, SIGNAL( clicked() ),
			this, SLOT( slotToggleSize() ) );

	//Add icons to Push Buttons
	ui->OpenButton->setIcon( KIcon("fileopen") );
	ui->SaveButton->setIcon( KIcon("filesave") );
	ui->SaveAsButton->setIcon( KIcon("filesaveas") );
	ui->WizardButton->setIcon( KIcon("wizard") );
	ui->MiniButton->setIcon( KIcon("window_nofullscreen") );

	ui->CenterButton->setEnabled( false );
	ui->ScopeButton->setEnabled( false );
	ui->DetailsButton->setEnabled( false );
	ui->AVTButton->setEnabled( false );
	ui->RemoveButton->setEnabled( false );
	ui->NotesLabel->setEnabled( false );
	ui->NotesEdit->setEnabled( false );
}

bool ObservingList::contains( const SkyObject *q ) {
	foreach ( SkyObject* o, obsList() ) {
		if ( o == q ) return true;
	}

	return false;
}


//SLOTS

void ObservingList::slotAddObject( SkyObject *obj ) {
	if ( ! obj ) obj = ks->map()->clickedObject();

	//First, make sure object is not already in the list
	foreach ( SkyObject *o, obsList() ) {
		if ( obj == o ) {
			ks->statusBar()->changeItem( i18n( "%1 is already in the observing list.", obj->name() ), 0 );
			return;
		}
	}

	//Insert object in obsList
	m_ObservingList.append( obj );
	m_CurrentObject = obj;
	if ( ! isModified ) isModified = true;

	//Set string for magnitude
	QString smag("--");
	if ( obj->mag() < 90.0 ) smag = QString::number( obj->mag(), 'g', 2 );

	//Insert object entry in FullTable
	//First, add a row
	int irow = ui->FullTable->rowCount();
	ui->FullTable->insertRow( irow );
	ui->FullTable->setItem( irow, 0, new QTableWidgetItem( obj->translatedName() ) );
	ui->FullTable->setItem( irow, 1, new QTableWidgetItem( obj->ra()->toHMSString() ) );
	ui->FullTable->setItem( irow, 2, new QTableWidgetItem( obj->dec()->toDMSString() ) );
	ui->FullTable->setItem( irow, 3, new QTableWidgetItem( smag ) );
	ui->FullTable->setItem( irow, 4, new QTableWidgetItem( obj->typeName() ) );

	//Insert object entry in TinyTable
	ui->TinyTable->insertItem( irow, obj->translatedName() );

	//Note addition in statusbar
	ks->statusBar()->changeItem( i18n( "Added %1 to observing list.", obj->name() ), 0 );
}

void ObservingList::slotRemoveObject( SkyObject *o ) {
	if ( !o )
		o = ks->map()->clickedObject();

	int i = obsList().indexOf( o );
	if ( i < 0 ) return; //object not in observing list

	m_ObservingList.removeAll(o);
	if ( ! isModified ) isModified = true;

	if ( o == LogObject )
		saveCurrentUserLog();

	//Remove entry from FullTable
	bool objectFound = false;
	for ( i=0; i<ui->FullTable->rowCount(); ++i ) {

		//If the object is named "star" then match coordinates instead of name
		if ( o->translatedName() == i18n( "star" ) ) {
			if ( ui->FullTable->item( i, 1 )->text() == o->ra()->toHMSString()
				&& ui->FullTable->item( i, 2 )->text() == o->dec()->toDMSString() ) {
				ui->FullTable->removeRow( i );
				objectFound = true;
				break;
			}

		//Otherwise match by name
		} else {
			if ( ui->FullTable->item( i, 0 )->text() == o->translatedName() ) {
				ui->FullTable->removeRow( i ); //using same i from FullTable for-loop
				objectFound = true;
				break;
			}
		}
	}

	if ( ! objectFound ) {
		kDebug() << i18n( "Cannot remove Object %1; not found in table." , o->translatedName()) << endl;
	} else {
		//Remove object from TinyTable also
		delete ui->TinyTable->takeItem(i);
	}
}

void ObservingList::slotRemoveObjects() {
	if ( m_SelectedObjects.size() == 0) return;

	foreach ( SkyObject *o, m_SelectedObjects )
		slotRemoveObject( o );

	slotNewSelection();
}

void ObservingList::slotNewSelection() {
	//If the TinyTable is visible, we need to sync the selection in the FullTable
	if ( sender() == ui->TinyTable ) syncTableSelection();

	//Construct list of selected objects
	m_SelectedObjects.clear();
	for ( int i=0; i < ui->FullTable->rowCount(); ++i ) {
		QTableWidgetItem *itName = ui->FullTable->item( i, 0 );
		if ( ui->FullTable->isItemSelected( itName ) ) {
			foreach ( SkyObject *o, obsList() ) {
				if ( itName->text() == i18n( "star" ) ) {
					if ( ui->FullTable->item( i, 1 )->text() == o->ra()->toHMSString()
						&& ui->FullTable->item( i, 2 )->text() == o->dec()->toDMSString() ) {
						m_SelectedObjects.append( o );
						break;
					}
				} else if ( itName->text() == o->translatedName() ) {
					m_SelectedObjects.append( o );
					break;
				}
			}
		}
	}

	//Enable widgets when one object selected
	if ( m_SelectedObjects.size() == 1 ) {
		QString newName( m_SelectedObjects[0]->translatedName() );

		//Enable buttons
		ui->CenterButton->setEnabled( true );
		ui->ScopeButton->setEnabled( true );
		ui->DetailsButton->setEnabled( true );
		ui->AVTButton->setEnabled( true );
		ui->RemoveButton->setEnabled( true );

		//Find the selected object in the obsList,
		//then break the loop.  Now obsList.current()
		//points to the new selected object (until now it was the previous object)
		bool found = obsList().contains( m_SelectedObjects[0] );
		if ( found ) m_CurrentObject = m_SelectedObjects[0];

		if ( ! found ) {
			kDebug() << i18n( "Object %1 not found in observing ist.", newName ) << endl;
		} else if ( newName != i18n( "star" ) ) {
			//Display the object's current user notes in the NotesEdit
			//First, save the last object's user log to disk, if necessary
			saveCurrentUserLog();

			//set LogObject to the new selected object
			LogObject = currentObject();

			ui->NotesLabel->setEnabled( true );
			ui->NotesEdit->setEnabled( true );

			ui->NotesLabel->setText( i18n( "observing notes for %1:", LogObject->translatedName() ) );
			if ( LogObject->userLog.isEmpty() ) {
				ui->NotesEdit->setPlainText( i18n("Record here observation logs and/or data on %1.", LogObject->translatedName() ) );
			} else {
				ui->NotesEdit->setPlainText( LogObject->userLog );
			}
		} else { //selected object is named "star"
			//clear the log text box
			saveCurrentUserLog();
			ui->NotesLabel->setText( i18n( "observing notes (disabled for unnamed star)" ) );
			ui->NotesLabel->setEnabled( false );
			ui->NotesEdit->clear();
			ui->NotesEdit->setEnabled( false );
		}

	} else if ( m_SelectedObjects.size() == 0 ) {
		//Disable buttons
		ui->CenterButton->setEnabled( false );
		ui->ScopeButton->setEnabled( false );
		ui->DetailsButton->setEnabled( false );
		ui->AVTButton->setEnabled( false );
		ui->RemoveButton->setEnabled( false );
		ui->NotesLabel->setEnabled( false );
		ui->NotesEdit->setEnabled( false );
		m_CurrentObject = 0;

		//Clear the user log text box.
		saveCurrentUserLog();
	} else { //more than one object selected.
		ui->CenterButton->setEnabled( false );
		ui->ScopeButton->setEnabled( false );
		ui->DetailsButton->setEnabled( false );
		ui->AVTButton->setEnabled( true );
		ui->RemoveButton->setEnabled( true );
		ui->NotesLabel->setEnabled( false );
		ui->NotesEdit->setEnabled( false );
		m_CurrentObject = 0;

		//Clear the user log text box.
		saveCurrentUserLog();
	}

}

void ObservingList::slotCenterObject() {
	if ( currentObject() ) {
		ks->map()->setClickedObject( currentObject() );
		ks->map()->setClickedPoint( currentObject() );
		ks->map()->slotCenter();
	}
}

void ObservingList::slotSlewToObject()
{

  INDI_D *indidev(NULL);
  INDI_P *prop(NULL), *onset(NULL);
  INDI_E *RAEle(NULL), *DecEle(NULL), *AzEle(NULL), *AltEle(NULL), *ConnectEle(NULL), *nameEle(NULL);
  bool useJ2000( false);
  int selectedCoord(0);
  SkyPoint sp;

  // Find the first device with EQUATORIAL_EOD_COORD or EQUATORIAL_COORD and with SLEW element
  // i.e. the first telescope we find!

  INDIMenu *imenu = ks->getINDIMenu();


  for (int i=0; i < imenu->mgr.size() ; i++)
  {
    for (int j=0; j < imenu->mgr.at(i)->indi_dev.size(); j++)
    {
       indidev = imenu->mgr.at(i)->indi_dev.at(j);
       indidev->stdDev->currentObject = NULL;
       prop = indidev->findProp("EQUATORIAL_EOD_COORD");
       	if (prop == NULL)
	{
		  prop = indidev->findProp("EQUATORIAL_COORD");
		  if (prop == NULL)
                  {
                    prop = indidev->findProp("HORIZONTAL_COORD");
                    if (prop == NULL)
        		continue;
                    else
                        selectedCoord = 1;		/* Select horizontal */
                  }
		  else
		        useJ2000 = true;
	}

       ConnectEle = indidev->findElem("CONNECT");
       if (!ConnectEle) continue;

       if (ConnectEle->state == PS_OFF)
       {
	 KMessageBox::error(0, i18n("Telescope %1 is offline. Please connect and retry again.", indidev->label));
	 return;
       }

        switch (selectedCoord)
        {
          // Equatorial
          case 0:
           if (prop->perm == PP_RO) continue;
           RAEle  = prop->findElement("RA");
       	   if (!RAEle) continue;
   	   DecEle = prop->findElement("DEC");
   	   if (!DecEle) continue;
           break;

         // Horizontal
         case 1:
          if (prop->perm == PP_RO) continue;
          AzEle = prop->findElement("AZ");
          if (!AzEle) continue;
          AltEle = prop->findElement("ALT");
          if (!AltEle) continue;
          break;
        }

        onset = indidev->findProp("ON_COORD_SET");
        if (!onset) continue;

        onset->activateSwitch("SLEW");

        indidev->stdDev->currentObject = currentObject();

      /* Send object name if available */
      if (indidev->stdDev->currentObject)
         {
             nameEle = indidev->findElem("OBJECT_NAME");
             if (nameEle && nameEle->pp->perm != PP_RO)
             {
                 nameEle->write_w->setText(indidev->stdDev->currentObject->name());
                 nameEle->pp->newText();
             }
          }

       switch (selectedCoord)
       {
         case 0:
            if (indidev->stdDev->currentObject)
		sp.set (indidev->stdDev->currentObject->ra(), indidev->stdDev->currentObject->dec());
            else
                sp.set (ks->map()->clickedPoint()->ra(), ks->map()->clickedPoint()->dec());

      	if (useJ2000)
	    sp.apparentCoord(ks->data()->ut().djd(), (long double) J2000);

    	   RAEle->write_w->setText(QString("%1:%2:%3").arg(sp.ra()->hour()).arg(sp.ra()->minute()).arg(sp.ra()->second()));
	   DecEle->write_w->setText(QString("%1:%2:%3").arg(sp.dec()->degree()).arg(sp.dec()->arcmin()).arg(sp.dec()->arcsec()));

          break;

       case 1:
         if (indidev->stdDev->currentObject)
         {
           sp.setAz(*indidev->stdDev->currentObject->az());
           sp.setAlt(*indidev->stdDev->currentObject->alt());
         }
         else
         {
           sp.setAz(*ks->map()->clickedPoint()->az());
           sp.setAlt(*ks->map()->clickedPoint()->alt());
         }

          AzEle->write_w->setText(QString("%1:%2:%3").arg(sp.az()->degree()).arg(sp.az()->arcmin()).arg(sp.az()->arcsec()));
          AltEle->write_w->setText(QString("%1:%2:%3").arg(sp.alt()->degree()).arg(sp.alt()->arcmin()).arg(sp.alt()->arcsec()));

         break;
       }

       prop->newText();

       return;
    }
  }

  // We didn't find any telescopes
  KMessageBox::sorry(0, i18n("KStars did not find any active telescopes."));

}

//FIXME: This will open multiple Detail windows for each object;
//Should have one window whose target object changes with selection
void ObservingList::slotDetails() {
	if ( currentObject() ) {
		DetailDialog dd( currentObject(), ks->data()->lt(), ks->geo(), ks );
		dd.exec();
	}
}

void ObservingList::slotAVT() {
	if ( m_SelectedObjects.size() ) {
		AltVsTime avt( ks );
		foreach ( SkyObject *o, m_SelectedObjects ) {
			avt.processObject( o );
		}

		avt.exec();
	}
}

//FIXME: On close, we will need to close any open Details/AVT windows
void ObservingList::slotClose() {
	//Save the current User log text
	if ( currentObject() && ! ui->NotesEdit->toPlainText().isEmpty() && ui->NotesEdit->toPlainText()
					!= i18n("Record here observation logs and/or data on %1.", currentObject()->name()) ) {
		currentObject()->saveUserLog( ui->NotesEdit->toPlainText() );
	}

	hide();
}

void ObservingList::saveCurrentUserLog() {
	if ( ! ui->NotesEdit->toPlainText().isEmpty() &&
				ui->NotesEdit->toPlainText() !=
					i18n("Record here observation logs and/or data on %1.", LogObject->translatedName() ) ) {
		LogObject->saveUserLog( ui->NotesEdit->toPlainText() );

		ui->NotesEdit->clear();
		ui->NotesLabel->setText( i18n( "Observing notes for object:" ) );
		LogObject = NULL;
	}
}

void ObservingList::slotOpenList() {
	KUrl fileURL = KFileDialog::getOpenUrl( QDir::homePath(), "*.obslist|KStars Observing List (*.obslist)" );
	QFile f;

	if ( fileURL.isValid() ) {
		if ( ! fileURL.isLocalFile() ) {
			//Save remote list to a temporary local file
			KTemporaryFile tmpfile;
			tmpfile.setAutoRemove(false);
			tmpfile.open();
			FileName = tmpfile.fileName();
			if( KIO::NetAccess::download( fileURL, FileName, this ) )
				f.setFileName( FileName );

		} else {
			FileName = fileURL.path();
			f.setFileName( FileName );
		}

		if ( !f.open( QIODevice::ReadOnly) ) {
			QString message = i18n( "Could not open file %1", f.fileName() );
			KMessageBox::sorry( 0, message, i18n( "Could Not Open File" ) );
			return;
		}

		saveCurrentList();
		//First line is the name of the list.  The rest of the file should
		//be object names, one per line.
		QTextStream istream(&f);
		QString line;
		ListName = istream.readLine();

		while ( ! istream.atEnd() ) {
			line = istream.readLine();
			SkyObject *o = ks->data()->objectNamed( line );
			if ( o ) slotAddObject( o );
		}

		//Newly-opened list should not trigger isModified flag
		isModified = false;

		f.close();

	} else if ( fileURL.path() != QString() ) {
		QString message = i18n( "The specified file is invalid.  Try another file?" );
		if ( KMessageBox::warningYesNo( this, message, i18n("Invalid File"), KGuiItem(i18n("Try Another")), KGuiItem(i18n("Do Not Try")) ) == KMessageBox::Yes ) {
			slotOpenList();
		}
	}
}

void ObservingList::saveCurrentList() {
	//Before loading a new list, do we need to save the current one?
	//Assume that if the list is empty, then there's no need to save
	if ( obsList().size() ) {
		if ( isModified ) {
			QString message = i18n( "Do you want to save the current list before opening a new list?" );
			if ( KMessageBox::questionYesNo( this, message,
					i18n( "Save Current List?" ), KStandardGuiItem::save(), KStandardGuiItem::discard() ) == KMessageBox::Yes )
				slotSaveList();
		}

		//If we ever allow merging the loaded list with
		//the existing one, that code would go here
		m_ObservingList.clear();
		m_CurrentObject = 0;
		ui->FullTable->clear();
	}
}

void ObservingList::slotSaveListAs() {
	bool ok(false);
	ListName = KInputDialog::getText( i18n( "Enter List Name" ),
			i18n( "List name:" ), QString(), &ok );

	if ( ok ) {
		KUrl fileURL = KFileDialog::getSaveUrl( QDir::homePath(), "*.obslist|KStars Observing List (*.obslist)" );

		if ( fileURL.isValid() )
			FileName = fileURL.path();

		slotSaveList();
	}
}

void ObservingList::slotSaveList() {
	if ( ListName.isEmpty() || FileName.isEmpty() ) {
		slotSaveListAs();
		return;
	}

	QFile f( FileName );
	if ( !f.open( QIODevice::WriteOnly) ) {
		QString message = i18n( "Could not open file %1.  Try a different filename?", f.fileName() );

		if ( KMessageBox::warningYesNo( 0, message, i18n( "Could Not Open File" ), KGuiItem(i18n("Try Different")), KGuiItem(i18n("Do Not Try")) ) == KMessageBox::Yes ) {
			FileName == QString();
			slotSaveList();
		}
		return;
	}

	QTextStream ostream(&f);
	ostream << ListName << endl;
	foreach ( SkyObject* o, obsList() ) {
		ostream << o->name() << endl;
	}

	f.close();
	isModified = false;
}

void ObservingList::slotWizard() {
	ObsListWizard wizard( ks );
	if ( wizard.exec() == QDialog::Accepted ) {
		//Make sure current list is saved
		saveCurrentList();

		foreach ( SkyObject *o, wizard.obsList() ) {
			slotAddObject( o );
		}
	}
}

void ObservingList::slotToggleSize() {
	if ( isLarge() ) {
		ui->MiniButton->setIcon( KIcon("window_fullscreen") );

		//switch widget stack to show TinyTable
		ui->TableStack->setCurrentWidget( ui->TinyPage );

		//Abbreviate text on each button
		ui->CenterButton->setText( i18nc( "First letter in 'Center'", "C" ) );
		ui->ScopeButton->setText( i18nc( "First letter in 'Scope'", "S" ) );
		ui->DetailsButton->setText( i18nc( "First letter in 'Details'", "D" ) );
		ui->AVTButton->setText( i18nc( "First letter in 'Alt vs Time'", "A" ) );
		ui->RemoveButton->setText( i18nc( "First letter in 'Remove'", "R" ) );

		//Hide Observing notes
		ui->NotesLabel->hide();
		ui->NotesEdit->hide();

		syncTableSelection( false ); //sync TinyTable with FullTable
		adjustSize();
		bIsLarge = false;

	} else {
		ui->MiniButton->setIcon( KIcon("window_nofullscreen") );

		//switch widget stack to show FullTable
		ui->TableStack->setCurrentWidget( ui->FullPage );

		//Expand text on each button
		ui->CenterButton->setText( i18n( "Center" ) );
		ui->ScopeButton->setText( i18n( "Scope" ) );
		ui->DetailsButton->setText( i18n( "Details" ) );
		ui->AVTButton->setText( i18n( "Alt vs Time" ) );
		ui->RemoveButton->setText( i18n( "Remove" ) );

		//Show Observing notes
		ui->NotesLabel->show();
		ui->NotesEdit->show();

		syncTableSelection( true ); //sync FullTable with TinyTable
		adjustSize();
		bIsLarge = true;
	}
}

//FullTable uses SelectionBehavior=SelectRows, so selecting the first cell
//in a row should select the whole row
void ObservingList::syncTableSelection( bool syncFullTable ) {
	for ( int i=0; i<ui->FullTable->rowCount(); ++i ) {
		if ( syncFullTable ) {
			ui->FullTable->setItemSelected( ui->FullTable->item( i, 0 ), ui->TinyTable->isItemSelected( ui->TinyTable->item(i) ) );
		} else {
			ui->TinyTable->setItemSelected( ui->TinyTable->item(i), ui->FullTable->isItemSelected( ui->FullTable->item( i, 0 ) ) );
		}
	}
}

#include "observinglist.moc"
