/**
 * Copyright Michel Filippi <mfilippi@sade.rhein-main.de>
 *           Robert Williams
 *           Andrew Chant <andrew.chant@utoronto.ca>
 *           André Luiz dos Santos <andre@netvision.com.br>
 * Copyright (C) 2004-2005 Benjamin C. Meyer <ben at meyerhome dot net>
 *
 * This file is part of the ksnake package
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

/*
  note: the code to lookup and insert the pixmaps
  into the Options menu was copied and adapted
  from KReversi.
  thanks.
  */
#include <QDir>
#include <QRegExp>
#include <kxmlguifactory.h>


#include <klocale.h>
#include <kconfigdialog.h>

#include <kstandardgameaction.h>
#include <kscoredialog.h>
#include <kstatusbar.h>
#include <kstandardaction.h>
#include <ktoggleaction.h>
#include <kactioncollection.h>

#include "rattler.h"
#include "game.h"
#include "startroom.h"
#include "levels.h"
#include "progress.h"
#include "view.h"
#include "settings.h"
#include "ui_appearance.h"
#include "ui_general.h"

#define SCORE 1
#define LIVES 2

class Appearance : public QWidget, public Ui::Appearance
{
public:
	Appearance(QWidget* parent)
		: QWidget(parent)
	{
		setupUi(this);
	}
};

class General : public QWidget, public Ui::General
{
public:
	General(QWidget* parent)
		: QWidget(parent)
	{
		setupUi(this);
	}
};

Game::Game(QWidget *parent) :  KMainWindow(parent)
{
	// create statusbar
	statusBar()->insertItem(i18n("Score: 0"), SCORE);
	statusBar()->insertItem(i18n("Lives: 0"), LIVES);

	levels = new Levels();

	view = new View(this);
	rattler = view->rattler;
	rattler->reloadRoomPixmap();
	rattler->setFocus();

	connect(rattler, SIGNAL(setPoints(int)), this, SLOT(scoreChanged(int)));
	connect(rattler, SIGNAL(setTrys(int)), this, SLOT(setTrys(int)));
	connect(rattler, SIGNAL(rewind()), view->progress, SLOT(rewind()));
	connect(rattler, SIGNAL(advance()), view->progress, SLOT(advance()));
	connect(view->progress, SIGNAL(restart()), rattler, SLOT(restartTimer()));

	connect(rattler, SIGNAL(togglePaused()), this, SLOT(togglePaused()));
	connect(rattler, SIGNAL(setScore(int)), this, SLOT(gameEnd(int)));

	setCentralWidget(view);

	createMenu();
	setupGUI(KMainWindow::Save | StatusBar | Create );
}

Game::~Game()
{
	delete levels;
}

void Game::scoreChanged(int score){
	statusBar()->changeItem(i18n("Score: %1", score), SCORE);
}

void Game::setTrys(int tries){
	statusBar()->changeItem(i18n("Lives: %1", tries), LIVES);
}

void Game::gameEnd(int score){
	KScoreDialog di(KScoreDialog::Name | KScoreDialog::Score | KScoreDialog::Date, this);
	KScoreDialog::FieldInfo scoreInfo;
	QString date = QDateTime::currentDateTime().toString();
	scoreInfo.insert(KScoreDialog::Date, date);
	if (di.addScore(score, scoreInfo, true))
		di.exec();
}

void Game::showHighScores()
{
	KScoreDialog d(KScoreDialog::Name | KScoreDialog::Score | KScoreDialog::Date, this);
	d.exec();
}

void Game::createMenu()
{
#ifdef __GNUC__
#warning "kde4: port it"
#endif
	//actionCollection()->setAutoConnectShortcuts(false);
	QAction *action = actionCollection()->addAction("Pl1Up");
        action->setText(i18n("Move Up"));
	action->setShortcut(Qt::Key_Up);
	action = actionCollection()->addAction("Pl1Down");
        action->setText(i18n("Move Down"));
	action->setShortcut(Qt::Key_Down);
	action = actionCollection()->addAction("Pl1Right");
        action->setText(i18n("Move Right"));
	action->setShortcut(Qt::Key_Right);
	action = actionCollection()->addAction("Pl1Left");
        action->setText(i18n("Move Left"));
	action->setShortcut(Qt::Key_Left);
	action = actionCollection()->addAction("Pl1TurnRight");
        action->setText(i18n("Turn Right"));
	action->setShortcut(Qt::Key_Period);
	action = actionCollection()->addAction("Pl1TurnLeft");
        action->setText(i18n("Turn Left"));
	action->setShortcut(Qt::Key_Comma);
#ifdef __GNUC__
#warning "kde4: port it"
#endif
	//actionCollection()->setAutoConnectShortcuts(true);
	rattler->setActionCollection(actionCollection());

	action = KStandardGameAction::gameNew(rattler, SLOT(restart()), this);
        actionCollection()->addAction(action->objectName(), action);
	pause = KStandardGameAction::pause(rattler, SLOT(pause()), this);
        actionCollection()->addAction(pause->objectName(), pause);
	action = KStandardGameAction::highscores(this, SLOT(showHighScores()), this);
        actionCollection()->addAction(action->objectName(), action);
	action = KStandardGameAction::quit( this, SLOT(close()), this);
        actionCollection()->addAction(action->objectName(), action);

	action = KStandardAction::preferences(this, SLOT(showSettings()), this);
        actionCollection()->addAction(action->objectName(), action);

	// TODO change and make custom function that pauses game or
	// modify widget to pause when loosing focus and remove this
	action = KStandardAction::keyBindings(guiFactory(), SLOT(configureShortcuts()), this);
        actionCollection()->addAction(action->objectName(), action);
}

void Game::togglePaused()
{
	static bool checked = false;
	checked = !checked;
	pause->setEnabled(checked);
}

/**
 * Show Settings dialog.
 */
void Game::showSettings(){
  if(KConfigDialog::showDialog("settings"))
    return;

  KConfigDialog *dialog = new KConfigDialog(this, "settings", Settings::self());
  dialog->addPage(new General(0), i18n("General"), "package_settings");

  Appearance *a = new Appearance(0);


  QStringList list;

    if (rattler->backgroundPixmaps.count() == 0) {
	list.append(i18n("none"));
    } else {
        QStringList::ConstIterator it = rattler->backgroundPixmaps.begin();
        for(unsigned i = 0; it != rattler->backgroundPixmaps.end(); i++, it++) {
	    // since the filename may contain underscore, they
	    // are replaced with spaces in the menu entry
            QString s = QFileInfo( *it ).baseName();
	    s = s.replace(QRegExp("_"), " ");
	    list.append(s);
	}
    }

	a->kcfg_bgimage->addItems(list);

	dialog->addPage(a, i18n("Appearance"), "style");

	dialog->addPage(new StartRoom(), i18n("First Level"), "folder_home");
	connect(dialog, SIGNAL(settingsChanged(const QString&)), rattler, SLOT(loadSettings()));
	dialog->show();
}

#include "game.moc"
