#include "misc_ui.h"
#include "misc_ui.moc"

#include <QPaintEvent>
#include <QAbstractEventDispatcher>

#include "base/piece.h"
#include "base/baseboard.h"
#include "baseprefs.h"
#include "common_factory.h"

const uint GIFT_SHOWER_TIMEOUT = 800;
const uint GIFT_POOL_TIMEOUT = 2000;

const uint SHADOW_HEIGHT = 10;

const uint GI_WIDTH = 15;
const uint GI_HEIGHT = 11;
const uint ARROW_HEIGHT = 3;
const uint ARROW_WIDTH = 7;

const uint LED_WIDTH = 15;
const uint LED_HEIGHT = 15;
const uint LED_SPACING = 5;

/*****************************************************************************/
ShowNextPiece::ShowNextPiece(BaseBoard *board, QWidget *parent)
    : QGraphicsView(parent)
{
    setScene(board->next());
    setFrameStyle(QFrame::Panel | QFrame::Sunken);
    zoomChanged();
}

void ShowNextPiece::zoomChanged()
{
    setFixedSize( sizeHint() );
}

QSize ShowNextPiece::sizeHint() const
{
    if ( scene()==0 ) return QSize();
    return QSize((int)scene()->width(), (int)scene()->height()) + 2 * QSize(frameWidth(), frameWidth());
}

/*****************************************************************************/
Shadow::Shadow(BaseBoard *board, QWidget *parent)
    : QWidget(parent), _xOffset(board->frameWidth()),
     _board(board), _show(false)
{
    setObjectName("shadow");
    connect(board, SIGNAL(updatePieceConfigSignal()), SLOT(update()));
}

QSize Shadow::sizeHint() const
{
	return QSize(_xOffset + _board->blocksMatrix().width() * BasePrefs::blockSize(),
				 SHADOW_HEIGHT);
}

QSizePolicy Shadow::sizePolicy() const
{
	return QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
}

void Shadow::zoomChanged()
{
    adjustSize();
}

void Shadow::setDisplay(bool show)
{
    _show = show;
    update();
}

void Shadow::paintEvent(QPaintEvent *)
{
	if ( !_show ) return;

	const Piece *piece = _board->currentPiece();
	uint pf = piece->min().first + _board->currentPos().first;
	uint pl = pf + piece->size().first - 1;

    QPainter p(this);
    p.setBrush(Qt::black);
    p.setPen(Qt::black);
	for (uint i=pf; i<=pl; i++) {
		QRect r(_xOffset + i * BasePrefs::blockSize() + 1 , 0,
				BasePrefs::blockSize() - 2, SHADOW_HEIGHT);
		p.drawRect(r);
	}
}


/*****************************************************************************/
class Led : public QWidget
{
 public:
	Led(const QColor &c, QWidget *parent)
		: QWidget(parent), col(c), _on(false) {}

	QSizePolicy sizePolicy() const
		{ return QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed); }
	QSize sizeHint() const { return QSize(LED_WIDTH, LED_HEIGHT); }

	void on()  { if (!_on) { _on = true; repaint(); } }
	void off() { if (_on)  {_on = false; repaint(); } }
	void setColor(const QColor &c) { if (c!=col) { col = c; repaint(); } }

 protected:
	void paintEvent(QPaintEvent *) {
		QPainter p(this);
		p.setBrush((_on ? col.light() : col.dark()));
		p.setPen(Qt::black);
		p.drawEllipse(0, 0, width(), height());
	}

 private:
	QColor col;
	bool   _on;
};

GiftPool::GiftPool(QWidget *parent)
    : KHBox(parent), nb(0), ready(false)
{
	setObjectName("gift_pool");
	setSpacing(LED_SPACING);
	leds.resize(cfactory->cbi.nbGiftLeds);
	for (int i=0; i<leds.size(); i++)
            leds[i] = new Led(Qt::yellow, this);
}

QSize GiftPool::sizeHint() const
{
	QSize s = (leds.size() ? leds[0]->sizeHint() : QSize());
	return QSize((s.width()+LED_SPACING)*leds.size()-LED_SPACING, s.height());
}

QSizePolicy GiftPool::sizePolicy() const
{
	return QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
}

void GiftPool::put(uint n)
{
	if ( n==0 ) return;
	if ( nb==0 && !ready )
        QTimer::singleShot(cfactory->cbi.giftPoolTimeout,
                           this, SLOT(timeout()));
	uint e = qMin((int)(nb+n), leds.size());
	for (uint i=nb; i<e; i++) leds[i]->on();
	uint f = qMin((int)(nb+n-e), leds.size());
	for (uint i=0; i<f; i++) leds[i]->setColor(Qt::red);
	nb += n;
}

uint GiftPool::take()
{
	Q_ASSERT(ready);
	for (int i=0; i<leds.size(); i++) {
		leds[i]->setColor(Qt::yellow);
		leds[i]->off();
	}
    uint max = cfactory->cbi.maxGiftsToSend;
	if ( nb>max ) {
		uint p = nb - max;
		nb = 0;
		put(p);
		return max;
	} else {
		ready = false;
		uint t = nb;
		nb = 0;
		return t;
	}
}

void GiftPool::reset()
{
	QAbstractEventDispatcher::instance()->unregisterTimers(this);
    ready = false;
    nb = 0;
    for (int i=0; i<leds.size(); i++) {
		leds[i]->setColor(Qt::yellow);
		leds[i]->off();
	}
}

//-----------------------------------------------------------------------------
PlayerProgress::PlayerProgress(BaseBoard *board, QWidget *parent)
  : KGameProgress(Qt::Vertical, parent), _board(board)
{
  setMinimum(0);
  setMaximum(board->blocksMatrix().height());
  setValue(0);
  QPalette palette;
  palette.setColor( backgroundRole(), Qt::lightGray );
  setPalette( palette );
  setTextEnabled(false);
  setBarColor(Qt::blue);
}

QSize PlayerProgress::sizeHint() const
{
  return QSize(10, _board->blocksMatrix().height() * BasePrefs::blockSize())
    + 2 * QSize(frameWidth(), frameWidth());
}

QSizePolicy PlayerProgress::sizePolicy() const
{
  return QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
}

void PlayerProgress::zoomChanged()
{
    adjustSize();
}
