#include "types.h"

#include <klocale.h>
#include "version.h"

cId::cId(const QString &_gameName, const QString &_gameId)
: libId(MULTI_ID), gameName(_gameName), gameId(_gameId)
{}

void cId::check(const cId &id)
{
	if ( libId!=id.libId )            state = LibIdClash;
	else if ( gameName!=id.gameName ) state = GameNameClash;
	else if ( gameId!=id.gameId )     state = GameIdClash;
	else                              state = Accepted;
}

QString cId::errorMessage(const cId &id) const
{
	KLocalizedString str = ki18n("\nServer: \"%1\"\nClient: \"%2\"");

	switch (state) {
	 case Accepted: return QString::null;
	 case LibIdClash:
		return i18n("The MultiPlayer library of the server is incompatible")
			+ str.subs(libId).subs(id.libId).toString();
	 case GameNameClash:
		return i18n("Trying to connect a server for another game type")
			+ str.subs(gameName).subs(id.gameName).toString();
	 case GameIdClash:
		return i18n("The server game version is incompatible")
			+ str.subs(gameId).subs(id.gameId).toString();
	}
	Q_ASSERT(0);
	return QString::null;
}

QDataStream &operator << (QDataStream &s, const cId &id)
{
	s << id.libId << id.gameName << id.gameId << (quint8)id.state;
	return s;
}

QDataStream &operator >> (QDataStream &s, cId &id)
{
	quint8 state;
	s >> id.libId >> id.gameName >> id.gameId >> state;
	id.state = (cId::State)state;
	return s;
}

//-----------------------------------------------------------------------------
QDataStream &operator << (QDataStream &s, const MeetingMsgFlag &f)
{
	s << (quint8)f;
	return s;
}

QDataStream &operator >> (QDataStream &s, MeetingMsgFlag &f)
{
	quint8 i;
	s >> i; f = (MeetingMsgFlag)i;
	return s;
}

//-----------------------------------------------------------------------------
QDataStream &operator << (QDataStream &s, const TextInfo &ti)
{
	s << (quint32)ti.i << ti.text;
	return s;
}

QDataStream &operator >> (QDataStream &s, TextInfo &ti)
{
	quint32 i;
	s >> i >> ti.text; ti.i = i;
	return s;
}

//-----------------------------------------------------------------------------
QDataStream &operator << (QDataStream &s, const MeetingCheckBox::Type &t)
{
	s << (quint8)t;
	return s;
}

QDataStream &operator >> (QDataStream &s, MeetingCheckBox::Type &t)
{
	quint8 i;
	s >> i; t = (MeetingCheckBox::Type)i;
	return s;
}

//-----------------------------------------------------------------------------
QDataStream &operator << (QDataStream &s, const TypeInfo &ti)
{
	s << (quint32)ti.i << ti.type;
	return s;
}

QDataStream &operator >> (QDataStream &s, TypeInfo &ti)
{
	quint32 i;
	s >> i >> ti.type; ti.i = i;
	return s;
}

//-----------------------------------------------------------------------------
QDataStream &operator << (QDataStream &s, const BoardData &bd)
{
	s << (quint8)bd.type << bd.name;
	return s;
}

QDataStream &operator >> (QDataStream &s, BoardData &bd)
{
    quint8 i;
	s >> i >> bd.name;
    bd.type = (PlayerComboBox::Type)i;
	return s;
}

//-----------------------------------------------------------------------------
QDataStream &operator << (QDataStream &s, const ExtData &ed)
{
	s << ed.bds << ed.text << ed.type;
	return s;
}

QDataStream &operator >> (QDataStream &s, ExtData &ed)
{
	s >> ed.bds >> ed.text >> ed.type;
	return s;
}

//-----------------------------------------------------------------------------
QDataStream &operator << (QDataStream &s, const MeetingLineData &pld)
{
	s << pld.ed << (quint8)pld.own;
	return s;
}

QDataStream &operator >> (QDataStream &s, MeetingLineData &pld)
{
	quint8 b;
	s >> pld.ed >> b; pld.own = b;
	return s;
}

//-----------------------------------------------------------------------------
QDataStream &operator << (QDataStream &s, const MetaFlag &f)
{
	s << (quint8)f;
	return s;
}

QDataStream &operator >> (QDataStream &s, MetaFlag &f)
{
	quint8 i;
	s >> i; f = (MetaFlag)i;
	return s;
}


//-----------------------------------------------------------------------------
Stream::Stream(QIODevice::OpenModeFlag _mode)
: mode(_mode)
{
	setDevice(&buf);
	Q_ASSERT( _mode==QIODevice::ReadOnly || _mode==QIODevice::WriteOnly );
	buf.open(_mode);
}

void Stream::clear()
{
	buf.close();
	buf.open(mode | QIODevice::Truncate);
}

void Stream::setArray(QByteArray &a)
{
	buf.close();
	buf.setBuffer(&a);
	buf.open(mode);
}

bool ReadingStream::readOk()
{
	return buf.isOpen();
}

void ReadingStream::clearRead()
{
	int i = buf.pos();
	if ( i==0 ) return;
	buf.close();
	QByteArray a;
	a.duplicate(buffer().data() + i, size() - i);
	buf.setBuffer(&a);
	buf.open(QIODevice::ReadOnly);
}

//-----------------------------------------------------------------------------
void IOBuffer::writingToReading()
{
	// this should do the trick :)
	reading.setArray(writing.buffer());
#ifdef __GNUC__
#warning this is very suspicious, check what to do!
#endif
/*   aacid - with that it crashes
	QByteArray a;
	writing.setArray(a);*/
}

//-----------------------------------------------------------------------------
void BufferArray::clear(uint k)
{
	for (int i=k; i<a.size(); i++) delete a[i];
}

BufferArray::~BufferArray()
{
	clear(0);
}

void BufferArray::resize(uint nb)
{
	uint s = a.size();
	if ( nb<s ) clear(nb);
	a.resize(nb);
	for (uint i=s; i<nb; i++) a[i] = new IOBuffer;
}

QDataStream &operator <<(QDataStream &s, const BufferArray &b)
{
	for (uint i=0; i<b.size(); i++) {
		s.writeBytes(b[i]->writing.buffer().data(), b[i]->writing.size());
//		debug("BUFFERARRAY : << (i=%i size=%i)", i, b[i]->writing.size());
		b[i]->writing.clear();
	}
	return s;
}

QDataStream &operator >>(QDataStream &s, BufferArray &b)
{
	uint size;
	char *c = 0;
	for (uint i=0; i<b.size(); i++) {
		s.readBytes(c, size);
		QByteArray a(c, size);
                delete [] c; // deep copy in QBA
		b[i]->reading.setArray(a);
//		debug("BUFFERARRAY : >> (i=%i c=%i size=%i s=%i)",
//			  i, (int)c, size, b[i]->reading.size());
	}
	return s;
}
