/*
  Copyright (C) 2006 Daren Sawkey
  daren@sawkey.net

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  $Id: kbgcompoffline.cpp 2006-11-30 08:16:54Z darens $

*/

#include "kbgofflinecalc.h"
#include "kbgoffline.h"

#include <QLayout>
#include <QGroupBox>
#include <QCheckBox>
#include <QTimer>
#include <QSpinBox>
#include <kvbox.h>

#include <QLineEdit>
#include <QTabWidget>
#include <QLabel>
#include <QGridLayout>
#include <QMenu>
#include <ktoggleaction.h>
#include <kmessagebox.h>
#include <kinputdialog.h>
#include <kstandardaction.h>
#include <kconfig.h>
#include <klocale.h>
#include <kxmlguiwindow.h>
#include <kaction.h>
#include <krandomsequence.h>
#include <kglobal.h>
#include <kicon.h>
#include <vector>
#include <cmath>

#include "version.h"

using namespace std;

KBgEngineOfflineCalc::KBgEngineOfflineCalc(KXmlGuiWindow *parent, QMenu *pmenu)
{
	Q_UNUSED(parent);
	Q_UNUSED(pmenu);
	//constructor
}

KBgEngineOfflineCalc::~KBgEngineOfflineCalc()
{
	//destructor
}

int KBgEngineOfflineCalc::convertGame( KBgStatus &game, int *points, std::vector<int> &dice )
{
	/*
	 * given a KBg game, extract the information needed for the calculation of computer move
	 */

	//convert player
	int p = game.turn();

	//convert points
	if ( game.direction() == -1  ) {
		for ( int i = 1; i < 25; ++i ) {
			points[ i ] = -game.board( i );
		}
	} else {
		for ( int i = 1; i < 25; ++i ) {
			points[ i ] = -game.board( 25 - i );
		}
	}

	points[  0 ] = -game.home(   US );
	points[ 25 ] = -game.home( THEM );
	points[ 31 ] = -game.bar (   US );
	points[ 32 ] = -game.bar ( THEM );

	//convert dice
	dice.push_back( game.dice( p, 0 ) );
	dice.push_back( game.dice( p, 1 ) );
	if ( dice.at(0) == dice.at(1) ) {
		dice.push_back( dice.at( 0 ) );
		dice.push_back( dice.at( 0 ) );
	}
	return p;
}


QString KBgEngineOfflineCalc::getComputerMove( KBgStatus &game )
{
	/*
	 * a wrapper function; calls the function computeMove to determine the best computer move
	 * then determines the move string to pass back to the game
	 */
	int points[33];
	vector<int> dice;
	vector<moveScore> movesList;
	vector<Move> list;
	int p;
	g = &game;

	p = convertGame( game, points, dice );
	// number of moves possible
	m_moves = game.moves();

	computeMove( points, dice, movesList, p );

	if ( movesList.size() > 0 ) {
		list = (movesList.at(0)).moves;
	}
	vector<Move>::iterator it;
	QString s;

	s.append( QString::number( list.size() ) + ' ');
	for ( it = list.begin(); it != list.end(); ++it ) {
		int sp = (*it).startPoint;
		int dv = (*it).dieValue;

		if ( game.direction() == 1 && sp < 25 )
			sp = 25 - sp;
		sp > 30 ? s.append( "bar" ) : s.append( QString::number( sp ) );

		(*it).kicked == true ? s.append( "+" ) : s.append( "-" );

		if ( (sp-dv < 1 || sp-dv > 24) && sp > 0 && sp < 25 )
			s.append( "off" );
		else if ( sp > 30 )
			s.append( QString::number( 25 * p  - dv ) );
		else
			s.append( QString::number( sp-dv ) );
		s.append( " " );
	}
	return s;
}

int KBgEngineOfflineCalc::getDouble( KBgStatus &game )
{
	/*
	 * Evaluate whether the computer should accept a double
 	 * TODO computer should ask for doubles of own initiative, too
	 */

	int points[33];
	vector<int> dice;
	vector<moveScore> movesList;
	int p;
	float us, them;

	p = convertGame( game, points, dice );
	us   = pointsScore( points,  p );
	them = pointsScore( points, !p );
	if ( us == 0 )
		us += 0.001;
	if ( ( us - them ) / fabs(us) > -.25 )
		return 1;
	else return 0;
}

int KBgEngineOfflineCalc::endp( int sp, int dv )
{
	/*
	 *  given startpoint sp and signed die value dv, return the endpoint
	 * takes into account bar, home
	 */
	int ep;
	int p;
	dv > 0 ? p = 1 : p = 0;
	sp > 30 ? ep = 25 * ( 1-p ) + dv : ep = sp + dv;
	if ( ep > 25 )
		ep = 25;
	if ( ep < 0 )
		ep = 0;
	return ep;
}

void KBgEngineOfflineCalc::computeMove( int *points, vector<int> &dice, vector<moveScore> &movesList, int p )
{
	/*
	 * given a checker configuration points[33], the dice, and which player it is, we calculate the highest valued move
	 * calculation is bylooping through all possible moves
	 * TODO if doubles are rolled, order is not important--rewrite to save computing
	 */
	Move move_temp;
	moveScore moveScore_temp;

	vector<moveScore>::iterator listIt;
	vector<int>::iterator itD1, itD2, itD3, itD4; // iterators over dice
	vector<Move> move, tempMove1, tempMove2, tempMove3, tempMove4, tempMove;
	vector<int> tempDice1, tempDice2, tempDice3, tempDice4;
	int tempPoints1[33], tempPoints2[33], tempPoints3[33], tempPoints4[33];

	int ps = 2 * p - 1;

	bool movePoss2 = false;
	bool movePoss3 = false;
	bool movePoss4 = false;

	int ep;
	int front, back;
	p == 0 ? front = 1  : front = 24;
	p == 0 ? back  = 24 : back  =  1;

	if ( dice.at(0) != dice.at(1) ) {
		for ( itD1 = dice.begin(); itD1 != dice.end(); ++itD1 ) {
			// take into account that player 0 moves from 24 to 1 whereas player 0 moves from 1 to 24
			for ( int i = back; i > 0 && i < 25; i += ps ) {
				if ( points[ 31 + p ] * ps > 0 ) // if there is a checker on bar we must move it
					i = 31 + p;
				tempMove1.clear();
				if ( points[ i ] * ps > 0 ) {
					move_temp.dieValue = *itD1 * ps;
					move_temp.startPoint = i;
					ep = endp( i, *itD1 * ps );
					if ( isValidSingleMove( points, move_temp, p ) ) {
						updateTemps( points, tempPoints1, dice, tempDice1, i, itD1, p );
						points[ ep ] * ps == -1 ? move_temp.kicked = true : move_temp.kicked = false;
						tempMove1.push_back( move_temp );
						// TODO itD2 should be replaced by an int, but it is passed in updateTemps
						itD2 = tempDice1.begin();
						// save computation by starting from where previous moved started
						for ( int ii = ( i < 26 ? i : back ); ii > 0 && ii < 25; ii += ps ) {
							if ( tempPoints1[ 31 + p ] * ps > 0 ) // if there is a checker on bar we must move it
								ii = 31 + p;
							if ( tempPoints1[ ii ] * ps > 0 ) {
								move_temp.dieValue = *itD2 * ps;
								move_temp.startPoint = ii;
								ep = endp( ii, *itD2 * ps );
								tempMove2.clear();
								tempMove2.assign( tempMove1.begin(), tempMove1.end() );
								if ( isValidSingleMove( tempPoints1, move_temp, p ) ) {
									movePoss2 = true;
									tempPoints1[ ep ] * ps == -1 ? move_temp.kicked = true : move_temp.kicked = false;
									tempMove2.push_back( move_temp );
									updateTemps( tempPoints1, tempPoints2, tempDice1, tempDice2, ii, itD2, p );
									if ( isValidTotalMove( points, tempMove2, dice, p ) )
										pushMove( tempPoints2, tempMove2, movesList, p );
								} //end of if ( isValidSingleMove( tempPoints1, move_temp ) ) {
							}
						}
						if ( m_moves < 2 ) {// have made it through second die without finding a valid move
							if ( isValidTotalMove( points, tempMove1, dice, p ) )
								pushMove( tempPoints1, tempMove1, movesList, p );
						}
					}
				}
			}
		}

	}//have looped over all possible moves

	else { // dice.at(0) == dice.at(1)
		itD1 = dice.begin();
		for ( int i = back; i > 0 && i < 25; i += ps ) {
			if ( points[ 31 + p ] * ps > 0 ) // if there is a checker on bar we must move it
				i = 31 + p;
			tempMove1.clear();
			if ( points[ i ] * ps > 0 ) {
				move_temp.dieValue = *itD1 * ps;
				move_temp.startPoint = i;
				ep = endp( i, *itD1 * ps );
				if ( isValidSingleMove( points, move_temp, p ) ) {
					updateTemps( points, tempPoints1, dice, tempDice1, i, itD1, p );
					points[ ep ] * ps == -1 ? move_temp.kicked = true : move_temp.kicked = false;
					tempMove1.push_back( move_temp );
					itD2 = tempDice1.begin();
					for ( int ii = ( i < 26 ? i : back ); ii > 0 && ii < 25; ii += ps ) {
						if ( tempPoints1[ 31 + p ] * ps > 0 ) // if there is a checker on bar we must move it
							ii = 31 + p;
						if ( tempPoints1[ ii ] * ps > 0 ) {
							move_temp.dieValue = *itD2 * ps;
							move_temp.startPoint = ii;
							ep = endp( ii, *itD2 * ps );
							tempMove2.clear();
							tempMove2.assign( tempMove1.begin(), tempMove1.end() );
							if ( isValidSingleMove( tempPoints1, move_temp, p ) ) {
								movePoss2 = true;
								tempPoints1[ ep ] * ps == -1 ? move_temp.kicked = true : move_temp.kicked = false;
								tempMove2.push_back( move_temp );
								updateTemps( tempPoints1, tempPoints2, tempDice1, tempDice2, ii, itD2, p );
								itD3 = tempDice2.begin();
								for ( int iii = ( ii < 26 ? ii : back ); iii > 0 && iii < 25; iii += ps ) {
									if ( tempPoints2[ 31 + p ] * ps > 0 ) // if there is a checker on bar we must move it
									iii = 31 + p;
									tempMove3.clear();
									tempMove3.assign( tempMove2.begin(), tempMove2.end() );
									if ( tempPoints2[ iii ] * ps > 0 ) {
										move_temp.dieValue = *itD3 * ps;
										move_temp.startPoint = iii;
										ep = endp( iii, *itD3 * ps );
										if ( isValidSingleMove( tempPoints2, move_temp, p ) ) {
											movePoss3 = true;
											tempPoints2[ ep ] * ps == -1 ? move_temp.kicked = true : move_temp.kicked = false;
											tempMove3.push_back( move_temp );
											updateTemps( tempPoints2, tempPoints3, tempDice2, tempDice3, iii, itD3, p );
							// no iteration over dice as there's only one left
											for ( int iiii = ( iii < 26 ? iii : back ); iiii > 0 && iiii < 25; iiii += ps ) {
												itD4 = tempDice3.begin();
												if ( tempPoints3[ 31 + p ] * ps > 0 ) // if there is a checker on bar we must move it
												iiii = 31 + p;
												tempMove4.clear();
												tempMove4.assign( tempMove3.begin(), tempMove3.end() );
												if ( tempPoints3[ iiii ] * ps > 0 ) {
													move_temp.dieValue = *itD4 * ps;
													move_temp.startPoint = iiii;
													ep = endp( iiii, *itD4 * ps );
													if ( isValidSingleMove( tempPoints3, move_temp, p ) ) {
														movePoss4 = true;
														tempPoints3[ ep ] * ps == -1 ? move_temp.kicked = true : move_temp.kicked = false;
														tempMove4.push_back( move_temp );
														updateTemps( tempPoints3, tempPoints4, tempDice3, tempDice4, iiii, itD4, p );
														if ( isValidTotalMove( points, tempMove4, dice, p ) )
															pushMove( tempPoints4, tempMove4, movesList, p );
													}
												}
											}// end of iiii loop
											if ( m_moves < 4 ) {// have made it through third die without finding a valid move
											if ( isValidTotalMove( points, tempMove3, dice, p ) )
												pushMove( tempPoints3, tempMove3, movesList, p );
											}//end of if ( m_moves < 4 )
										}
									}
								} //end of iii loop
								if ( m_moves < 3 ) {// have made it through third die without finding a valid move
									if ( isValidTotalMove( points, tempMove2, dice, p ) )
										pushMove( tempPoints2, tempMove2, movesList, p );
								} // end of ( m_moves<3 )
								else { //doubles not rolled ( dice.size() !> 2 )
									if ( isValidTotalMove( points, tempMove2, dice, p ) ) {
									pushMove( tempPoints2, tempMove2, movesList, p );
									}
								}
							} //end of if ( isValidSingleMove( tempPoints1, move_temp ) ) {
						}
					}
					if ( m_moves < 2 ) {// have made it through second die without finding a valid move
						if ( isValidTotalMove( points, tempMove1, dice, p ) )
							pushMove( tempPoints1, tempMove1, movesList, p );
					}
				}
			}
		}
	}//have looped over all possible moves

	return;
}


int KBgEngineOfflineCalc::updateTemps( int *oldPoints, int *newPoints, vector<int> &oldDice, vector<int> &newDice, int startpoint, vector<int>::iterator &die, int p )
{
	/*
	 * a helper function for getMaxMovePossible: update the temporary storage
	 * return value is the number of points moved
	 *after a virtual move is made, call this function to get the new dice, points, checkers, and move
	 */
	int endpoint;
	int delta = 0;
	int ps = 2 * p - 1;
	vector<int>::iterator it;

	newDice.clear();
	for ( it = oldDice.begin(); it != oldDice.end(); ++it ) {
		if ( it != die )
			newDice.push_back( *it );
	}

	for ( int i = 0; i < 33; ++i ) {// reset the temporary points, for evaluating next move
		newPoints[ i ] = oldPoints[ i ];
	}

	newPoints[ startpoint ] -= ps;

	if ( startpoint > 30 ) //{ // coming from bar
		p == 1 ? startpoint = 0 : startpoint = 25;
	endpoint = endp ( startpoint, *die * ps );
	delta = ( endpoint - startpoint ) * ps;

	if ( oldPoints[ endpoint ] == -ps ) { // opponent had 1 checker on point
		newPoints[ 32 - p ] -= ps;
		newPoints[ endpoint ] = 0; // will be incremented below
	}

	newPoints[ endpoint ] += ps;
	return delta;
}

void KBgEngineOfflineCalc::pushMove( int *newPoints, vector<Move> &move, vector<moveScore> &list, int p )
{
	/*
	 *
	 * helper function for computeMove.  Add a move to the list of moves
	 * List is ranked according to score
	 */
	float score;
	vector<moveScore>::iterator listIt;
	moveScore moveScore_temp;
	score = pointsScore( newPoints, p );
	moveScore_temp.score = score;
	moveScore_temp.moves = move;
	if ( list.size() == 0 ) {
		list.push_back( moveScore_temp );
		return;
	} else {
		for ( listIt = list.begin(); listIt != list.end(); ++listIt ) {
			if ( score > (*listIt).score ) {
				list.insert( listIt, moveScore_temp );
				if ( list.size() > 10 )
					list.pop_back();
				return;
			}
		}
	}
	// if we've made it here, this is the worst move. add to the back
	list.push_back( moveScore_temp );
	if ( list.size() > 10 ) //keep the size small
		list.pop_back();
	return;
}

bool KBgEngineOfflineCalc::isValidSingleMove( int *points, Move move, int p )
{
	/*
	 * return true if the single move, from startpoint a distance dieval, is allowed
	 * sign of dieval determines orientation
	 * note: the player with points > 0 has dieval > 0
	 */

	bool allowed = false;
	bool tempAllowed = true;

	int startpoint, endpoint;
	int ps = 2 * p - 1;

	if ( move.startPoint == 0 || move.startPoint == 25 )//can't move a checker from home
		return false;

	if ( points[ 31 + p ] > 0 ) {// point on bar, must move it
		if ( move.startPoint < 26 )
			return false;
	}

	move.startPoint > 30 ? startpoint = ( 25 * ( 1 - p ) ) : startpoint = move.startPoint;
	endpoint = endp( startpoint, move.dieValue );
	if ( ( ps * points[ endpoint ] ) < -1 ) // opponent has > 1 checker on point
		return false;

	int delta = ( endpoint - startpoint ) * ps;

	int sumBearOff = 0;
	if ( p == 0 ) { // if all the checkers are in the home court, or home, we are allowed to bear off
		for ( int i = 0; i < 7; ++i ) {
			if ( points[ i ] * ps > 0 )
				sumBearOff += points[ i ] * ps;//FIXME combine into 1
		}
	}
	if ( p == 1 ) {
		for ( int i = 19; i < 26; ++i ) {
			if ( points[ i ] * ps > 0 )//FIXME only check this if we try to bear off
				sumBearOff += points[ i ] * ps;
		}
	}
	if ( ( sumBearOff < 15 ) && ( ( endpoint == 0 ) || ( endpoint == 25 ) ) )//trying to bear off when not allowed
		return false;

	if ( ( sumBearOff == 15 ) && ( endpoint == ( 25 * p ) ) ) {// bearing off

		if ( delta  > move.dieValue * ps )
			allowed = false;

		else if ( delta == move.dieValue * ps )  //FIXME these elses might not all be necessary
			allowed = true;
		 	// say you have checkers on 3 and 5, and roll a 4. you must move 5->1.
		else if ( delta < move.dieValue * ps ) {  //maxDieValue > move > minDieValue
			for ( int i = startpoint + 1; i < 7; ++i ) {
				if ( points[ 25 * p - i * ps ] * ps > 0 ) {
					tempAllowed = false;
					break;
				}
			}
			if ( tempAllowed )
				allowed = true;
		}
		if ( allowed )
			return true;
		else
			return false;
	}

	if ( ( startpoint == ( 25 * (1 - p ) ) ) && ( endpoint == ( 25 * (1 - p) + move.dieValue ) ) ) // checker from p i's bar
		return true;

	if ( ( startpoint > 0 ) && ( startpoint < 25 ) && ( endpoint > 0 ) && ( endpoint < 25 ) ) // nothing left to check for ordinary moves
		return true;

	return false;  //
}

bool KBgEngineOfflineCalc::isValidTotalMove( int *points, vector<Move> &moves, vector<int> &dice, int p )
{	/*
	 * Determine if the total move is valid
	 * both moves may be individually valid, but not the combination.
	 * FIXME player not needed because moves.dieValue is signed
	 * Calculation is:
	 * 1. if sum of moves = sum of dice, ok
	 * 2. if one move made, one move possible, and move equals larger die possible, ok
	 * 3. if doubles rolled, 2 or 3 moves made, equal to number possible, and not bearing off, ok
	 * 4. if bearing off, and sum of moves < sum of dice: check to see if any valid greater moves
	 */

	int delta, ep, sp;
	int sumMoves = 0;
	int sumDice = 0;
	vector<int>::iterator it;
	vector<Move>::iterator mIt;
	int minDie, maxDie;

	if ( moves.size() != m_moves )
		return false;
	int ps = 2*p-1;
	// 1. if sum of moves = sum of dice, ok
	for ( it = dice.begin(); it != dice.end(); ++it )
		sumDice += *it;
	for ( mIt = moves.begin(); mIt != moves.end(); ++mIt ) {
		(*mIt).startPoint > 25 ? sp = ( 25 * ( 1 - p ) ) : sp = (*mIt).startPoint;
		ep = endp( sp, (*mIt).dieValue );
		delta = ep - sp;
		sumMoves += delta;
	}
	if ( abs(sumMoves) == sumDice )
		return true;
	// 2. if one move made, one move possible, and move equals larger die possible, ok
	if ( m_moves == 1 && moves.size() == 1 ) {
		moves.at(0).startPoint > 25 ? sp = ( 25 * ( 1 - p ) ) : sp = moves.at(0).startPoint;
		ep = endp( sp, moves.at(0).dieValue );
		delta = abs(ep - sp);
		if ( dice.at(0) >= dice.at(1) ) {
			minDie = dice.at(1);
			maxDie = dice.at(0);
		} else {
			minDie = dice.at(0);
			maxDie = dice.at(1);
		}
		if ( delta == maxDie )
			return true;
		Move m;
		for ( int i = 1; i < 25; ++i ) {
			if ( points[ 31 + p ] * ps > 0 ) // if there is a checker on bar we must move it
				i = 31 + p;
			if ( points[ i ] * ps > 0 ) {
				m.dieValue = maxDie * ps;
				m.startPoint = i;
				if ( isValidSingleMove( points, m, p ) ) {
					ep = endp( m.startPoint, m.dieValue );
					if ( ep != 0 && ep != 25 ) // can't rule out bearing off at this point
						return false; // asking to use only the smaller die, when larger one is possible
				}
			}
		}
	}
	// 3. if doubles rolled, 2 or 3 moves made, equal to number possible, and not bearing off, ok
	bool poss = true;
	if ( dice.at(0) == dice.at(1) ) {
		for ( mIt = moves.begin(); mIt != moves.end(); ++mIt ) {
			(*mIt).startPoint > 25 ? sp = ( 25 * ( 1 - p ) ) : sp = (*mIt).startPoint;
			ep = sp + (*mIt).dieValue;
			if ( ep > 25 || ep < 0 ) //bearing off
				poss = false;
		}
		if ( poss == true )
			return true;
	}
	//4 if bearing off, and sum of moves < sum of dice: check to see if any valid greater moves
	bool m = false;
	int tp[33];
	for ( int i = 0; i < 33; ++i ) //make a copy of the points
		tp[i] = points[i];
	vector<int> tempDice1, tempDice2;
	vector<int>::iterator it1, it2;
	tempDice1.assign( dice.begin(), dice.end() ); // make a copy of the dice
	for ( mIt = moves.begin(); mIt != moves.end(); ++mIt ) {
		(*mIt).startPoint > 25 ? sp = ( 25 * ( 1 - p ) ) : sp = (*mIt).startPoint;
		ep = endp( sp, (*mIt).dieValue );
		delta = ep - sp;
		m = false;
		for ( it1 = tempDice1.begin(); it1 != tempDice1.end(); ++it1 ) {
			if ( abs(delta) == (*it1) && m == false ) {
				it2 = it1;
				m = true;
			}
		}
		if ( m == true ) {  // found which die was used, so take it out and remove the checker
			tempDice2.clear();
			for ( it1 = tempDice1.begin(); it1 != tempDice1.end(); ++it1 ) {
				if ( it1 != it2 ) {
					tempDice2.push_back( *it1 );
				}
			}
			tempDice1.clear();
			tempDice1.assign( tempDice2.begin(), tempDice2.end() );
			tp[ sp ] -= ps; //remove the checker
		}
		// no matching die--see if there exist any checkers further from home
		// nb. this forces order of moves to be furthest start point first
		else {
			if ( p == 0 ) {
				for ( int i = sp + 1; i < 25; ++i ) {
					if ( tp[i] < 0 )
						return false;
				}
			} else { // p == 1
				for ( int i = sp - 1; i > 0; --i ) {
					if ( tp[i] > 0 )
						return false;
				}
			}
			// move valid. remove the largest die and move the checker
			maxDie = 0;
			for ( it1 = tempDice1.begin(); it1 != tempDice1.end(); ++it1 ) {
				if ( *it1 > maxDie ) {
					maxDie = *it1;
					it2 = it1;
				}
			}
			tempDice2.clear();
			for ( it1 = tempDice1.begin(); it1 != tempDice1.end(); ++it1 ) {
				if ( it1 != it2 ) {
					tempDice2.push_back( *it1 );
				}
			}
			tempDice1.clear();
			tempDice1.assign( tempDice2.begin(), tempDice2.end() );
			tp[ sp ] -= ps;


		}
	}
	// only thing left is bearing off legally
	return true;
}

float KBgEngineOfflineCalc::pointsScore( int *points, int p )
{
	/*
	 * given a checker layout points (array[33]), and which player p it is, we calculate the score for that position
	 * use to calculate the best move
	 *TODO improve!
	 *TODO suggestions for bearing off
	 */

	double c0[58];

	c0[0]=9.35; c0[1]=79.8; c0[2]=5.22; c0[3]=46.6; c0[4]=168.6; c0[5]=8.7; c0[6]=411.8; c0[7]=36.6; c0[8]=30.5; c0[9]=13.2; c0[10]=22.3; c0[11]=82.6; c0[12]=0.24; c0[13]=0.012; c0[14]=0.09; c0[15]=0.005; c0[16]=1.5; c0[17]=14.4; c0[18]=57.4; c0[19]=0.002; c0[20]=355.5; c0[21]=5.1; c0[22]=83.9; c0[23]=2470; c0[24]=11.5; c0[25]=314.7; c0[26]=80.3; c0[27]=455; c0[28]=5.5; c0[29]=30.2; c0[30]=0.014; c0[31]=12.4; c0[32]=0.08; c0[33]=3185; c0[34]=20.0; c0[35]=0.0002; c0[36]=0.02; c0[37]=0.09; c0[38]=0.9; c0[39]=10.0; c0[40]=0.01; c0[41]=0.005; c0[42]=907; c0[43]=143; c0[44]=0.49; c0[45]=405; c0[46]=0.38; c0[47]=0.00519632; c0[48]=6.9; c0[49]=0.32; c0[50]=269; c0[51]=411; c0[52]=1.9; c0[53]=57.5; c0[54]=23.2; c0[55]=4.6; c0[56]=1.9; c0[57]=0.95;


	float score = 0;
	float doubleScore = 0;

	int po[25];
	int ps = 2 * p - 1;
	int q = 1-p;//p is us, q is them


	if ( p == 0 ) {
		for ( int i = 1; i < 25; ++i ) { // point label po is valid for either player
			po[i] = i;     // takes same index as player 0, home court = po[1] through [6]
		}			//  NB home and bar unchanged, use 31+player etc
	}				// do not use to calculate score, because it changes with player. use i,p,ps instead
	else {
		for ( int i = 1; i < 25; ++i ) {
			po[i] = 25 - i;
		}
	}

	//get pip counts
	int pipcount[2] = {0,0};
	for ( int i = 1; i < 25; ++i ) {
		if ( points[i] * ps > 0 )
			pipcount[p] += (25*p-i*ps) * points[ i ] * ps;
		else if ( points[i] * ps < 0 )
			pipcount[q] += (25*q+i*ps) * points[ i ] * (-ps);
	}
	pipcount[p] += points[ 31 + p ] * 25;
	pipcount[q] += points[ 32 - p ] * 25;

	// if the game is a race, move the furthest checkers
	//FIXME for use with doubling cube (deciding when to double) need some way to compare scores calculated different ways
	int sum = 0;
	bool isRace = true;
	for ( int i = 0; i < 19; ++i ) {//just need to check for 1 player!
		if ( ( points[ i ] > 0 ) && ( sum != -15 ) )
			isRace = false;
		if ( points[ i ] < 0 )
			sum += points[ i ];
	}
	if ( sum != -15 )
		isRace = false;
	if ( ( points[ 31 ] != 0 ) || ( points[ 32 ] != 0 ) )
		isRace = false;
	// isRace == true, race is on
	// move furthest checkers, bear off if possible
	if ( isRace ) {
		for ( int i = 24; i > 0; --i ) {
			if ( points[ po[ i ] ] * ps > 0 ) {
				score -= po[i] * po[i] * points[ po[ i ] ] * ps;
			}//from i*i*...
		}
		score += points[ 25 * p ] * 1000 * ps;
		return score;
	}

	//doubles are good
	for ( int i = 1; i < 25; ++i ) {
		if ( points[ po[i] ] * ps > 1 )
			score += c0[0];
	}

	//some doubles are even better
	for ( int i = 1; i < 24; ++i ) { //omit the ace point
		if ( points[ po[ i ] ] * ps > 1 )
			doubleScore += c0[i];
	}

	for ( int i = 2; i < 9; ++i ) {
		if ( points[ po[i] ] * ps > 1 )
			doubleScore *= (1+c0[24]);  //more blocked points are better
	}

	score += doubleScore;

	//however we don't want too much pileup in the home board

	for ( int i = 1; i < 13; ++i ) {
		if ( points[ po[ i ] ] * ps > 2 )
			score -= c0[24+i] * ( ( points[ po[ i ] ] * ps) - 2 );
	}

	// nor in the opponent's home board
	// FIXME numbering is out of order here!
	if ( points[ po[24] ] * ps > 2 )
		score -= c0[37] * ( ( points[ po[24] ] * ps) - 2 );
	if ( points[ po[23] ] * ps > 2 )
		score -= c0[38] * ( ( points[ po[23] ] * ps) - 2 );
	if ( points[ po[22] ] * ps > 2 )
		score -= c0[39] * ( ( points[ po[22] ] * ps) - 2 );
	if ( points[ po[21] ] * ps > 2 )
		score -= c0[40] * ( ( points[ po[21] ] * ps) - 2 );
	if ( points[ po[20] ] * ps > 2 )
		score -= c0[48] * ( ( points[ po[20] ] * ps) - 2 );
	if ( points[ po[19] ] * ps > 2 )
		score -= c0[41] * ( ( points[ po[19] ] * ps) - 2 );
	if ( points[ po[18] ] * ps > 2 )
		score -= c0[42] * ( ( points[ po[18] ] * ps) - 2 );

	//opponent's points on bar are good
	// the more of home court we have blocked off, the better
	// if we are far behind in the pip count, want to hit

	int b = 0;//count how many home court points blocked
	for ( int i = 1; i < 7; ++i ) {
		if ( points[ po[i] ] * ps > 1 )
			++b;
	}
	if ( points[ 32 - p ] != 0 ) {
		score += ( c0[43] * std::pow( (double)abs(points[ 32 - p ]), c0[44] ) ) * (std::pow( (double)(1+b), c0[52] ) ); // opponents checkers are opposite sign
		int trail = pipcount[p] - pipcount[q]; //how many points we trail by
		if ( trail > c0[53] ) {
			score += c0[54] * std::pow( (double)trail, c0[55] );
		}
	}

	//singles within striking distance are bad
	// score -= prob of being hit * f(how many blocked home court points) * f(how close to home)

	int blocked = 0;  // number of opponents home court points blocked
	for ( int i = 19; i < 25; ++i ) {
		if ( points[ po[ i ] ] * ps < -1 )
			blocked += 1;
	}

	float prob[12] = {0.306, 0.333, 0.389, 0.417, 0.417, 0.472, 0.167, 0.167, 0.139, 0.083, .056, .028};//should we take into account > 12?
	for ( int i = 1; i < 25; ++i ) {
		if ( points[ i ] * ps == 1 ) {
			for ( int j = 1; j < 13; ++j ) {
				if ( i + (j*ps) < 25 && i + (j*ps) > 0 ) {//FIXME should be a while loop
					if ( points[ i + (j*ps)] * ps < 0 )
						score -= c0[45] * ( std::pow((double)(25*q+i*ps),2.0 )  + c0[56]*std::pow((double)(25*q+i*ps),3.0))* (c0[57]*std::pow( (double)blocked, 1.0 )+ c0[47]*std::pow((double)blocked, 2.0)) * prob[j-1];
				}
			}
			if ( po[i] < 13 && points[32-p] != 0 )
				score -= c0[45] * ( std::pow((double)(25*q+i*ps),2.0 ) )+ c0[56]*std::pow((double)(25*q+i*ps),3.0)* (c0[57]*std::pow( (double)blocked, 1.0 )+ c0[47]*std::pow((double)blocked, 2.0)) * prob[po[i]-1];
		}
	}
	// incentive to move last two
	if ( points[ po[24] ] * ps < 2 )
		score += c0[50];
	if ( points[ po[24] ] * ps < 1 )
		score += c0[51];

	return score;
}

