/***************************************************************************
 *   Copyright (C) 2005-2006 by Stefan Kebekus                             *
 *   kebekus@kde.org                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.         *
 ***************************************************************************/

#include <config.h>

#include <kfiledialog.h>
#include <kmessagebox.h>
#include <kdeprint/kprintdialogpage.h>
#include <kparts/genericfactory.h>
#include <kprinter.h>
#include <kprocio.h>
#include <ktemporaryfile.h>

#include <Q3ValueList>
#include <qapplication.h>
#include <qcheckbox.h>
#include <qlayout.h>

#include "faxprintsettings.h"
#include "faxmultipage.h"
#include "kvs_debug.h"



K_EXPORT_COMPONENT_FACTORY(ligaturePlugin_TIFF, KGenericFactory<FaxMultiPage>( "ligaturePlugin_TIFF" ) )


FaxMultiPage::FaxMultiPage(QObject *parent, const QStringList&)
  : ligaturePluginGUI(parent),
    faxRenderer(this)
{
  setXMLFile("kfaxview.rc");
  
  /* It is very important that this method is called in the
     constructor. Otherwise ligaturePluginGUI does not know how to render
     files, and crashes may result. */
  setRenderer(&faxRenderer);
}


FaxMultiPage::~FaxMultiPage()
{
  ;
}


KAboutData* FaxMultiPage::createAboutData()
{
  /* You obviously want to change this to match your setup */
  KAboutData* about = new KAboutData("kfaxview", I18N_NOOP("KFaxView"), "0.1",
				     I18N_NOOP("Ligature Fax Plugin."),
				     KAboutData::License_GPL,
				     "Stefan Kebekus",
				     I18N_NOOP("This program previews fax (g3) files."));

  about->addAuthor ("Stefan Kebekus",
                    I18N_NOOP("Current Maintainer."),
                    "kebekus@kde.org",
                    "http://www.mi.uni-koeln.de/~kebekus");
  return about;
}


void FaxMultiPage::print()
{
  // Paranoid safety checks
  if (faxRenderer.isEmpty())
    return;

  // Allocate the printer structure
  KPrinter *printer = getPrinter(false);
  if (printer == 0)
    return;

  FaxPrintSettings* faxSettings = new FaxPrintSettings();
  if (faxSettings == 0)
  {
    kError(kvs::shell) << "FaxMultiPage::print(): Cannot allocate new FaxPrintSettings structure" << endl;
    delete printer;
    return;
  }
  printer->addDialogPage(faxSettings);

  // initialize the printer using the print dialog
  if (printer->setup(parentWdg, i18n("Print %1", m_file.section('/', -1))))
  {
    // Now do the printing.
    Q3ValueList<int> pageList = printer->pageList();
    if (pageList.isEmpty())
      printer->abort();
    else
    {
      QPainter painter;
      painter.begin(printer);


      const bool fullpage = printer->option(FAX_SCALE_FULLPAGE) == "true";
      const bool center_h = printer->option(FAX_CENTER_HORZ) == "true";
      const bool center_v = printer->option(FAX_CENTER_VERT) == "true";

      int currentpage = 0;
      bool first_page_printed = false;

      for (int i = 0; i < pageList.size(); i++)
      {
        // Printing usually takes a while.
        // This is to keep the GUI updated.
        qApp->processEvents();
	
        currentpage = pageList[i];

        QImage image = faxRenderer.getRawImage(currentpage);

        if (first_page_printed)
          printer->newPage();
        first_page_printed = true;

        const QSize printersize( painter.device()->width(), painter.device()->height() );
        kDebug(kvs::fax) << "Printersize = " << printersize << endl;
        // print Image in original size if possible, else scale it.

        QPoint dpi = faxRenderer.getDPI(currentpage);

        const QSize size(  // logical size of the image
          image.width()  * painter.device()->logicalDpiX() / dpi.x(),
          image.height() * painter.device()->logicalDpiY() / dpi.y()
        );

        kDebug(kvs::fax) << "Org image size = " << image.width() << "x" << image.height()
                  << " logical picture res = " << dpi.x() << "x" << dpi.y() << endl;
        kDebug(kvs::fax) << "New image size = " << size 
                  << " logical printer res = " << painter.device()->logicalDpiX() << "x" << painter.device()->logicalDpiY() << endl;

        uint top, left, bottom, right;
        if (fullpage)
          top = left = bottom = right = 0;
        else
          printer->margins( &top, &left, &bottom, &right );
        kDebug(kvs::fax) << "Margins = " << top << " " << left << " " << bottom << " " << right << endl;

        const QSize maxSize( printersize.width()-left-right, printersize.height()-top-bottom );
        QSize scaledImageSize = size;
        if (size.width() > maxSize.width() || size.height() > maxSize.height() )
        {
          // Image does not fit - scale it and print centered
          scaledImageSize.scale( maxSize, Qt::KeepAspectRatio );
          kDebug(kvs::fax) << "Image does not fit - scaling to " << maxSize << endl;
        }
        else
        {
          // Image does fit - print it in original size, but centered
          scaledImageSize.scale( size, Qt::KeepAspectRatio );
          kDebug(kvs::fax) << "Image does fit - scaling to " << size << endl;
        }
        kDebug(kvs::fax) << "Final image size " << scaledImageSize << endl;
        int x,y;
        if (center_h)
          x = (maxSize.width()-scaledImageSize.width())/2 + left;
        else
          x = left;
        if (center_v)
          y = (maxSize.height()-scaledImageSize.height())/2 + top;
        else
          y = top;

        painter.drawImage( QRect(x,y,scaledImageSize.width(), scaledImageSize.height()), image );
      }

      painter.end();
    }
    delete printer;
  }
}


#include "faxmultipage.moc"
