/***************************************************************************
 *   Copyright (C) 2005--2006 by Stefan Kebekus                            *
 *   kebekus@kde.org                                                       *
 *                                                                         *
 *   Copyright (C) 2005 by Wilfried Huss                                   *
 *   Wilfried.Huss@gmx.at                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.          *
 ***************************************************************************/

#include <kaction.h>
#include <kconfigdialog.h>
#include <kdialog.h>
#include <kfiledialog.h>
#include <kmessagebox.h>
#include <kparts/genericfactory.h>
#include <kprinter.h>
#include <ktemporaryfile.h>
#include <qtooltip.h>

#include "pageInsertionWidget.h"
#include "pageRangeWidget.h"
#include "pdfMultipage.h"
#include "pdfWidget.h"
#include "pdfPageCache.h"
#include "optionDialogPdfWidget.h"

#include "prefs.h"

K_EXPORT_COMPONENT_FACTORY( ligaturePlugin_PDF, KGenericFactory<PdfMultiPage>( "ligaturePlugin_PDF" ) )


PdfMultiPage::PdfMultiPage(QObject *parent, const QStringList&)
  : ligaturePluginGUI(parent),
    pdfRenderer(this)
{

  pdfRenderer.setObjectName("Pdf renderer");

  setXMLFile("pdfMultipage.rc");

  /* It is very important that this method is called in the
     constructor. Otherwise ligaturePluginGUI does not know how to render
     files, and crashes may result. */
  setRenderer(&pdfRenderer);

  docInfoAction = new KAction(KIcon("info"), i18n("Document &Properties"), actionCollection(), "info_pdf");
  connect(docInfoAction, SIGNAL(triggered(bool)), this, SLOT(slotShowInfo()));
  setBackgroundAction = new KAction(i18n("Set Document Background..."), actionCollection(), "set_background");
  connect(setBackgroundAction, SIGNAL(triggered(bool)), this, SLOT(slotSetBackground()));
  insertPagesAction = new KAction(i18n("Insert Pages..."), actionCollection(), "insert_pages");
  connect(insertPagesAction, SIGNAL(triggered(bool)), this, SLOT(slotInsertPages()));
  deletePagesAction = new KAction(i18n("Delete Pages..."), actionCollection(), "delete_pages");
  connect(deletePagesAction, SIGNAL(triggered(bool)), this, SLOT(slotDeletePages()));

  enableActions(false);
}


PdfMultiPage::~PdfMultiPage()
{
  Prefs::writeConfig();

  delete docInfoAction;
}


KAboutData* PdfMultiPage::createAboutData()
{
  /* You obviously want to change this to match your setup */
  KAboutData* about = new KAboutData("kpdfview", I18N_NOOP("KPDFView"), "0.3",
                          I18N_NOOP("Ligature PDF Plugin"),
                          KAboutData::License_GPL,
                          "Wilfried Huss",
                          I18N_NOOP("This program displays PDF files."));

  about->addAuthor ("Stefan Kebekus",
                    I18N_NOOP("Ligature plugin template"),
                    "kebekus@kde.org",
                    "http://www.mi.uni-koeln.de/~kebekus");

  about->addAuthor ("Wilfried Huss",
                    I18N_NOOP("PDF file loading, Inverse- and Forwardsearch"),
                    "Wilfried.Huss@gmx.at");

  return about;
}


void PdfMultiPage::setFile(bool r)
{
  enableActions(r);
}


void PdfMultiPage::enableActions(bool b)
{
  ligaturePluginGUI::enableActions(b);

  docInfoAction->setEnabled(b);
  setBackgroundAction->setEnabled(b);
  deletePagesAction->setEnabled(b);
  insertPagesAction->setEnabled(b);
}


DocumentWidget* PdfMultiPage::createDocumentWidget(PageView *parent, DocumentPageCache *cache)
{
  PdfWidget* documentWidget = new PdfWidget(parent, cache);

  connect (documentWidget, SIGNAL(inverseSearch(const PageNumber&, QPoint, bool)), this, SLOT(inverseSearch(const PageNumber&, QPoint, bool)));
  
  return documentWidget;
}


void PdfMultiPage::addConfigDialogs(KConfigDialog* configDialog)
{
  OptionDialogPdfWidget* pdfConfigWidget = new OptionDialogPdfWidget(parentWdg);

  configDialog->addPage(pdfConfigWidget, Prefs::self(), i18n("Inverse Search"), "find");
  //configDialog->setHelp("preferences", "pdf");
}


void PdfMultiPage::inverseSearch(const PageNumber& pageNumber, QPoint point, bool call)
{
  Length x;
  Length y;

  x.setLength_in_inch(point.x());
  y.setLength_in_inch(point.y());
  
  pdfRenderer.inverseSearch(pageNumber, x, y, call);
}


void PdfMultiPage::print()
{
  // Paranoid safety checks
  if (pdfRenderer.isEmpty())
    return;
  
  // Allocate the printer structure
  KPrinter *printer = getPrinter(false);
  if (printer == 0)
    return;
  
  // initialize the printer using the print dialog
  if ( printer->setup(parentWdg, i18n("Print %1", m_file.section('/', -1))) ) {
    // Now do the printing.
    QList<int> pageList = printer->pageList();
    if (pageList.isEmpty())
      printer->abort();
    else {
      KTemporaryFile tmpPSFile;
      tmpPSFile.setSuffix(".ps");
      tmpPSFile.open();

      if (pdfRenderer.convertToPSFile(tmpPSFile.fileName(), pageList ) == true)
	printer->printFiles( QStringList(tmpPSFile.fileName()), true );
      else
	printer->abort();
    }
    delete printer;
  }
}


bool PdfMultiPage::isReadWrite() const
{
  return true;
}


bool PdfMultiPage::isModified() const
{
  return pdfRenderer.isModified();
}


bool PdfMultiPage::slotSave(const QString &fileName)
{
  // Paranoid safety checks
  if (pdfRenderer.isEmpty())
    return false;
  if (fileName.isEmpty())
    return false;

  bool r = pdfRenderer.save(fileName);
  if (r == false)
    KMessageBox::error(parentWdg, i18n("<qt>Error saving the document to the file <strong>%1</strong>. The document is <strong>not</strong> saved.</qt>", fileName),
                       i18n("Error saving document"));
  return r;
}


void PdfMultiPage::slotShowInfo()
{
  // deleting pages from an empty document doesn't make much sense
  if (dataModel->numberOfPages() == 0)
    return;

  if (pdfRenderer.showInfo())
    emit documentHasBeenModified();
}


void PdfMultiPage::slotDeletePages()
{
  // deleting pages from an empty document doesn't make much sense
  if (dataModel->numberOfPages() == 0)
    return;

  KDialog dialog( parentWdg);
  dialog.setCaption( i18n("Delete Pages") );
  dialog.setButtons( KDialog::Ok|KDialog::Cancel );
  dialog.setDefaultButton( KDialog::Ok );

  PageRangeWidget range( 1, dataModel->numberOfPages(), dataModel->currentPageNumber(), &dialog );
  range.setToolTip( i18n( "Select the pages you wish to delete." ) );
  dialog.setButtonText(KDialog::Ok, i18n("Delete Pages"));
  dialog.setMainWidget(&range);
  if (dialog.exec() != QDialog::Accepted)
    return;

  pdfRenderer.deletePages(range.getFrom(), range.getTo());
  dataModel->deletePages(range.getFrom(), range.getTo());
  emit documentHasBeenModified();
}


void PdfMultiPage::slotInsertPages()
{
  // Inserting into an empty document doesn't make much sense
  if (dataModel->numberOfPages() == 0)
    return;

  // Obtain file name for insertion
  QString filter = i18n("*.pdf|PDF file (*.pdf)");
  QString fname = KFileDialog::getOpenFileName(KUrl("kfiledialog:///pdf?global"), filter, parentWdg, i18n("Insert PDF file..."));
  if (fname.isEmpty())
    return;

  // Show dialog, so user can choose where to insert the file
  KDialog dialog( parentWdg);
  dialog.setCaption( i18n("Insert Pages...") );
  dialog.setModal( true );
  dialog.setButtons( KDialog::Ok|KDialog::Cancel );
  dialog.setDefaultButton( KDialog::Ok );
  PageInsertionWidget pgins( dataModel->currentPageNumber(), dataModel->numberOfPages(), fname, filter, &dialog );
  pgins.setToolTip( i18n("Select the file you wish to insert, and the plase where the file should be inserted."));
  dialog.setButtonText(KDialog::Ok, i18n("Insert Pages"));
  dialog.setMainWidget(&pgins);

  // Enable OK-button only if the selected file exists
#ifdef __GNUC__
#warning CHECK THIS
#endif  
  //  dialog.enableButtonOK(QFile::exists(pgins.getFile()));
  connect(&pgins, SIGNAL(have_valid_file(bool)), &dialog, SLOT(enableButtonOK(bool)));

  // Show dialog
  if (dialog.exec() != QDialog::Accepted)
    return;

  PageNumber oldNrPages = pdfRenderer.totalPages();
  pdfRenderer.insertPages(pgins.getFile(), pgins.getPage());

  int numInserted = pdfRenderer.totalPages() - oldNrPages;
  if (numInserted > 0)
    dataModel->insertPages(pgins.getPage(), numInserted);
  else {
    kError() << "Inserting pages did not increase the number of pages!" << endl;
    dataModel->setNumberOfPages(pdfRenderer.totalPages());
  }

  emit documentHasBeenModified();
}


void PdfMultiPage::slotSetBackground()
{
  // Paranoid safety checks
  if (pdfRenderer.isEmpty())
    return;
  if (m_file.isEmpty())
    return;

  // Obtain file name for insertion
  QString filter = i18n("*.pdf|PDF file (*.pdf)");
  QString fname = KFileDialog::getOpenFileName(KUrl("kfiledialog:///pdf?global"),  filter, parentWdg, i18n("Select PDF file for background..."));
  if (fname.isEmpty())
    return;

  pdfRenderer.setBackground(fname);
  emit documentHasBeenModified();
}


#include "pdfMultipage.moc"
