/***************************************************************************
 *   Copyright (C) 2005-2006 by Stefan Kebekus                             *
 *   kebekus@kde.org                                                       *
 *                                                                         *
 *   Copyright (C) 2005-2006 by Wilfried Huss                              *
 *   Wilfried.Huss@gmx.at                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.          *
 ***************************************************************************/

#include <kicontheme.h>
#include <kapplication.h>
#include <kdialog.h>
#include <kglobal.h>
#include <kio/job.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kmimetype.h>
#include <kprocess.h>
#include <kprocio.h>
#include <ktemporaryfile.h>
#include <kvbox.h>

#include <QCheckBox>
#include <QDomDocument>
#include <QFileInfo>
#include <QLayout>
#include <QLabel>
#include <QLineEdit>
#include <QListWidgetItem>
#include <QPainter>
#include <QPrinter>
#include <QRegExp>
#include <QTextEdit>
#include <QStack>
#include <QTextStream>

#include <stdlib.h>
#include <kconfiggroup.h>

#include "documentWidget.h"
#include "hyperlink.h"
#include "kvs_debug.h"
#include "pagetransition.h"
#include "pdfRenderer.h"
#include "pdfMultipage.h"
#include "sourceSplitter.h"
#include "textBox.h"

#include "prefs.h"

#define KF_DEBUG
#define DEBUG_PDFSYNC 0

//#define PDFSYNC_DEBUG

PdfRenderer::PdfRenderer(ligaturePluginGUI* _multiPage)
  : DocumentRenderer(_multiPage)
{
#ifdef KF_DEBUG
  kError(kvs::pdf) << "PdfRenderer()" << endl;
#endif

  document = 0;
  proc     = 0;
  export_errorString = QString::null;
}



PdfRenderer::~PdfRenderer()
{
#ifdef KF_DEBUG
  kDebug(kvs::pdf) << "~PdfRenderer" << endl;
#endif

  clear();

  // Wait for all access to this documentRenderer to finish
  QMutexLocker locker(&mutex);

  delete proc;
}


RenderedDocumentPagePixmap* PdfRenderer::drawPage(const JobId& id)
{
#ifdef KF_DEBUG
  kDebug() << "PdfRenderer::drawPage(JobId) called, page number " << id.pageNumber << endl;
#endif

  // Paranoid safety checks
  if (!id.pageNumber.isValid()) {
    kDebug() << "PdfRenderer::drawPage(JobId) called with a invalid page number" << endl;
    return 0;
  }

  // Wait for all access to this documentRenderer to finish
  QMutexLocker locker(&mutex);

  // more paranoid safety checks
  if (id.pageNumber > numPages) {
    kError() << "PdfRenderer::drawPage(JobId) called with page number " << id.pageNumber
	      << " but the current fax file has only " << numPages << " pages." << endl;
    return 0;
  }

  double res = id.resolution;

  QImage image;

  int pageNumber = id.pageNumber - 1;

  RenderedDocumentPagePixmap* page = multiPage->createDocumentPagePixmap(id);
  page->hyperlinks_are_marked = true; // We always mark hyperlinks, as
				      // specified in the PDF file

  SimplePageSize ps = sizeOfPage(page->getPageNumber());
  int pageHeight = ps.sizeInPixel(res).height();
  int pageWidth = ps.sizeInPixel(res).width();
  page->resize(pageWidth, pageHeight);

  if (document)
  {
    Poppler::Page* pdfPage = document->page(pageNumber);
    if (page)
    {
      if (page->getId().isThumbnail)
        image = pdfPage->renderToImage(res, res);
      else
      {
        image = pdfPage->renderToImage(res, res, -1, -1, -1, -1, true);
      }

      // Create Text information
      fillInText(page, pdfPage, res);

      // Set transition effect
      Poppler::PageTransition* popplerTransition = pdfPage->transition();
      if (popplerTransition) {
        KPDFPageTransition* transition = new KPDFPageTransition();
        transition->setType((KPDFPageTransition::Type)popplerTransition->type());
        transition->setAlignment((KPDFPageTransition::Alignment)popplerTransition->alignment());
        transition->setDirection((KPDFPageTransition::Direction)popplerTransition->direction());

        page->setTransition(transition);
      }

      if (!page->getId().isThumbnail)
      {
        // Set hyperlinks
        QList<Poppler::Link*> links = pdfPage->links();

	kError() << "FOUND " << links.size() << " LINKS" << endl;

        for (int i = 0; i < links.size(); i++)
        {
          Poppler::Link* link = links[i];
          QRect area = link->linkArea().normalized().toRect();

          if (link->linkType() == Poppler::Link::Goto)
          {
            Poppler::LinkGoto* gotoLink = dynamic_cast<Poppler::LinkGoto*>(link);
            if (!gotoLink)
            {
              kError() << "Cast to Poppler::LinkGoto failed." << endl;
	      continue;
            }

	    Poppler::LinkDestination destination = gotoLink->destination();
            QString url(destination.toString());
	    Length y;
	    y.setLength_in_pixel(destination.top(), res);
            Anchor anchor(destination.pageNumber(), y);
	    Hyperlink hyperlink(area.bottom(), area, url, anchor);
	    //	    kdDebug(kvs::pdf) << "PdfRenderer::drawPage: hyperlink added " << hyperlink << endl;
	    page->hyperLinkList.push_back(hyperlink);
          }
          else if (link->linkType() == Poppler::Link::Browse)
          {
            Poppler::LinkBrowse* browseLink = dynamic_cast<Poppler::LinkBrowse*>(link);
            if (!browseLink)
            {
              kError() << "Cast to Poppler::LinkBrowse failed." << endl;
              continue;
            }
            Hyperlink hyperlink(area.bottom(), area, browseLink->url());
            page->hyperLinkList.push_back(hyperlink);
          }
        }
      }

      delete pdfPage;
    }
  }
  else
  {
    delete page;

    return 0;
  }

  page->setImage(image);

#ifdef PDFSYNC_DEBUG
  // Render Targets
  if (!pdfSyncTargets.isEmpty())
  {
    QVector<PdfSyncTarget>& pageTarget = pdfSyncTargets[pageNumber];
    for (unsigned int i = 0; i < pageTarget.count(); i++)
    {
      PdfSyncTarget& target = pageTarget[i];
      int x = target.x.getLength_in_pixel(res);
      int y = target.y.getLength_in_pixel(res);
      foreGroundPaint->setPen(Qt::red);
      foreGroundPaint->drawEllipse(x, y, 5, 5);
    }
  }
#endif

  page->isEmpty = false;
  return page;
}

RenderedDocumentPagePixmap* PdfRenderer::getText(const JobId& id)
{
  QMutexLocker locker(&mutex);

  int pageNumber = id.pageNumber - 1;

  if (document)
  {
    RenderedDocumentPagePixmap* page = multiPage->createDocumentPagePixmap(id);

    Poppler::Page* pdfPage = document->page(pageNumber);
    fillInText(page, pdfPage, 72);
    delete pdfPage;

    return page;
  }
  else
    return 0;
}


void PdfRenderer::fillInText(RenderedDocumentPagePixmap* page, Poppler::Page* pdfPage, double res)
{
  // Create Text information
  SimplePageSize ps = sizeOfPage(page->getPageNumber());
  int pageWidth = ps.sizeInPixel(res).width();
  int pageHeight = ps.sizeInPixel(res).height();

  double scaleX = pageWidth / (double)(pdfPage->pageSize().width());
  double scaleY = pageHeight / (double)(pdfPage->pageSize().height());

  QList<Poppler::TextBox *> pdfTextList = pdfPage->textList();
  QList<Poppler::TextBox *>::iterator it;
  for ( it = pdfTextList.begin(); it != pdfTextList.end(); ++it ) {
    Poppler::TextBox* pdfTextBox = (*it);

    QRect textRect((int)(pdfTextBox->boundingBox().left() * scaleX + 0.5),
		   (int)(pdfTextBox->boundingBox().top() * scaleY + 0.5),
		   (int)(pdfTextBox->boundingBox().width() * scaleX + 0.5),
		   (int)(pdfTextBox->boundingBox().height() * scaleY + 0.5));
    TextBox textBox(textRect, pdfTextBox->text() + " ");
    page->textBoxList.push_back(textBox);
  }
}


bool PdfRenderer::setFile(const QString& fname, const KUrl&)
{
#ifdef KF_DEBUG
  kDebug(kvs::pdf) << "PdfRenderer::setFile(" << fname << ") called" << endl;
#endif

  // Check if we are reloading the file.
  bool reload = (fname == filename);

  clear();

  // Wait for all access to this documentRenderer to finish
  QMutexLocker locker(&mutex);
  
  // If fname is the empty string, then this means: "close", and we
  // exit here.
  if (fname.isEmpty()) {
    kDebug(kvs::pdf) << "PdfRenderer::setFile( ... ) called with empty filename. Closing the file." << endl;
    return true;
  }

  // Paranoid saftey checks: make sure the file actually exists, and
  // that it is a file, not a directory. Otherwise, show an error
  // message and exit..
  QFileInfo fi(fname);
  filename = fi.absoluteFilePath();
  workingCopyFileName = filename;

  if (!fi.exists() || fi.isDir()) {
    KMessageBox::error( parentWidget,
			i18n("<qt><strong>File error.</strong> The specified file '%1' does not exist.</qt>", filename),
			i18n("File Error"));
    // the return value 'false' indicates that this operation was not successful.
    return false;
  }

  // Now we assume that the file is fine and load the file.
  document = Poppler::Document::load(filename.toAscii());
  if (!document || document->isLocked())
  {
    KMessageBox::error( parentWidget,
			i18n("<qt><strong>File error.</strong> The specified file '%1' could not be loaded.</qt>", filename),
			i18n("File Error!"));
    kDebug(kvs::pdf) << "Loading of document failed." << endl;
    return false;
  }
  
  // Set the number of pages page sizes
  kDebug(kvs::pdf) << "get number of pages." << endl;
  numPages = document->numPages();

  kDebug(kvs::pdf) << "Number of Pages = " << numPages << endl;

  // Set the page sizes in the pageSizes array
  pageSizes.resize(numPages);
  Length w,h;

  //kDebug(kvs::pdf) << "get page resolution." << endl;
  //int resolution = dpdf_page_get_resolution(page);

  kDebug(kvs::pdf) << "get page width." << endl;
  //double pageWidth = document->getPageWidth(0);

  kDebug(kvs::pdf) << "get page height." << endl;
  //int pageHeight = document->getPageHeight(0);

  //kDebug(kvs::pdf) << "document page size (" << pageHeight << ", " << pageWidth << ")" << endl;

  for (int i=0; i<numPages; i++)
  {
    Poppler::Page* pdfPage = document->page(i);
    QSize pageSize = pdfPage->pageSize();
    int width = pageSize.width();
    int height = pageSize.height();

    if (pdfPage->orientation() == Poppler::Page::Landscape ||
        pdfPage->orientation() == Poppler::Page::Seascape)
    {
      width = pageSize.height();
      height = pageSize.width();
    }
    delete pdfPage;
    pdfPage = 0;

    w.setLength_in_bigPoints(width);
    h.setLength_in_bigPoints(height);
    pageSizes[i].setPageSize(w, h);
  }

  // Store path to current file
  path = fi.absolutePath();

  // Build the table of contents
  getOutline();
  
  // Load pdfsync file if available
  QString pdfsync_filename = path + "/" + fi.baseName() + ".pdfsync";
  kDebug(kvs::pdf) << "search for pdfsync file " << pdfsync_filename << endl;
  QFileInfo pdfsync_fi(pdfsync_filename);
  if (pdfsync_fi.exists())
  {
    // Parse pdfsync
    kDebug(kvs::pdf) << "parse pdfsync file format" << endl;

    QFile pdfsync_file(pdfsync_filename);
    if (pdfsync_file.open(QIODevice::ReadOnly))
    {
      // Tell the user (once) if the PDF has an corresponding pdfsync file.
      // ... we don't want our great feature to go unnoticed.

      // Show the dialog as soon as event processing is finished, and
      // the program is idle
      if (!reload)
        QTimer::singleShot(0, this, SLOT(showThatSourceInformationIsPresent()));

      QTextStream stream(&pdfsync_file);
      parsePdfSync(fi.baseName(), stream);
      pdfsync_file.close();
    }
  }

  // the return value 'true' indicates that this operation was not successful.
  return true;
}


void PdfRenderer::getOutline()
{
  if (document == 0)
    return;

  bookmarks.clear();

  QDomDocument* toc = document->toc();

  // not all documents have a table of content
  if (!toc)
    return;

  //kDebug() << toc->toString() << endl;
  
  QDomElement root = toc->documentElement();

  QDomNode topLevelNode = root;
  while (topLevelNode.isElement())
  {
    QString name = topLevelNode.toElement().tagName();
    QString destinationName = topLevelNode.toElement().attribute("DestinationName"); 
    Poppler::LinkDestination* destination = document->linkDestination(destinationName);

    int pageNumber = 1;
    if (destination)
    {
      pageNumber = destination->pageNumber();
    }
 
    Bookmark* bookmarkEntry = new Bookmark(name, pageNumber, Length());
    bookmarks.append(bookmarkEntry);

    getOutlineSub(&topLevelNode, bookmarkEntry);
    
    topLevelNode = topLevelNode.nextSibling();
  }

  delete toc;
}


void PdfRenderer::getOutlineSub(QDomNode* tocParent, Bookmark* bookmarkParent)
{
  if (!tocParent || !bookmarkParent)
    return;

  QDomNode n = tocParent->firstChild();
  while (n.isElement())
  {
    QDomElement e = n.toElement();
    QString name = e.tagName();
    QString destinationName = e.attribute("DestinationName"); 
    
    Poppler::LinkDestination* destination = document->linkDestination(destinationName);
    int pageNumber = 1;
    if (destination)
    {
      pageNumber = destination->pageNumber();
    }
 
    Bookmark* bookmarkEntry = new Bookmark(name, pageNumber, Length());

    bookmarkParent->subordinateBookmarks.append(bookmarkEntry);
    
    getOutlineSub(&n, bookmarkEntry);

    n = n.nextSibling();
  }
}


void PdfRenderer::clear(void)
{
  DocumentRenderer::clear();

  // Wait for all access to this documentRenderer to finish
  QMutexLocker locker(&mutex);

  // Clear previously loaded document
  delete document;
  document = 0;

  // Clear data structures
  pdfSyncLines.clear();
  pdfSyncTargets.clear();

  QFileInfo wc(workingCopyFileName);
  QFileInfo oc(filename);
  if (wc.absoluteFilePath() != oc.absoluteFilePath())
    QFile::remove(workingCopyFileName);
}


bool PdfRenderer::isValidFile(const QString& fileName) const
{
  Poppler::Document* doc = Poppler::Document::load(fileName.toAscii());
  if (doc)
  {
    delete doc;
    return true;
  }
  return false;
}

bool PdfRenderer::parsePdfSync(QString basename, QTextStream& stream)
{
  // The first line should contain the basename
  QString pdfsync_basename = stream.readLine();
  if (basename != pdfsync_basename)
  {
    kDebug(kvs::pdf) << "Wrong basename: " << pdfsync_basename << endl;
    return false;
  }

  // The second line should contain the version number
  QRegExp versionExp = QRegExp("version\\s+(\\d+)");
  if (!versionExp.exactMatch(stream.readLine())) //TODO: check version number
  {
    kDebug(kvs::pdf) << "Wrong version number" << endl;
    return false;
  }

  // The top of the stack always holds the name of the file, that the
  // current entries correspond to. If the stack is empty the entries
  // correspond to the basefile.
  QStack<QString> filestack;

  QRegExp openBraceExp = QRegExp("\\((.*)");
  QRegExp pageExp = QRegExp("s\\s+(\\d+)");
  QRegExp lineExp = QRegExp("l\\s+(\\d+)\\s+(\\d+)");
  QRegExp pointExp = QRegExp("p\\s+(\\d+)\\s+(\\d+)\\s+(\\d+)");

  QString line;
  int currentPage = 1;

  pdfSyncTargets.resize(numPages);

  while (!stream.atEnd())
  {
    line = stream.readLine(); // line of text excluding '\n'
    if (openBraceExp.exactMatch(line))
    {
      QString filename = openBraceExp.capturedTexts().last();
      //kDebug(kvs::pdf) << "switch to file: " << filename << endl;
      filestack.push(filename);
    }
    else if (line == ")")
    {
      //kDebug(kvs::pdf) << "finish parsing file : " << filestack.top() << endl;
      filestack.pop();
    }
    else if (pageExp.exactMatch(line))
    {
      currentPage = pageExp.capturedTexts().last().toInt();
      //kDebug(kvs::pdf) << "switch to page " << currentPage << endl;
    }
    else if (lineExp.exactMatch(line))
    {
      int record = lineExp.capturedTexts()[1].toInt();
      int linenumber = lineExp.capturedTexts()[2].toInt();

      //kDebug(kvs::pdf) << "line:   " << line << endl;
      //kDebug(kvs::pdf) << "parsed: " << "l " << record << " " << linenumber << endl;

      QString filename;
      if (filestack.isEmpty())
        filename = basename;
      else
        filename = filestack.top();

      pdfSyncLines[record] = PdfSyncLine(linenumber, filename);
    }
    else if (pointExp.exactMatch(line))
    {
      int record = pointExp.capturedTexts()[1].toInt();
      int x = pointExp.capturedTexts()[2].toInt();
      int y = pointExp.capturedTexts()[3].toInt();

      Length pageHeight = pageSizes[currentPage].height();

      Length x_;
      Length y_;

      x_.setLength_in_scaledPoints(x);
      y_.setLength_in_scaledPoints(y);
      PdfSyncTarget target(x_, pageHeight-y_, record);

      if (currentPage >= 1 && currentPage <= numPages)
      {
        QVector<PdfSyncTarget>& pageTargets = pdfSyncTargets[currentPage-1];
        pageTargets.push_back(target);
      }

      //kDebug(kvs::pdf) << "line:   " << line << endl;
      //kDebug(kvs::pdf) << "parsed: " << "p " << record << " " << x << " " << y << endl;
    }
  }

  //for (unsigned int i = 0; i < pdfSyncTargets.count(); i++)
  //{
  //  kDebug(kvs::pdf) << "page " << i+1 << " contains " << pdfSyncTargets[i].count() << " targets" << endl;
  //}
  return true;
}


bool PdfRenderer::convertToPSFile(QString filename, QList<int> &pageList )
{
  if (document == 0)
    return false;
  
  QPrinter dummy(QPrinter::PrinterResolution);
  dummy.setFullPage(true);
#ifdef __GNUC__
#warning set real page size for printing.
#endif
  dummy.setPageSize(QPrinter::A4);
  int paperWidth = dummy.width();
  int paperHeight = dummy.height();
#ifdef __GNUC__
#warning THIS BLOCKS THE GUI FOR SOME TIME, AN ASYNC METHOD WOULD BE BETTER.
#endif
  return document->print(filename, pageList, 600, 600, 0, paperWidth, paperHeight, 0, 0, 0, 0, false);
}


bool PdfRenderer::save(const QString &new_filename)
{
  if (new_filename.isEmpty())
    return false;

  QFileInfo src(workingCopyFileName);
  QFileInfo dest(new_filename);

  if (src.absoluteFilePath() == dest.absoluteFilePath())
    return true;

  KProcIO proc;
  proc << "cp" << src.absoluteFilePath() << dest.absoluteFilePath();
  if (proc.start(KProcess::Block) == false) {
    KMessageBox::error(parentWidget, i18n("<qt>Could not start the <strong>cp</strong> copy command. The document is <strong>not</strong> saved.</qt>"),
                       i18n("Error saving document"));
    return false;
  }

  if (!proc.normalExit() || (proc.exitStatus() != 0)) {
    KMessageBox::error(parentWidget, i18n("<qt>Error saving the document. The document is <strong>not</strong> saved.</qt>"),
                       i18n("Error saving document"));
    return false;
  }

  _isModified = false;
  return true;
}


void PdfRenderer::inverseSearch(PageNumber pageNumber, Length x, Length y, bool call)
{
  if (!pageNumber.isValid() || pageNumber > numPages)
    return;

  if (pdfSyncTargets.isEmpty())
    return;

  kDebug(kvs::pdf) << "inverseSearch on page " << pageNumber 
            << " (" << x.getLength_in_cm() << "cm, " << y.getLength_in_cm() << "cm)" << endl;

  QVector<PdfSyncTarget>& pageTarget = pdfSyncTargets[pageNumber - 1];

  if (pageTarget.isEmpty())
    return;

  double minimum = 0;
  int minIndex = 0;

  for (int i = 0; i < pageTarget.count(); i++)
  {
    // Remember the closest source link
    double dx = x.getLength_in_cm() - pageTarget[i].x.getLength_in_cm();
    double dy = y.getLength_in_cm() - pageTarget[i].y.getLength_in_cm();

    if (dx*dx + dy*dy < minimum || i == 0)
    {
      minIndex = i;
      minimum = dx*dx + dy*dy;
    }
  }
  int record = pageTarget[minIndex].record;

  kDebug(kvs::pdf) << "minimum abs = " << minimum << "cm" << endl;
  kDebug(kvs::pdf) << "record = " << pageTarget[minIndex].record << endl;
  kDebug(kvs::pdf) << "target = (" << pageTarget[minIndex].x.getLength_in_cm() << ", "
                   << pageTarget[minIndex].y.getLength_in_cm() << ")" << endl;

  QString filename = pdfSyncLines[record].filename;

  int linenumber = pdfSyncLines[record].linenumber;

  kDebug(kvs::pdf) << "jump to line " << linenumber << " of file " << filename << endl;
  
  QString temp = filename;
  if (filename.right(4) != ".tex")
    temp += ".tex";
  emit setStatusBarText(QString("line %1 of file %2").arg(linenumber).arg(temp));

  if (call)
    handleSRCLink(filename, linenumber);
}


void PdfRenderer::handleSRCLink(const QString& filename, int linenumber)
{
  //TODO: smarter filename guessing
  QString texFile = path + "/" + filename;
  QFileInfo fi(texFile);
  if (!fi.exists())
    texFile = texFile + ".tex";

  QFileInfo fitex(texFile);
  if (!fitex.exists())
  {
    KMessageBox::sorry(parentWidget, QString("<qt>") +
      i18n("The PDF-file refers to the TeX-file "
      "<strong>%1</strong> which could not be found.", KShellProcess::quote(texFile)) +
      QString("</qt>"),
      i18n( "Could Not Find File" ));
    return;
  }

  QString command = Prefs::editorCommand();
  if (command.isEmpty() == true)
  {
    int r = KMessageBox::warningContinueCancel(parentWidget, QString("<qt>") +
            i18n("You have not yet specified an editor for inverse search. "
                 "Please choose your favorite editor in the "
                 "<strong>PDF options dialog</strong> "
                 "which you will find in the <strong>Settings</strong>-menu.") +
                 QString("</qt>"),
            i18n("Need to Specify Editor"),
            KGuiItem(i18n("Use KDE's Editor Kate for Now")));
    if (r == KMessageBox::Continue)
      command = "kate --line %l %f";
    else
      return;
  }
  QString linenumberStr;
  linenumberStr = linenumberStr.setNum(linenumber);
  command = command.replace("%l", linenumberStr).replace("%f", KProcess::quote(texFile));
  
  // Set up a shell process with the editor command.
  proc = new KShellProcess();
  if (proc == 0)
  {
    kError(kvs::pdf) << "Could not allocate ShellProcess for the editor command." << endl;
    return;
  }

  proc->clearArguments();
  *proc << command;
  proc->closeStdin();

  if (proc->start(KProcess::NotifyOnExit, KProcess::AllOutput) == false)
  {
    KMessageBox::sorry(parentWidget, i18n("<qt>You have asked Ligature to jump to the" 
        "line %1 in the TeX-file <strong>%2</strong>,"
        "but Ligature was not able to start the selected texteditor."
        "Please choose a texteditor in the Configuration Dialog that is"
        "installed on your system.</qt>", linenumber, filename),
        i18n("Could Not Open Texteditor"));
    return;
  }
}


void PdfRenderer::showThatSourceInformationIsPresent()
{
  // In principle, we should use a KMessagebox here, but we want to
  // add a button "Explain in more detail..." which opens the
  // Helpcenter. Thus, we practically re-implement the KMessagebox
  // here. Most of the code is stolen from there.

  // Check if the 'Don't show again' feature was used
  KConfig* config = kapp->sessionConfig();
  KConfigGroup saver(config, "Notification Messages");
  bool showMsg = config->readEntry("Ligature-info_on_pdfsync", true);

  if (showMsg)
  {
    KDialog* dialog= new KDialog(parentWidget);
    dialog->setCaption( i18n("Ligature: Information" ) );
    dialog->setButtons( KDialog::Yes );
    dialog->setModal( true );
    KVBox* topcontents = new KVBox(dialog);
    topcontents->setSpacing(KDialog::spacingHint() * 2);
    topcontents->setMargin(KDialog::marginHint() * 2);

    QWidget* contents = new QWidget(topcontents);
    QHBoxLayout* lay = new QHBoxLayout(contents);

    lay->setSpacing(KDialog::spacingHint() * 2);
    lay->addStretch(1);

    QLabel* label1 = new QLabel(contents);
    label1->setPixmap(QMessageBox::standardIcon(QMessageBox::Information));
    lay->addWidget(label1);
    QLabel* label2 = new QLabel(i18n("<qt>A pdfsync file has been found. You may click into the text with the "
      "middle mouse button, and an editor will open the TeX-source file immediately.</qt>"), contents);
    label2->setMinimumSize(label2->sizeHint());
    lay->addWidget(label2);
    lay->addStretch(1);

    QSize extraSize = QSize(50,30);
    QCheckBox* checkbox = new QCheckBox(i18n("Do not show this message again"), topcontents);
    extraSize = QSize(50,0);

    dialog->setHelpLinkText(i18n("Explain in more detail..."));
    dialog->setHelp("inverse-search", "kpdfview");
    dialog->enableLinkedHelp(true);
    dialog->setMainWidget(topcontents);
    dialog->showButtonSeparator(false);
    dialog->setInitialSize(extraSize);
    dialog->exec();
    delete dialog;

    showMsg = !checkbox->isChecked();
    if (!showMsg)
    {
      KConfigGroup saver(config, "Notification Messages");
      config->writeEntry("Ligature-info_on_pdfsync", showMsg);
    }
    config->sync();
  }
}


Anchor PdfRenderer::parseReference(const QString& reference)
{
  //QMutexLocker locker(&mutex);

#ifdef DEBUG_DVIRENDERER
  kDebug(kvs::pdf) << "dviRenderer::parseReference( " << reference << " ) called" << endl;
#endif

  if (document == 0)
  {
    return Anchor();
  }

  // case 1: The reference is a number, which we'll interpret as a
  // page number.
  bool ok;
  int page = reference.toInt(&ok);
  if (ok == true)
  {
    if (page < 1)
      page = 1;
    if (page > numPages)
      page = numPages;

    return Anchor(page, Length());
  }

  // case 2: The reference is of form "src:1111Filename", where "1111"
  // points to line number 1111 in the file "Filename". Ligature then
  // looks for pdfsync targets that are close to the given line number
  if (reference.indexOf("src:", 0, Qt::CaseInsensitive) == 0)
  {
    // Extract the file name and the numeral part from the reference string
    SourceFileSplitter splitter(reference, filename);
    int refLineNumber = splitter.line();
    QString refFileName = splitter.filePath();

    kDebug(kvs::pdf) << "Ref file = " << refFileName << endl;
    kDebug(kvs::pdf) << "Ref line = " << refLineNumber << endl;

    if (pdfSyncLines.isEmpty() || pdfSyncTargets.isEmpty())
    {
      KMessageBox::sorry(parentWidget, i18n("<qt>You have asked Ligature to locate the place in the PDF file which corresponds to "
        "line %1 in the TeX-file <strong>%2</strong>. It seems, however, that there does not exist a corresponding "
        "pdfsync file, which contains the necessary source file information. "
        "We refer to the manual of Ligature for a detailed explanation on how to include this "
        "information. Press the F1 key to open the manual.</qt>", refLineNumber, refFileName),
        i18n("Could Not Find Reference"));
      return Anchor();
    }

    bool referenceFound = false;
    int key = 0;
    int minimumDelta = 0;

    QMap<int, PdfSyncLine>::Iterator it;
    for (it = pdfSyncLines.begin(); it != pdfSyncLines.end(); ++it)
    {
      PdfSyncLine& pdfSyncLine = it.value();
      QString targetFilename = path + "/" + pdfSyncLine.filename;
      //kDebug(kvs::pdf) << "test target filename = " << targetFilename << endl;
      if (targetFilename == refFileName || targetFilename + ".tex" == refFileName)
      {
        if (pdfSyncLine.linenumber == refLineNumber)
        {
          key = it.key();
          minimumDelta = 0;
          kDebug(kvs::pdf) << "Exact Key found: " << key << endl;
          break;
        }
        else
        {
          int delta = pdfSyncLine.linenumber - refLineNumber;
          if (delta < 0)
            delta = -delta;

          if (delta < minimumDelta || !referenceFound)
          {
            referenceFound = true;
            minimumDelta = delta;
            key = it.key();
          }
        }
      }
    }

    kDebug(kvs::pdf) << "Key found: " << key << ", Offset: " << minimumDelta << endl;

    for (int i = 0; i < pdfSyncTargets.count(); i++)
    {
      QVector<PdfSyncTarget>& targetList = pdfSyncTargets[i];
      for (int j = 0; j < targetList.count(); j++)
      {
        if (targetList[j].record == key)
        {
          kDebug(kvs::pdf) << "Reference to page " << i << endl;
          kDebug(kvs::pdf) << "Distance from top " << targetList[j].y.getLength_in_cm() << "cm" << endl;
          return Anchor(i+1, targetList[j].y);
        }
      }
    }
  }
  return Anchor();
}


QString toPDFTKencoding(QString in)
{
  QString out;

  for(int i=0; i < in.length(); i++) {
    const QChar c = in[i];
    if (!c.isPrint() || (c.toLatin1() < 33) || (c.toLatin1() > 126))
      out += "&#" + QString::number( c.toLatin1() ) + ";";
    else
      out += c;
  }
  return out;
}


bool PdfRenderer::showInfo()
{
  KDialog dialog(parentWidget);
  dialog.setCaption( i18n("Document Properties") );
  dialog.setButtons( KDialog::Ok|KDialog::Cancel );
  dialog.setModal( true );


  //  KDialog dialog( parentWidget, "urldialog", true, i18n("Document Properties"), KDialog::Ok|KDialog::Cancel, KDialog::Ok, true );

  // Setup the main properties widget
  DocumentInfoWidget_base infoWdg( &dialog );
  KMimeType::Ptr mimetype = KMimeType::findByPath( filename );
  infoWdg.mimeTypePix->setPixmap(mimetype->pixmap(K3Icon::NoGroup, 48));
  QFileInfo fi(filename);
  infoWdg.m_filename->setText( fi.fileName() );
  infoWdg.m_title->setText( document->info("Title") );
  infoWdg.m_title->setCursorPosition(0);
  infoWdg.m_subject->setText( document->info("Subject") );
  infoWdg.m_subject->setCursorPosition(0);
  infoWdg.m_author->setText( document->info("Author") );
  infoWdg.m_author->setCursorPosition(0);
  infoWdg.m_keywords->setText( document->info("Keywords") );
  infoWdg.m_keywords->setCursorPosition(0,0);
  infoWdg.m_path->setText( fi.absoluteFilePath() );
  infoWdg.m_size->setText( KIO::convertSize(fi.size()) );
  infoWdg.m_modified->setText( KGlobal::locale()->formatDateTime(fi.lastModified()) );
  infoWdg.m_pages->setText( QString("%1").arg(numPages) );

#ifdef __GNUC__
#warning FIXME
#endif
  /*
  // Setup the fonts widget
  QString fontTypeNames[8] =
    { i18n("unknown"), i18n("Type 1"), i18n("Type 1C"), i18n("Type 3"), i18n("TrueType"), i18n("CID Type 0"),
      i18n("CID Type 0C"), i18n("CID TrueType") };

  QList<Poppler::FontInfo> fontList = document->fonts();
  for (unsigned i = 0; i < fontList.count(); i++) {
    QListWidgetItem* item = new QListWidgetItem(infoWdg.m_fonts);
    QString s;
    s = fontList[i].name();
    item->setText(0, s);
    s = fontTypeNames[fontList[i].type()]
    item->setText(1, s);

    if (fontList[i].isEmbedded()) {
      item->setText(2, i18n("Yes"));

      if (fontList[i].isSubset())
        item->setText(3, i18n("Yes"));
      else
        item->setText(3, i18n("No"));
    } else
      item->setText(2, i18n("No"));
  }
  */
#ifdef __GNUC__
#warning TODO: Make sure 'OK' can only be clicked if text fields were changed
#endif

  // Show the dialog
  //  dialog.setMainWidget(&infoWdg);
  dialog.exec();

  if (dialog.result() != QDialog::Accepted)
    return false;

  if (infoWdg.m_title->isModified() || infoWdg.m_subject->isModified() ||
      infoWdg.m_author->isModified() || infoWdg.m_keywords->isModified() ) {
    KTemporaryFile new_info;
    new_info.setSuffix(".info");
    if (!new_info.open()) {
      kError() << "Could not open temporary info file for writing." << endl;
      return false;
    }
    QTextStream str ( &new_info );

    if (infoWdg.m_title->isModified())
      str << "InfoKey: Title" << endl
          << QString("InfoValue: ") << toPDFTKencoding(infoWdg.m_title->text()) << endl;
    if (infoWdg.m_subject->isModified())
      str << QString("InfoKey: Subject") << endl
          << QString("InfoValue: ") << toPDFTKencoding(infoWdg.m_subject->text()) << endl;
    if (infoWdg.m_author->isModified())
      str << QString("InfoKey: Author") << endl
          << QString("InfoValue: ") << toPDFTKencoding(infoWdg.m_author->text()) << endl;
    if (infoWdg.m_keywords->isModified())
      str << QString("InfoKey: Keywords") << endl
          << QString("InfoValue: ") << toPDFTKencoding(infoWdg.m_keywords->text().simplified()) << endl;
    str.flush();

    KTemporaryFile new_wcopy;
    new_wcopy.setSuffix(".pdf");
    new_wcopy.setAutoRemove(false);
    new_wcopy.open();

    KProcIO proc;
    proc << "pdftk" << workingCopyFileName << "update_info" << new_info.fileName() << "output" << new_wcopy.fileName();
    if (proc.start(KProcess::Block, true) == false) {
      KMessageBox::detailedError( parentWidget,
                                 i18n("<qt>An error occurred. The command <strong>pdftk</strong> could not be started, "
                                      "and the background could not be set.</qt>"),
                                 i18n("<p>This error occurs most likely because the program <strong>pdftk</strong>, which "
                                      "is part of the PDF toolkit, is not installed on your computer. If you believe "
                                      "the toolkit is installed, you could try to start the <strong>pdftk</strong> "
                                      "command from the shell in order to check that.</p>"
                                      "<p>The PDF toolkit can be downloaded for free from http://www.accesspdf.com/pdftk.</p>"
                                      "<p>The following paths were tried to locate the <strong>pdftk</strong> program: "
                                      "%1</p>", QString(getenv("PATH")).replace(":", "\n")),
                                 i18n("Error setting info data!"));
      QFile::remove(new_wcopy.fileName());
      return false;
    }

    QString MSG, line;
    while(proc.readln(line) != -1)
      MSG += line;

    if (proc.exitStatus() != 0) {
      KMessageBox::detailedError( parentWidget,
                                 i18n("<qt>An error occurred. The info data could not be set.</qt>"),
                                 QString("<qt>The <strong>pdftk</strong> command gave the following output: <quote>%1</quote></qt>").arg(MSG),
                                 i18n("Error changing info data background!"));
      QFile::remove(new_wcopy.fileName());
      return false;
    }

    _isModified = true;

    clear();
    workingCopyFileName = new_wcopy.fileName();

    // Now we assume that the file is fine and load the file.
    document = Poppler::Document::load(workingCopyFileName.toAscii());
    if (!document || document->isLocked()) {
      KMessageBox::error( parentWidget,
                         i18n("<qt><strong>File error.</strong> The specified file '%1' could not be loaded.</qt>", filename),
                         i18n("File Error!"));
      return true; //    return false;
    }

    // Set the number of pages page sizes
    numPages = document->numPages();

    // Set the page sizes in the pageSizes array
    pageSizes.resize(numPages);
    Length w,h;
    for (int i=0; i<numPages; i++) {
      Poppler::Page* pdfPage = document->page(i);
      QSize pageSize = pdfPage->pageSize();
      int width = pageSize.width();
      int height = pageSize.height();

      if (pdfPage->orientation() == Poppler::Page::Landscape ||
         pdfPage->orientation() == Poppler::Page::Seascape) {
       width = pageSize.height();
       height = pageSize.width();
      }
      delete pdfPage;
      pdfPage = 0;

      w.setLength_in_bigPoints(width);
      h.setLength_in_bigPoints(height);
      pageSizes[i].setPageSize(w, h);
    }

    // the return value 'true' indicates that this operation was successful.
    return true;
  }
  return false;
}


void PdfRenderer::setBackground(QString bgFile)
{
  KTemporaryFile new_wcopy;
  new_wcopy.setSuffix(".pdf");
  new_wcopy.setAutoRemove(false);
  new_wcopy.open();

  KProcIO proc;
  proc << "pdftk" << workingCopyFileName << "background" << bgFile << "output" << new_wcopy.fileName();
  if (proc.start(KProcess::Block, true) == false) {
    KMessageBox::detailedError( parentWidget,
                               i18n("<qt>An error occurred. The command <strong>pdftk</strong> could not be started, "
                                    "and the background could not be set.</qt>"),
                               i18n("<p>This error occurs most likely because the program <strong>pdftk</strong>, which "
                                    "is part of the PDF toolkit, is not installed on your computer. If you believe "
                                    "the toolkit is installed, you could try to start the <strong>pdftk</strong> "
                                    "command from the shell in order to check that.</p>"
                                    "<p>The PDF toolkit can be downloaded for free from http://www.accesspdf.com/pdftk.</p>"
                                    "<p>The following paths were tried to locate the <strong>pdftk</strong> program: "
                                    "%1</p>", QString(getenv("PATH")).replace(":", "\n")),
                               i18n("Error setting background!"));
    QFile::remove(new_wcopy.fileName());
    return;
  }

  QString MSG, line;
  while(proc.readln(line) != -1)
    MSG += line;

  if (proc.exitStatus() != 0) {
    KMessageBox::detailedError( parentWidget,
                               i18n("<qt>An error occurred. The background could not be set.</qt>"),
                               QString("<qt>The <strong>pdftk</strong> command gave the following output: <quote>%1</quote></qt>").arg(MSG),
                               i18n("Error setting background!"));
    QFile::remove(new_wcopy.fileName());
    return;
  }

  _isModified = true;

  clear();
  workingCopyFileName = new_wcopy.fileName();

  // Now we assume that the file is fine and load the file.
  document = Poppler::Document::load(workingCopyFileName.toAscii());
  if (!document || document->isLocked()) {
    KMessageBox::error( parentWidget,
                       i18n("<qt><strong>File error.</strong> The specified file '%1' could not be loaded.</qt>", filename),
                       i18n("File Error!"));
    return; //    return false;
  }

  // Set the number of pages page sizes
  numPages = document->numPages();

  // Set the page sizes in the pageSizes array
  pageSizes.resize(numPages);
  Length w,h;
  for (int i=0; i<numPages; i++) {
    Poppler::Page* pdfPage = document->page(i);
    QSize pageSize = pdfPage->pageSize();
    int width = pageSize.width();
    int height = pageSize.height();

    if (pdfPage->orientation() == Poppler::Page::Landscape ||
        pdfPage->orientation() == Poppler::Page::Seascape) {
      width = pageSize.height();
      height = pageSize.width();
    }
    delete pdfPage;
    pdfPage = 0;

    w.setLength_in_bigPoints(width);
    h.setLength_in_bigPoints(height);
    pageSizes[i].setPageSize(w, h);
  }

  // the return value 'true' indicates that this operation was not successful.
  return; //  return true;
}


void PdfRenderer::deletePages(quint16 from, quint16 to)
{
  KTemporaryFile new_wcopy;
  new_wcopy.setSuffix(".pdf");
  new_wcopy.setAutoRemove(false);
  new_wcopy.open();

  KProcIO proc;
  if (from == 1)
    // Delete pages from the beginning of the document
    proc << "pdftk" << workingCopyFileName << "cat" << QString("%1-end").arg(to+1) << "output" << new_wcopy.fileName();
  else
    if (to == numPages)
      // Delete pages from the end of the document
      proc << "pdftk" << workingCopyFileName << "cat" << QString("1-%1").arg(to) << "output" << new_wcopy.fileName();
    else
      // Delete pages from the middle of the document
      proc << "pdftk" << workingCopyFileName << "cat" << QString("1-%1").arg(from-1) << QString("%1-end").arg(to+1) << "output" << new_wcopy.fileName();

  if (proc.start(KProcess::Block, true) == false) {
    KMessageBox::detailedError( parentWidget,
                               i18n("<qt>An error occurred. The command <strong>pdftk</strong> could not be started, "
                                    "and the background could not be set.</qt>"),
                               i18n("<p>This error occurs most likely because the program <strong>pdftk</strong>, which "
                                    "is part of the PDF toolkit, is not installed on your computer. If you believe "
                                    "the toolkit is installed, you could try to start the <strong>pdftk</strong> "
                                    "command from the shell in order to check that.</p>"
                                    "<p>The PDF toolkit can be downloaded for free from http://www.accesspdf.com/pdftk.</p>"
                                    "<p>The following paths were tried to locate the <strong>pdftk</strong> program: "
                                    "%1</p>", QString(getenv("PATH")).replace(":", "\n")),
                               i18n("Error deleting page!"));
    QFile::remove(new_wcopy.fileName());
    return;
  }

  QString MSG, line;
  while(proc.readln(line) != -1)
    MSG += line;

  if (proc.exitStatus() != 0) {
    KMessageBox::detailedError( parentWidget,
                               i18n("<qt>An error occurred. The pages could not be deleted.</qt>"),
                               QString("<qt>The <strong>pdftk</strong> command gave the following output: <quote>%1</quote></qt>").arg(MSG),
                               i18n("Error deleting page!"));
    QFile::remove(new_wcopy.fileName());
    return;
  }

  _isModified = true;

  clear();
  workingCopyFileName = new_wcopy.fileName();

  // Now we assume that the file is fine and load the file.
  document = Poppler::Document::load(workingCopyFileName.toAscii());
  if (!document || document->isLocked()) {
    KMessageBox::error( parentWidget,
                       i18n("<qt><strong>File error.</strong> The specified file '%1' could not be loaded.</qt>", filename),
                       i18n("File Error!"));
    return; //    return false;
  }

  // Set the number of pages page sizes
  numPages = document->numPages();

  // Set the page sizes in the pageSizes array
  pageSizes.resize(numPages);
  Length w,h;
  for (int i=0; i<numPages; i++) {
    Poppler::Page* pdfPage = document->page(i);
    QSize pageSize = pdfPage->pageSize();
    int width = pageSize.width();
    int height = pageSize.height();

    if (pdfPage->orientation() == Poppler::Page::Landscape ||
        pdfPage->orientation() == Poppler::Page::Seascape) {
      width = pageSize.height();
      height = pageSize.width();
    }
    delete pdfPage;
    pdfPage = 0;

    w.setLength_in_bigPoints(width);
    h.setLength_in_bigPoints(height);
    pageSizes[i].setPageSize(w, h);
  }

  // the return value 'true' indicates that this operation was not successful.
  return; //  return true;
}


void PdfRenderer::insertPages(QString file, PageNumber page)
{
  KTemporaryFile new_wcopy;
  new_wcopy.setSuffix(".pdf");
  new_wcopy.setAutoRemove(false);
  new_wcopy.open();

  KProcIO proc;
  if (page == 1)
    // Prepend the file to the current document
    proc << "pdftk" << file << workingCopyFileName << "cat" << "output" << new_wcopy.fileName();
  else
    if (page == numPages+1)
      // Append the file to the current document
      proc << "pdftk" << workingCopyFileName << file << "cat" << "output" << new_wcopy.fileName();
    else
      // Insert the file in the middle of the current docu,ent
      proc << "pdftk" << QString("A=%1").arg(workingCopyFileName) << QString("B=%1").arg(file)
          << "cat" << QString("A1-%1").arg(page-1) << "B" << QString("A%1-end").arg(page) << "output" << new_wcopy.fileName();

  if (proc.start(KProcess::Block, true) == false) {
    KMessageBox::detailedError( parentWidget,
                               i18n("<qt>An error occurred. The command <strong>pdftk</strong> could not be started, "
                                    "and the background could not be set.</qt>"),
                               i18n("<p>This error occurs most likely because the program <strong>pdftk</strong>, which "
                                    "is part of the PDF toolkit, is not installed on your computer. If you believe "
                                    "the toolkit is installed, you could try to start the <strong>pdftk</strong> "
                                    "command from the shell in order to check that.</p>"
                                    "<p>The PDF toolkit can be downloaded for free from http://www.accesspdf.com/pdftk.</p>"
                                    "<p>The following paths were tried to locate the <strong>pdftk</strong> program: "
                                    "%1</p>", QString(getenv("PATH")).replace(":", "\n")),
                               i18n("Error inserting page!"));
    QFile::remove(new_wcopy.fileName());
    return;
  }

  QString MSG, line;
  while(proc.readln(line) != -1)
    MSG += line;

  if (proc.exitStatus() != 0) {
    KMessageBox::detailedError( parentWidget,
                               i18n("<qt>An error occurred. The file could not be inserted.</qt>"),
                               QString("<qt>The <strong>pdftk</strong> command gave the following output: <quote>%1</quote></qt>").arg(MSG),
                               i18n("Error inserting page!"));
    QFile::remove(new_wcopy.fileName());
    return;
  }

  _isModified = true;

  clear();
  workingCopyFileName = new_wcopy.fileName();

  // Now we assume that the file is fine and load the file.
  document = Poppler::Document::load(workingCopyFileName.toAscii());
  if (!document || document->isLocked()) {
    KMessageBox::error( parentWidget,
                       i18n("<qt><strong>File error.</strong> The specified file '%1' could not be loaded.</qt>", filename),
                       i18n("File Error!"));
    return; //    return false;
  }

  // Set the number of pages page sizes
  numPages = document->numPages();

  // Set the page sizes in the pageSizes array
  pageSizes.resize(numPages);
  Length w,h;
  for (int i=0; i<numPages; i++) {
    Poppler::Page* pdfPage = document->page(i);
    QSize pageSize = pdfPage->pageSize();
    int width = pageSize.width();
    int height = pageSize.height();

    if (pdfPage->orientation() == Poppler::Page::Landscape ||
        pdfPage->orientation() == Poppler::Page::Seascape) {
      width = pageSize.height();
      height = pageSize.width();
    }
    delete pdfPage;
    pdfPage = 0;

    w.setLength_in_bigPoints(width);
    h.setLength_in_bigPoints(height);
    pageSizes[i].setPageSize(w, h);
  }

  // the return value 'true' indicates that this operation was not successful.
  return; //  return true;
}


#include "pdfRenderer.moc"
