// -*- C++ -*-

/***************************************************************************
 *   Copyright (C) 2005-2006 by Stefan Kebekus                             *
 *   kebekus@kde.org                                                       *
 *                                                                         *
 *   Copyright (C) 2005-2006 by Wilfried Huss                              *
 *   Wilfried.Huss@gmx.at                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.          *
 ***************************************************************************/

#ifndef PSRENDERER_H
#define PSRENDERER_H

#include "documentRenderer.h"
#include "internaldocument.h"
#include "pageNumber.h"

#include "ui_documentInfoWidget_base.h"

/*! \brief Well-documented minimal implementation of a documentRenderer
  
This class provides a well-documented reference implementation of a
documentRenderer, suitable as a starting point for a real-world
implementation. This class is responsible for document loading and
rendering. Apart from the constructor and the descructor, it
implements only the necessary methods setFile() and drawPage(). The
method setFile() ignores the file content and simulates a document
with 10 empty pages of A4 size and a few anchors and bookmarks.
*/

class DocumentInfoWidget_base : public QWidget, public Ui::DocumentInfoWidget_base
{
public:
  DocumentInfoWidget_base( QWidget *parent ) : QWidget( parent ) {
    setupUi( this );
  }
};

class GSInternalDocument;
class GSInterpreterLib;
class RenderedDocumentPagePixmap;

class PsRenderer : public DocumentRenderer
{
  Q_OBJECT

public:
   /** Default constructor

       This constructor simply prints a message and calls the default constructor.
   */
   PsRenderer(ligaturePluginGUI*);

   /** Destructor

       The destructor simpley prints a message. It uses the mutex to
       ensure that this class is not destructed while another thread
       is currently using it.
   */
  ~PsRenderer();

  /** Opening a file

      This implementation does the necessary consistency checks and
      complains, e.g. if the file does not exist, but otherwise completely
      disregards the file content. It simulates a document of 10 empty pages of
      A4 size, with a few sample bookmarks and  anchors "p1", "p2", "p3"
      for page 1, 2 and 3, respectively.

      @param fname the name of the file that should be opened.
  */
  virtual bool setFile(const QString& fname, const KUrl&);
  virtual void clear();
  
  /** Saves the file */
  bool save(const QString &new_filename);

  /** Check if the give file is a valid PDF Document. */
  virtual bool isValidFile(const QString fileName);

  /** Rendering a page

      This implementation does almost nothing. First, it is checked if
      the arguments are in a reasonable range, and error messages are
      printed if this is not so. Secondly, the page is cleared and the
      background set to white. Nothing else is drawn.

      @param id The JobId describing the requested page.
  */
  virtual RenderedDocumentPagePixmap* drawPage(const JobId& id);

  virtual bool supportsTextSearch() const { return false; };

  /** Produce a PS file for a single page

      This method extracts a single page from the PS file, and stores
      in the file 'filename'. This file will then be a full PS file
      which can be rendered, printed, etc. This method will not work
      with EPS files.

      @param filename Name of the file that will be generated

      @page Number of the page that shall be extracted
      
      @returns true if everything worked fine, false otherwise

  */
  bool generatePSFileForPage(const QString &outfilename, QList<int> &pages)
  { return (internalDoc == 0) ? false : internalDoc->savePages(outfilename, pages); };

  /** Deletes pages from the document */
  void deletePages(quint16 from, quint16 to);

public slots:
  /** Returns 'true' is the document was modified */
  bool showInfo();

private:
  QString path;
  QString filename;

  // To allow file editing, the renderer uses a "working copy" of the
  // PDF file. When a file is loaded, the workingCopyFileName is the
  // same as the fileName. Whenever the file is modified, a working
  // copy is created, whose file name is specified here. That file is
  // then used instead of the original file.
  //
  // When the file is closed, the working copy (if different from the
  // original file) will be deleted.
  QString workingCopyFileName;
  RenderedDocumentPagePixmap* currentRenderingRequest;

  GSInternalDocument* internalDoc;

  PageNumber currentPageNumber;

  bool eps;
};

#endif
