// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-
//
// anchor.h
//
// Part of LIGATURE - A framework for multipage text/gfx viewers
//
// (C) 2004-2005 Stefan Kebekus
// Distributed under the GPL


#ifndef ANCHOR_H
#define ANCHOR_H

#include <QString>

#include "length.h"
#include "pageNumber.h"


/** \brief Page number and vertical position in physical coordinates

This very simple class contains a page number and a vertical position
in physical coordiantes. The vertical position is given by the
distance from the top of the page. Anchors are completely independent
of documents, there is no need for a document to exists that contains
the given page, nor does the page number need to be valid.

@author Stefan Kebekus <kebekus@kde.org>
@version 1.0 0
*/

class Anchor {
 public:

  /** \brief Constructs an anchor that points to an invalid page. The target is
      not a link. */
  Anchor()
    : isLinkTarget(false), isHistoryLink(false)
  {
  }

  /** \brief Constructs an anchor that points to the top of a given
      page

      The class contains no code to make sure in any way that the page
      number pg exists The distance_from_top is set to 0, the Anchor
      is not supposed to be a link.

       @param pg number of the page
   */
  Anchor(const PageNumber& pg)
    : page(pg), distance_from_top(), isLinkTarget(false), isHistoryLink(false)
  {
  }

  /** \brief Constructs an snchor that points to a given position on a
      given page

      The class contains no code to make sure in any way that the page
      number pg exists, and that page pg, if it exists, is taller than
      distance_from_top

      @param pg number of the page
      @param _distance_from_top distance from the top of the page
      @param _islink set to true if the anchor is the target of a link
  */
  Anchor(const PageNumber& pg, const Length& _distance_from_top, bool _isLink=true)
    : page(pg), distance_from_top(_distance_from_top), isLinkTarget(_isLink), isHistoryLink(false)
  {}

  /** \brief quick validity check for anchors

  @returns true if the page number is valid, and 0mm <= distance_from_top <= 2m
  */
  bool isValid() const
  {
    return page.isValid() && (0.0 <= distance_from_top.getLength_in_mm()) &&  (distance_from_top.getLength_in_mm() <= 2000.0);
  }

  bool operator== (const Anchor& anchor) const
  {
    return page == anchor.page && distance_from_top.isNearlyEqual(anchor.distance_from_top);
  }

  /** \brief Page number that this anchor point to */
  PageNumber page;

  /** \brief Distance from the top of the page in inch */
  Length   distance_from_top;

  /** \brief  indicates if the anchor is the target of a link

      This flag is used by the GUI. For instance, if the user goes to
      the target of a link, there is a flashing frame that indicates
      where the anchor is.
    */
  bool isLinkTarget;

  /** \brief indicates that the anchor comes from the navigation history */
  bool isHistoryLink;

  /** \brief This method implements typecasts to QString */
  operator QString() const
  {
    return QString("(page=%1, %2 from top, isLink=%3)").arg(page.toQString()).arg(distance_from_top).arg(isLinkTarget);
  }
};


#endif
