// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-
//
// Class: DocumentWidget
//
// Widget for displaying TeX DVI files.
// Part of KDVI- A previewer for TeX DVI files.
//
// (C) 2001 Stefan Kebekus
// Copyright (C) 2004-2006 Wilfried Huss <Wilfried.Huss@gmx.at>
// Distributed under the GPL
//

#include <config.h>

#include "documentWidget.h"
#include "documentPageCache.h"
#include "hyperlink.h"
#include "kvs_debug.h"
#include "kvsprefs.h"
#include "pageView.h"
#include "textBox.h"

#include <kglobalsettings.h>
#include <kiconloader.h>
#include <klocale.h>

#include <QAction>
#include <QApplication>
#include <QClipboard>
#include <QMenu>
#include <QPaintEvent>

//#define DEBUG_DOCUMENTWIDGET

const int DocumentWidget::bottom_right_corner[16] =
  { 61, 71, 85, 95,
    71, 78, 89, 96,
    85, 89, 95, 98,
    95, 97, 98, 99 };

const int DocumentWidget::bottom_left_corner[16] =
  { 95, 85, 71, 61,
    97, 89, 78, 71,
    98, 95, 89, 85,
    99, 98, 96, 95 };

const int DocumentWidget::shadow_strip[4] =
  { 56, 67, 83, 94 };

QColor DocumentWidget::backgroundColorForCorners;

QPointer<QMenu> DocumentWidget::contextMenu;

QPointer<QAction> DocumentWidget::addBookmarkAction;
QPointer<QAction> DocumentWidget::removeBookmarkAction;

namespace {

/** Holds the icon used as a overlay on pages which are not drawn yet. */
QPixmap* busyIcon = 0;

/** Internal storages used in the shadow drawing routines in the
    drawContents method.*/
QPixmap* URShadow = 0;
QPixmap* BRShadow = 0;
QPixmap* BLShadow = 0;

} // namespace anon


DocumentWidget::DocumentWidget(PageView *sv, DocumentPageCache *cache)
  : QWidget(sv->viewport()), indexOfUnderlinedLink(-1)
{
  selectionNeedsUpdating = false;

  measuringInProgress = false;

  // Variables used in animation.
  animationCounter = 0;
  timerIdent       = 0;
  documentCache    = cache;
  scrollView       = sv;

  scrollGuide = -1;

  setMouseTracking(true);
  setFocusPolicy(Qt::ClickFocus);

  connect(&clearStatusBarTimer, SIGNAL(timeout()), this, SLOT(clearStatusBar()));
  connect(&clearScrollGuideTimer, SIGNAL(timeout()), this, SLOT(clearScrollGuide()));

  connect(this, SIGNAL(showPopupMenu(const PageNumber&, const QPoint&)),
          this, SLOT(slotShowPopupMenu(const PageNumber&, const QPoint&)));

  setAttribute(Qt::WA_NoSystemBackground, true);

  if (!busyIcon)
  {
    busyIcon = new QPixmap(KIconLoader::global()->loadIcon("gear", K3Icon::NoGroup, K3Icon::SizeMedium));

    URShadow = new QPixmap(4, 4);
    BRShadow = new QPixmap(4, 4);
    BLShadow = new QPixmap(4, 4);
  }
}


void DocumentWidget::setPageNumber(PageNumber nr)
{
  pageNumber = nr;

  selectionNeedsUpdating = true;

  // We have to reset this, because otherwise we might crash in the mouseMoveEvent
  // After switching pages in SinglePageMode or OverviewMode.
  indexOfUnderlinedLink = -1;

  // Resize Widget if the size of the new page is different than the size of the old page.
  QSize _pageSize;
  if (dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Landscape ||
      dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Seascape)
  {
    QSize temp = documentCache->sizeOfPageInPixel(pageNumber);
    _pageSize = QSize(temp.height(), temp.width());
  }
  else
  {
    _pageSize = documentCache->sizeOfPageInPixel(pageNumber);
  }
  if (_pageSize != pageSize())
  {
    setPageSize(_pageSize);
  }
  update();
}


QRect DocumentWidget::linkFlashRect()
{
  if (dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Landscape ||
      dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Seascape)
  {
    int width = pageSize().height()/(11 - animationCounter);
    int height = pageSize().width()/(60 - 4 * animationCounter);
    return QRect((pageSize().height()-width)/2, flashOffset - height/2, width, height);
  }
  else
  {
    int width = pageSize().width()/(11 - animationCounter);
    int height = pageSize().height()/(60 - 4 * animationCounter);
    return QRect((pageSize().width()-width)/2, flashOffset - height/2, width, height);
  }
}


void DocumentWidget::timerEvent( QTimerEvent *e )
{
  if (animationCounter == 0) {
    killTimer(e->timerId());
    timerIdent = startTimer(50); // Proceed with the animation in 1/10s intervals
  }

  animationCounter++;

  QRect flashRect = linkFlashRect();
  flashRect.adjust(-1, -1, 1, 1);

  if (animationCounter >= 10) {
    killTimer(e->timerId());
    timerIdent       = 0;
    animationCounter = 0;
  }

  repaint(map(flashRect));
}


void DocumentWidget::flash(int fo)
{
  if (timerIdent != 0)
  {
    killTimer(timerIdent);
    // Delete old flash rectangle
    animationCounter = 10;
    QRect flashRect = linkFlashRect();
    flashRect.adjust(-1, -1, 1, 1);
    repaint(flashRect);
  }
  animationCounter = 0;
  flashOffset      = fo;
  timerIdent       = startTimer(50); // Start the animation. The animation proceeds in 1/10s intervals
}


void DocumentWidget::paintEvent(QPaintEvent *e)
{
#ifdef DEBUG_DOCUMENTWIDGET
  kDebug(kvs::shell) << "DocumentWidget::paintEvent() called" << endl;
#endif

  // Check if this widget is really visible to the user. If not, there
  // is nothing to do. Remark: if we don't do this, then under QT
  // 3.2.3 the following happens: when the user changes the zoom
  // value, all those widgets are updated which the user has EVER
  // seen, not just those that are visible at the moment. If the
  // document contains several thousand pages, it is easily possible
  // that this means that a few hundred of these are re-painted (which
  // takes substantial time) although perhaps only three widgets are
  // visible and *should* be updated. I believe this is some error in
  // QT, but I am not positive about that ---Stefan Kebekus.
  if (!scrollView->isVisible(this))
  {
    //kDebug(kvs::shell) << "widget of page " << pageNumber << " is not visible. Abort rendering" << endl;
    return;
  }

  QPainter p(this);
  p.setClipRegion(e->region());

  // Paint a black border around the widget
  //p.setRasterOp(Qt::CopyROP);
  p.setBrush(Qt::NoBrush);
  p.setPen(Qt::black);
  QRect outlineRect = pageRect();
  outlineRect.adjust(-1, -1, 0, 0);
  p.drawRect(outlineRect);

  // Paint page shadow
  QColor backgroundColor = palette().color(QPalette::Mid);

  // (Re-)generate the Pixmaps for the shadow corners, if necessary
  if (backgroundColor != backgroundColorForCorners)
  {
    backgroundColorForCorners = backgroundColor;
    QImage tmp(4, 4, QImage::Format_RGB32);
    for(int x=0; x<4; x++)
      for(int y=0; y<4; y++)
        tmp.setPixel(x, y, backgroundColor.light(bottom_right_corner[x+4*y]).rgb() );

    //FIXME make BRShadow a QPixmap if possible
    QPixmap tmpPixmap = QPixmap::fromImage(tmp);
    BRShadow = new QPixmap(tmpPixmap);

    for(int x=0; x<4; x++)
      for(int y=0; y<4; y++)
        tmp.setPixel(x, y, backgroundColor.light(bottom_left_corner[x+4*y]).rgb() );

    //FIXME make BLShadow a QPixmap if possible
    QPixmap tmpPixmap2 = QPixmap::fromImage(tmp);
    BLShadow = new QPixmap(tmpPixmap2);

    //FIXME make URShadow a QPixmap if possible
    QPixmap tmpPixmap3 = QPixmap::fromImage(tmp.mirrored(true, true));
    URShadow = new QPixmap(tmpPixmap3);
  }

  // Draw right and bottom shadows
  for(int i=0; i<4; i++)
  {
    p.setPen(backgroundColor.light(shadow_strip[i]));
    // Right shadow
    p.drawLine(pageSize().width()+i+2, 8, pageSize().width()+i+2, pageSize().height()+2);
    // Bottom shadow
    p.drawLine(8, pageSize().height()+i+2, pageSize().width()+2, pageSize().height()+i+2);
  }
  // Draw shadow corners
  p.drawPixmap(pageSize().width()+2, pageSize().height()+2, *BRShadow);
  p.drawPixmap(4, pageSize().height()+2, *BLShadow);
  p.drawPixmap(pageSize().width()+2, 4, *URShadow);

  // Draw corners
  p.fillRect(0, pageSize().height()+2, 4, 4, backgroundColor);
  p.fillRect(pageSize().width()+2, 0, 4, 4, backgroundColor);

  if (!documentCache->isPageCached(pageNumber))
  {
    QRect destRect = e->rect().intersect(pageRect());
    p.fillRect(destRect, dataModel->paperColor());

    // Draw busy indicator.
    // Im not really sure if this is a good idea.
    // While it is nice to see an indication that something is happening for pages which
    // take long to redraw, it gets quite annoing for fast redraws.
    // TODO: Disable or find something less distractiong.
    p.drawPixmap(10, 10, *busyIcon);

    // Request page pixmap.
    documentCache->getPage(pageNumber);
    return;
  }

  RenderedDocumentPagePixmap *pageData = documentCache->getPage(pageNumber);
  if (pageData == 0) {
#ifdef DEBUG_DOCUMENTWIDGET
    kDebug(kvs::shell) << "DocumentWidget::paintEvent: no documentPage generated" << endl;
#endif
    return;
  }

  QVector<QRect> damagedRects = e->region().rects();
  for (int i = 0; i < damagedRects.count(); i++)
  {
    // FIXME: The overlay drawing bug where hyperlinks and selections
    // where overwritten by the bitblt when the damaged region was not
    // rectangular shows up again, but only in very few cases.

    // Paint the page where it intersects with the damaged area.
    QRect destRect = damagedRects[i].intersect(pageRect());

    // The actual page starts at point (1,1) because of the outline.
    // Therefore we need to shift the destination rectangle.
    QRect pixmapRect = destRect;
    pixmapRect.translate(-1, -1);

    // Paint widget contents
    QPixmap const pixmap = pageData->pixmap();
    if (!pixmap.isNull())
    {
      p.drawPixmap(destRect.topLeft(), pixmap, pixmapRect);
    }
    else
    {
      // If the renderer has returned a null image for some reason. Fill
      // the widget with the background color.
      p.fillRect(destRect, dataModel->paperColor());
    }
  }

  p.save();

  p.setMatrix(transform());

  // Underline hyperlinks
  if (pageData->hyperlinks_are_marked == false) // If hyperlinks are marked already, we don't underline them
    if (dataModel->preferences()->underlineLinks() == KVSPrefs::EnumUnderlineLinks::Enabled ||
        dataModel->preferences()->underlineLinks() == KVSPrefs::EnumUnderlineLinks::OnlyOnHover) {
      int h = 2; // Height of line.
      for(int i = 0; i < (int)pageData->hyperLinkList.size(); i++) {
        if (dataModel->preferences()->underlineLinks() == KVSPrefs::EnumUnderlineLinks::OnlyOnHover &&
            i != indexOfUnderlinedLink)
          continue;
        int x = pageData->hyperLinkList[i].box.left();
        int w = pageData->hyperLinkList[i].box.width();
        int y = pageData->hyperLinkList[i].baseline;
        
        QRect hyperLinkRect(x, y, w, h);
        //if (hyperLinkRect.intersects(inverseMap(e->rect()))) //FIXME: need to transform events
        {
#ifdef DEBUG_DOCUMENTWIDGET
          kDebug(kvs::shell) << "Underline hyperlink \"" << pageData->hyperLinkList[i].linkText << "\"" << endl;
#endif
          p.fillRect(x, y, w, h, KGlobalSettings::linkColor());
        }
      }
    }
  
  // Paint flashing frame, if appropriate
  if (animationCounter > 0 && animationCounter < 10)
    {
    int gbChannel = 255 - (9-animationCounter)*28;
    p.setPen(QPen(QColor(255, gbChannel, gbChannel), 3));
    p.drawRect(linkFlashRect());
  }

  p.restore();

  // For the textselection we don't use the transformed painter, because
  // Qt seems to have problems with painting in XOR-mode when using coordinate
  // transforms. Therefore we transform the selected region first, before we
  // split it into rectangles.

  // Mark selected text.
  TextSelection selection = dataModel->selectedText();
  if ((selection.getPageNumber() != 0) && (selection.getPageNumber() == pageNumber))
  {
    if (selectionNeedsUpdating)
    {
      //The zoom value has changed, therefore we need to recalculate
      //the selected region.
      selectedRegion = pageData->selectedRegion(selection);
      selectionNeedsUpdating = false;
    }

    p.setPen(Qt::NoPen);
    //p.setBrush(Qt::white);
    //p.setRasterOp(Qt::XorROP);

    // FIXME: Draw selection using a semitransparent blue color.
    // Its broken, but still better than not seeing the selection at all.
    p.setBrush(QColor(0, 0, 255, 128));

    QVector<QRect> selectionRects = map(selectedRegion).rects();

    for (int i = 0; i < selectionRects.count(); i++)
      p.drawRect(selectionRects[i]);
  }

  // Draw scroll Guide
  if (scrollGuide >= 0)
  {
    // Don't draw over the page shadow
    p.setClipRegion(e->region().intersect(pageRect()));
    //p.setRasterOp(Qt::CopyROP);
    p.setPen(Qt::red);
    p.drawLine(1, scrollGuide, pageSize().width(), scrollGuide);
  }

  if (dataModel->preferences()->tool() == KVSPrefs::EnumTool::Ruler && measuringInProgress)
  {
    // Don't draw over the page shadow
    p.setClipRegion(e->region().intersect(pageRect()));
    p.setPen(Qt::red);

    p.drawLine(measuringOrigin.x(), measuringOrigin.y()-6, measuringOrigin.x(), measuringOrigin.y()+6);
    p.drawLine(measuringOrigin.x()-6, measuringOrigin.y(), measuringOrigin.x()+6, measuringOrigin.y());
    p.drawLine(measuringOrigin, measuringPoint);

    // Restore clipping region, so we don't forget it in the future.
    p.setClipRegion(e->region());
  }

  if (dataModel->preferences()->tool() == KVSPrefs::EnumTool::RectSelect && rectSelectionStartPoint != rectSelectionEndPoint)
  {
    QRect selectedRect(rectSelectionStartPoint, rectSelectionEndPoint);
    selectedRect = selectedRect.normalized();
    // don't overdraw the page border
    selectedRect = selectedRect.intersect(pageRect());

    p.setPen(Qt::red);
    p.drawRect(selectedRect);
  }
}

void DocumentWidget::drawScrollGuide(int ycoord)
{
  if (clearScrollGuideTimer.isActive())
  {
    clearScrollGuideTimer.stop();
    clearScrollGuide();
  }
  scrollGuide = ycoord;
  update(QRect(1, scrollGuide, pageSize().width(), 1));
  clearScrollGuideTimer.setSingleShot(true);
  clearScrollGuideTimer.start(1000);
}

void DocumentWidget::clearScrollGuide()
{
  int temp = scrollGuide;
  scrollGuide = -1;
  update(QRect(1, temp, pageSize().width(), 1));
}

void DocumentWidget::select(const TextSelection& newSelection)
{
  // Get a pointer to the page contents
  RenderedDocumentPage* pageData = documentCache->getPage(pageNumber, false);
  if (pageData == 0) {
    kDebug(kvs::shell) << "DocumentWidget::select() pageData for page #" << pageNumber << " is empty" << endl;
    return;
  }

  dataModel->selectText(newSelection);

  selectedRegion = pageData->selectedRegion(dataModel->selectedText());
  selectionNeedsUpdating = false;

  update();
}

void DocumentWidget::selectAll()
{
  if (!pageNumber.isValid())
    return;

  // Get a pointer to the page contents
  RenderedDocumentPage* pageData = documentCache->getPage(pageNumber, false);
  if (pageData == 0) {
    kDebug(kvs::shell) << "DocumentWidget::selectAll() pageData for page #" << pageNumber << " is empty" << endl;
    return;
  }

  TextSelection selection;
  // mark everything as selected
  QString selectedText("");
  for(int i = 0; i < pageData->textBoxList.size(); i++) {
    selectedText += pageData->textBoxList[i].text;
    selectedText += "\n";
  }
  selection.set(pageNumber, 0, pageData->textBoxList.size()-1, selectedText);

  selectedRegion = pageData->selectedRegion(selection);

  dataModel->selectText(selection);

  // Re-paint
  update();
}


void DocumentWidget::mousePressEvent ( QMouseEvent * e )
{
#ifdef DEBUG_DOCUMENTWIDGET
  kDebug(kvs::shell) << "DocumentWidget::mousePressEvent(...) called" << endl;
#endif

  // Make sure the event is passed on to the higher-level widget;
  // otherwise QT gets the coordinates in the mouse move events wrong
  e->ignore();

  // Safety check
  if (!pageNumber.isValid())
    return;

  if (e->button() == Qt::RightButton)
  {
    emit showPopupMenu(pageNumber, e->globalPos());
    return;
  }

  // Get a pointer to the page contents
  RenderedDocumentPage* pageData = documentCache->getPage(pageNumber);
  if (pageData == 0) {
    kDebug(kvs::shell) << "DocumentWidget::selectAll() pageData for page #" << pageNumber << " is empty" << endl;
    return;
  }

  if (dataModel->preferences()->tool() == KVSPrefs::EnumTool::Ruler)
  {
    if (e->button() == Qt::LeftButton)
    {
      measuringInProgress = true;
      measuringOrigin = e->pos();
      measuringPoint = measuringOrigin;
      oldMeasuringPoint = measuringPoint;
      update();
    }
    return;
  }

  if (dataModel->preferences()->tool() == KVSPrefs::EnumTool::RectSelect)
  {
    if (e->button() == Qt::LeftButton)
    {
      rectSelectionStartPoint = e->pos();
      rectSelectionEndPoint = e->pos();
    }
    return;
  }

  // Check if the mouse is pressed on a regular hyperlink
  if (e->button() == Qt::LeftButton)
  {
    if (pageData->hyperLinkList.size() > 0)
    {
      for(int i = 0; i < pageData->hyperLinkList.size(); i++)
      {
        if (pageData->hyperLinkList[i].box.contains(inverseMap(e->pos())))
        {
#ifdef DEBUG_DOCUMENTWIDGET
          kDebug(kvs::shell) << "DocumentWidget::mousePressEvent(): mouse pressed on hyperlink " << pageData->hyperLinkList[i].linkText << endl;
#endif
          emit localLink(pageData->hyperLinkList[i]);
          return;
        }
      }
    }

    switch(dataModel->preferences()->tool())
    {
      case KVSPrefs::EnumTool::Move:
        setCursor(Qt::SizeAllCursor);
        break;
      case KVSPrefs::EnumTool::Select:
        setCursor(Qt::IBeamCursor);
        break;
      case KVSPrefs::EnumTool::RectSelect:
        setCursor(Qt::CrossCursor);
        break;
      case KVSPrefs::EnumTool::Ruler:
        setCursor(Qt::CrossCursor);
        break;
      default:
        setCursor(Qt::ArrowCursor);
    }
  }

  if (dataModel->preferences()->tool() == KVSPrefs::EnumTool::Select && e->button() == Qt::LeftButton)
  {
    setCursor(Qt::IBeamCursor);
    // If Shift is not pressed clear the current selection,
    // otherwise modify the existing selection.
    if (!(e->modifiers() & Qt::ShiftModifier))
    {
      firstSelectedPoint = inverseMap(e->pos());
      selectedRectangle = QRect();
      selectedRegion = QRegion();
      emit clearSelection();
    }
  }
}


void DocumentWidget::mouseReleaseEvent ( QMouseEvent *e )
{
  // Make sure the event is passed on to the higher-level widget;
  // otherwise the mouse cursor in the centeringScrollview is wrong
  e->ignore();

  if (dataModel->preferences()->tool() == KVSPrefs::EnumTool::Ruler && e->button() == Qt::LeftButton)
  {
    measuringInProgress = false;
    update();
    return;
  }

  if (dataModel->preferences()->tool() == KVSPrefs::EnumTool::RectSelect && e->button() == Qt::LeftButton)
  {
    // Compute the selected rectangle
    QRect selectedRect(rectSelectionStartPoint, rectSelectionEndPoint);
    selectedRect = selectedRect.normalized();

    // only show the menu, when the selected rectangle is not empty.
    if (selectedRect.width() > 4 && selectedRect.height() > 4)
      performRectangleSelection(e->globalPos(), selectedRect);

    selectedRect.adjust(0, 0, 1, 1);
    update(selectedRect);

    rectSelectionStartPoint = QPoint();
    rectSelectionEndPoint = QPoint();
    oldRectSelectionEndPoint = QPoint();
    return;
  }

  if (dataModel->preferences()->tool() == KVSPrefs::EnumTool::Select && e->button() == Qt::LeftButton)
  {
    // If the selectedRectangle is empty then there was only a single right click.
    if (firstSelectedPoint == inverseMap(e->pos()))
    {
      if (!pageNumber.isValid())
        return;

      // Get a pointer to the page contents
      RenderedDocumentPage* pageData = documentCache->getPage(pageNumber);
      if (pageData == 0)
      {
        kDebug(kvs::shell) << "DocumentWidget::mouseReleaseEvent() pageData for page #" << pageNumber << " is empty" << endl;
        return;
      }

      TextSelection newTextSelection = pageData->select(firstSelectedPoint);
      updateSelection(newTextSelection);
    }
  }

  //Reset the cursor to the usual arrow if we use the move tool, and
  // The textselection cursor if we use the selection tool.
  setStandardCursor();
}


void DocumentWidget::mouseDoubleClickEvent(QMouseEvent* e)
{
  if (dataModel->preferences()->viewMode() == KVSPrefs::EnumViewMode::Overview)
  {
    kdDebug() << "double click on page " << pageNumber << endl;
    PageNumber currentPageNumber = pageNumber;
    dataModel->setViewMode(KVSPrefs::EnumViewMode::Continuous);
    dataModel->setCurrentPageNumber(Anchor(currentPageNumber));
  }
}


void DocumentWidget::measureLength(bool restrictAxis)
{
  QPoint delta = measuringOrigin - measuringPoint;

  // Restricts the measurment to a
  // strictly vertical or horizontal line.
  if (restrictAxis)
  {
    if (std::abs(double(delta.x())) < std::abs(double(delta.y())))
      measuringPoint.setX(measuringOrigin.x());
    else
      measuringPoint.setY(measuringOrigin.y());

    delta = measuringOrigin - measuringPoint;
  }

  if (measuringPoint != oldMeasuringPoint)
  {
    QRect damagedRect(measuringOrigin, oldMeasuringPoint);
    damagedRect = damagedRect.normalized();
    damagedRect.adjust(-6, -6, 6, 6);

    oldMeasuringPoint = measuringPoint;

    update(damagedRect);
  }

  const int length = int(std::sqrt(double(delta.x()*delta.x() + delta.y()*delta.y())));
  QString sl = printLength(length);

  double angle = 90.0;
  if (delta.x() != 0)
    angle = atan(double(std::abs(double(delta.y()))/
                        std::abs(double(delta.x())))) * 180 / 3.14;

  emit setStatusBarText(ki18n("Length = %1, Angle = %2").subs(sl).subs(angle, 0, 'f', 2).toString());
}


void DocumentWidget::mouseMoveEvent ( QMouseEvent * e )
{
#ifdef DEBUG_DOCUMENTWIDGET
  kDebug(kvs::shell) << "DocumentWidget::mouseMoveEvent(...) called" << endl;
#endif

  // Left mouse button pressed -> Text scroll function. We implement
  // this functionality already here so that it works in all possible
  // circumstances, e.g. if the pageData is empty, or if the page
  // doesn't have a page number yet.
  if ((e->buttons() & Qt::LeftButton) && dataModel->preferences()->tool() == KVSPrefs::EnumTool::Move) {
    // Pass the mouse event on to the owner of this widget ---under
    // normal circumstances that is the centeringScrollView which will
    // then scroll the scrollview contents
    e->ignore();
  }

  // Safety check
  if (!pageNumber.isValid())
    return;

  // Get a pointer to the page contents
  RenderedDocumentPage *pageData = documentCache->getPage(pageNumber);
  if (pageData == 0) {
    kDebug(kvs::shell) << "DocumentWidget::mouseMoveEvent() pageData for page #" << pageNumber << " is empty" << endl;

    return;
  }

  if (dataModel->preferences()->tool() == KVSPrefs::EnumTool::Ruler)
  {
    setStandardCursor();

    // If no mouse button is pressed, we show the current (x,y)-coordinates in the status bar
    if (e->buttons() == Qt::NoButton)
    {
      QString sx = printLength(e->pos().x());
      QString sy = printLength(e->pos().y());
      emit setStatusBarText(i18n("x = %1, y = %2", sx, sy));
    }
    else if (e->buttons() & Qt::LeftButton)
    {
      measuringPoint = e->pos();
      // The control key restricts the measurment to a
      // strictly vertical or horizontal line.
      bool control = e->modifiers() & Qt::ControlModifier;
      measureLength(control);
    }
    return;
  }

  if (dataModel->preferences()->tool() == KVSPrefs::EnumTool::RectSelect)
  {
    setStandardCursor();

    if (e->buttons() & Qt::LeftButton)
    {
      oldRectSelectionEndPoint = rectSelectionEndPoint;
      rectSelectionEndPoint = e->pos();

      // Compute the damaged region
      QRect newRect(rectSelectionStartPoint, rectSelectionEndPoint);
      QRect oldRect(rectSelectionStartPoint, oldRectSelectionEndPoint);
      QRegion damagedRegion = QRegion(newRect.normalized()) + oldRect.normalized();
      QRect damagedRect = damagedRegion.boundingRect();
      damagedRect.adjust(0, 0, 1, 1);
      update(damagedRect);
    }
    return;
  }

  // If no mouse button pressed
  if (e->buttons() == Qt::NoButton)
  {
    // Remember the index of the underlined link.
    int lastUnderlinedLink = indexOfUnderlinedLink;
    // go through hyperlinks
    for (int i = 0; i < pageData->hyperLinkList.size(); i++)
    {
      if (pageData->hyperLinkList[i].box.contains(inverseMap(e->pos())))
      {
        clearStatusBarTimer.stop();
        setCursor(Qt::PointingHandCursor);
        QString link = pageData->hyperLinkList[i].linkText;
        if ( link.startsWith("#") )
          link = link.remove(0,1);

        emit setStatusBarText( i18n("Link to %1", link) );

        // Prerender the page the hyperlink links to, because
        // there is the chance that we will jump there real soon.
        Anchor anchor = pageData->hyperLinkList[i].anchor;
        if (anchor.isValid())
        {
          documentCache->getPage(anchor.page);
        }

        indexOfUnderlinedLink = i;
        if (dataModel->preferences()->underlineLinks() == KVSPrefs::EnumUnderlineLinks::OnlyOnHover &&
            indexOfUnderlinedLink != lastUnderlinedLink)
        {
          QRect newUnderline = pageData->hyperLinkList[i].box;
          // Increase Rectangle so that the whole line really lines in it.
          newUnderline.adjust(0, 0, 0, 2);
          // Redraw widget
          update(map(newUnderline));

          if (lastUnderlinedLink != -1 && lastUnderlinedLink < pageData->hyperLinkList.size())
          {
            // Erase old underline
            QRect oldUnderline = pageData->hyperLinkList[lastUnderlinedLink].box;
            oldUnderline.adjust(0, 0, 0, 2);
            update(map(oldUnderline));
          }
        }
        return;
      }
    }
    // Whenever we reach this the mouse hovers no link.
    indexOfUnderlinedLink = -1;
    if (dataModel->preferences()->underlineLinks() == KVSPrefs::EnumUnderlineLinks::OnlyOnHover &&
        lastUnderlinedLink != -1 && lastUnderlinedLink < pageData->hyperLinkList.size())
    {
      // Erase old underline
      QRect oldUnderline = pageData->hyperLinkList[lastUnderlinedLink].box;
      // Increase Rectangle so that the whole line really lines in it.
      oldUnderline.adjust(0, 0, 0, 2);
      // Redraw widget
      update(map(oldUnderline));
    }
    // Cursor not over hyperlink? Then let the cursor be the usual arrow if we use the move tool, and
    // The textselection cursor if we use the selection tool.
    setStandardCursor();
  }

  if (!clearStatusBarTimer.isActive())
  {
    clearStatusBarTimer.setSingleShot(true);
    clearStatusBarTimer.start(200); // clear the statusbar after 200 msec.
  }

  // Text selection works in the text tool with the left mouse button.
  if (dataModel->preferences()->tool() == KVSPrefs::EnumTool::Select && (e->buttons() & Qt::LeftButton))
  {
    QPoint currentPoint = inverseMap(e->pos());

    int lx = currentPoint.x() < firstSelectedPoint.x() ? currentPoint.x() : firstSelectedPoint.x();
    int rx = currentPoint.x() > firstSelectedPoint.x() ? currentPoint.x() : firstSelectedPoint.x();
    int ty = currentPoint.y() < firstSelectedPoint.y() ? currentPoint.y() : firstSelectedPoint.y();
    int by = currentPoint.y() > firstSelectedPoint.y() ? currentPoint.y() : firstSelectedPoint.y();
    selectedRectangle.setCoords(lx,ty,rx,by);

    // Select the right mode for calculating the selected region
    const bool normal = ((currentPoint.x() > firstSelectedPoint.x()) ^
                         (currentPoint.y() < firstSelectedPoint.y()));

    // Now that we know the rectangle, we have to find out which words
    // intersect it!
    TextSelection newTextSelection = pageData->select(selectedRectangle, normal);

    updateSelection(newTextSelection);
  }
}


void DocumentWidget::updateSelection(const TextSelection& newTextSelection)
{
  if (newTextSelection != dataModel->selectedText())
  {
    if (newTextSelection.isEmpty())
    {
      // Clear selection
      dataModel->deselectText();
      selectedRectangle = QRect();
      selectedRegion = QRegion();
      update();
    }
    else
    {
      if (!pageNumber.isValid())
        return;

      // Get a pointer to the page contents
      RenderedDocumentPage* pageData = documentCache->getPage(pageNumber, false);
      if (pageData == 0)
      {
        kDebug(kvs::shell) << "DocumentWidget::mouseReleaseEvent() pageData for page #" << pageNumber << " is empty" << endl;
        return;
      }

      dataModel->selectText(newTextSelection);

      QRegion newlySelectedRegion = pageData->selectedRegion(dataModel->selectedText());

      // Compute the region that needs to be updated
      QRegion updateRegion;
      if(!selectedRegion.isEmpty())
      {
        updateRegion = newlySelectedRegion.eor(selectedRegion);
      }
      else
      {
        updateRegion = newlySelectedRegion;
      }

      selectedRegion = newlySelectedRegion;

      QVector<QRect> rectangles = updateRegion.rects();
      for (int i = 0; i < rectangles.count(); i++)
      {
        repaint(map(rectangles[i]));
      }
    }
  }
}


void DocumentWidget::clearStatusBar()
{
  emit setStatusBarText( QString::null );
}


QSize DocumentWidget::pageSize() const
{
  // Substract size of the shadow.
  return size() - QSize(6, 6);
}

QRect DocumentWidget::pageRect() const
{
  QRect boundingRect = rect();
  // Substract the shadow.
  boundingRect.adjust(1, 1, -5, -5);
  return boundingRect;
}

void DocumentWidget::setPageSize(const QSize& pageSize)
{
  // When the page size changes this means either the paper size
  // has been changed, or the zoomlevel has been changed.
  // If the zoomlevel changes we need to recalculate the selected
  // region. We do this always, just to be on the safe side.
  selectionNeedsUpdating = true;

  // Add size of the shadow.
  resize(pageSize + QSize(6,6));
}

void DocumentWidget::setPageSize(int width, int height)
{
  setPageSize(QSize(width, height));
}


void DocumentWidget::slotSwitchTool()
{
  setStandardCursor();
}


void DocumentWidget::setStandardCursor()
{
  switch(dataModel->preferences()->tool()) {
    case KVSPrefs::EnumTool::Move:
      setCursor(Qt::ArrowCursor);
      break;
    case KVSPrefs::EnumTool::Select:
      setCursor(Qt::IBeamCursor);
      break;
    case KVSPrefs::EnumTool::RectSelect:
      setCursor(Qt::CrossCursor);
      break;
    case KVSPrefs::EnumTool::Ruler:
      setCursor(Qt::CrossCursor);
      break;
    default:
      setCursor(Qt::ArrowCursor);
  }
}


QString DocumentWidget::printLength(int l) const
{
  double res = dataModel->resolution();

  Length px;
  px.setLength_in_pixel(l, res);

  double x;

  QString unit;
  QString result;

  switch (dataModel->preferences()->unit())
  {
    case KVSPrefs::EnumUnit::pixel:
      unit = i18n("px");
      result = QString("%1px").arg(l);
      break;
    case KVSPrefs::EnumUnit::inch:
      unit = i18n("inch");
      x = px.getLength_in_inch();
      result = QString("%1inch").arg(x, 0, 'f', 1);
      break;
    case KVSPrefs::EnumUnit::TeXPoint:
      unit = i18n("TeXPoint");
      x = px.getLength_in_TeXPoints();
      result = QString("%1 TeX point").arg(x, 0, 'f', 1);
      break;
    case KVSPrefs::EnumUnit::bigPoint:
      unit = i18n("bigPoint");
      x = px.getLength_in_bigPoints();
      result = QString("%1 big point").arg(x, 0, 'f', 1);
      break;
    case KVSPrefs::EnumUnit::pica:
      unit = i18n("pica");
      x = px.getLength_in_pica();
      result = QString("%1pica").arg(x, 0, 'f', 1);
      break;
    case KVSPrefs::EnumUnit::didot:
      unit = i18n("didot");
      x = px.getLength_in_didot();
      result = QString("%1didot").arg(x, 0, 'f', 1);
      break;
    case KVSPrefs::EnumUnit::cicero:
      unit = i18n("cicero");
      x = px.getLength_in_cicero();
      result = QString("%1cicero").arg(x, 0, 'f', 1);
      break;
    default:
      unit = i18n("cm");
      x = px.getLength_in_cm();
      result = QString("%1cm").arg(x, 0, 'f', 1);
  }
  return result;
}


SelectedRectangle::SelectedRectangle(const QRect& rectangle,
                                     const QString & text,
                                     RenderedDocumentPagePixmap& page_data)
  : rectangle_(rectangle), text_(text), page_data_(page_data)
{}


void SelectedRectangle::copyText() const
{
  QApplication::clipboard()->setText(text_, QClipboard::Clipboard);
}


void SelectedRectangle::copyImage()
{
  // Access the correct pixmap of the current page
  QPixmap pixmap;
  //FIXME
  //if (dataModel->preferences()->changeColors() &&
  //    dataModel->preferences()->renderMode() != KVSPrefs::EnumRenderMode::Paper)
  //  pixmap = page_data_.accessiblePixmap();
  //else
    pixmap = page_data_.pixmap();

  // Paint the selected part of the page to a new pixmap
  QPixmap copyPixmap(rectangle_.width(), rectangle_.height());
  QPainter p(&copyPixmap);
  p.drawPixmap(QPoint(0, 0), pixmap, rectangle_);
  p.end();

  // Copy pixmap to the clipboard
  QApplication::clipboard()->setPixmap(copyPixmap, QClipboard::Clipboard);
  emit setStatusBarText(i18n("Image copied to clipboard"));
}


void DocumentWidget::performRectangleSelection(const QPoint& menuPos, const QRect& selectedRect)
{
  // Get a pointer to the page contents.
  // We use the syncron version to ensure that we really always get the data
  RenderedDocumentPagePixmap* pageData = documentCache->getPage(pageNumber, false);
  if (pageData == 0)
  {
    kDebug(kvs::shell) << "DocumentWidget::performRectangleSelection() pageData for page #"
                           << pageNumber << " is empty" << endl;
    return;
  }

  // compute selected text
  QString selectedText = pageData->rectangleSelect(inverseMap(selectedRect));

  SelectedRectangle sel_rect(selectedRect, selectedText, *pageData);
  connect(&sel_rect, SIGNAL(setStatusBarText(const QString&)),
          this, SIGNAL(setStatusBarText(const QString&)));

  // Create context menu
  QMenu menu(this);
  if (!selectedText.isEmpty())
  {
    QAction* action = menu.addAction(SmallIcon("editcopy"),
                                     i18np("Copy text to Clipboard (1 character)",
                                          "Copy text to Clipboard (%1 characters)",
                                          selectedText.length()));
    connect(action, SIGNAL(triggered()), &sel_rect, SLOT(copyText()));
  }
  QAction* action = menu.addAction(SmallIcon("image"),
                                   i18n("Copy image to Clipboard (%1 by %2 pixels)", selectedRect.width(), selectedRect.height()));
  connect(action, SIGNAL(triggered()), &sel_rect, SLOT(copyImage()));

  // Show context menu
  menu.exec(menuPos);
}


void DocumentWidget::slotShowPopupMenu(const PageNumber& pageNumber, const QPoint& position)
{
  if (contextMenu.isNull())
  {
    // Initialize Contextmenu
    contextMenu = new QMenu(scrollView);

    addBookmarkAction = new QAction(SmallIcon("bookmark_add"), i18n("Add &Bookmark"), contextMenu);
    removeBookmarkAction = new QAction(SmallIcon("bookmark"), i18n("Remove &Bookmark"), contextMenu);
  }

  // TODO: use new slot-based API, once it works reliable
  if (dataModel->isPageBookmarked(pageNumber))
  {
    contextMenu->removeAction(addBookmarkAction);
    contextMenu->addAction(removeBookmarkAction);
  }
  else
  {
    contextMenu->removeAction(removeBookmarkAction);
    contextMenu->addAction(addBookmarkAction);
  }

  // Show Contextmenu
  QAction* selected = contextMenu->exec(position);

  if (selected == addBookmarkAction)
  {
    dataModel->addBookmark(pageNumber, QString::null);
  }
  else
  {
    dataModel->removeBookmark(pageNumber);
  }
}


QMatrix DocumentWidget::transform() const
{
  QMatrix transformation;

  if (dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Upsidedown)
    transformation = transformation.translate(pageSize().width(), pageSize().height());
  else if (dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Landscape)
    transformation = transformation.translate(pageSize().width(), 0);
  else if (dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Seascape)
    transformation = transformation.translate(0, pageSize().height());

  transformation = transformation.rotate(dataModel->preferences()->rotation() * 90);

  return transformation;
}


QMatrix DocumentWidget::inverseTransform() const
{
  QMatrix transformation;

  if (dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Upsidedown)
    transformation = transformation.translate(pageSize().width(), pageSize().height());
  else if (dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Landscape)
    transformation = transformation.translate(0, pageSize().width());
  else if (dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Seascape)
    transformation = transformation.translate(pageSize().height(), 0);

  transformation = transformation.rotate(-dataModel->preferences()->rotation() * 90);

  return transformation;
}


QPoint DocumentWidget::map(const QPoint& p) const
{
  return transform().map(p);
}


QRect DocumentWidget::map(const QRect& r) const
{
  return transform().mapRect(r);
}


QRegion DocumentWidget::map(const QRegion& r) const
{
  return transform().map(r);
}


QPoint DocumentWidget::inverseMap(const QPoint& p) const
{
  return inverseTransform().map(p);
}


QRect DocumentWidget::inverseMap(const QRect& r) const
{
  return inverseTransform().mapRect(r);
}


QRegion DocumentWidget::inverseMap(const QRegion& r) const
{
  return inverseTransform().map(r);
}


QSize DocumentWidget::sizeOfPageInPixel()
{
  return documentCache->sizeOfPageInPixel(getPageNumber());
}


bool DocumentWidget::resizeWidget()
{
  QSize newPageSize = sizeOfPageInPixel();
  if (dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Landscape ||
      dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Seascape)
  {
    newPageSize = QSize(newPageSize.height(), newPageSize.width());
  }

  // Resize, if necessary
  if (newPageSize != pageSize())
  {
    setPageSize(newPageSize);
    return true;
  }
  else
  {
    return false;
  }
}

#include "documentWidget.moc"
