// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-
//
// Class: hyperlink
//
// Part of KDVI- A previewer for TeX DVI files.
//
// (C) 2004-2005 Stefan Kebekus. Distributed under the GPL.

#ifndef _hyperlink_h_
#define _hyperlink_h_

#include "anchor.h"

#include <QRect>
#include <QString>


/** Represents a named, rectangular region in a rendered documentPage

    This trivial class is used in the documentPage class to represent
    a hyperlink in a rendered documentPage.

    @author Stefan Kebekus <kebekus@kde.org>
    @version 1.0.0
*/

class Hyperlink
{
public:
  /** \brief Default Constructor

  The default constructor leaves all fields uninitialized.
  */
  Hyperlink()
    : baseline(0)
   {}

  /** \brief Constructor

  Trivial constructor leaves that initialized all members.

  @param bl value for the baseline field
  @param re value for the box
  @param lT value for the text field
  @param anch value for the anchor. Leave empty for external links.
  */
  Hyperlink(quint32 bl, const QRect& re, const QString& lT, const Anchor& anch = Anchor())
    : baseline(bl), box(re), linkText(lT), anchor(anch)
  {}

  bool isValid() const { return baseline > 0 && box.isValid(); }
  
  /** \brief Base line of a hyperlink

  This field specifies the Y-coordinate of the base line of the
  bounding box in the same coordinates that were used when the
  associated documentPage was rendered by the
  documentRenderer.drawPage() method. It is used to underline
  hyperlinks in blue. Note that this field does generally differ from
  the Y-coordinate of the bottom of the bounding box, e.g. if the text
  in the box contains characters with underlengths, such as 'y', 'j'
  or 'g'.
   */
  quint32 baseline;

  /** \brief Bounding box of the text or hyperlink

  This rectangle specifies where on the page the hyperlink is
  found. It uses the same coordinates that were used when the
  associated documentPage was rendered by the
  documentRenderer.drawPage() method. The box is used to determine if
  the mouse pointer hovers over the link.
  */
  QRect    box;

  /** \brief Name of the region

  This field contains the name of the target,
  e.g. "http://www.kde.org". If the Hyperlink class is used to
  represent text, then the text is stored here.
  */
  QString  linkText;

  /** \brief Target of the link

  The Anchor specifies the place in the document where this link points
  to. If it is an external link, e.g. "http://www.kde.org", the anchor
  should be set to invalid.
  */
  Anchor anchor;

  /** \brief This method implements typecasts to QString */
  operator QString() const { return QString("(name=%1, anchor=%2)").arg(linkText).arg(anchor); }
};



#endif
