// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-
#include <config.h>

#include "bookmarkList.h"
#include "dataModel.h"
#include "dlgpresentation.h"
#include "exportDialog.h"
#include "ligaturePluginGUI.h"
#include "ligaturepart.h"
#include "kvs_debug.h"
#include "kvsprefs.h"
#include "optionDialogAccessibilityWidget.h"
#include "optionDialogGUIWidget.h"
#include "pageSizeDialog.h"
#include "presentationwidget.h"
#include "searchWidget.h"
#include "tableOfContents.h"
#include "textBox.h"
#include "zoomlimits.h"

#include <kaboutdata.h>

#include <kaction.h>
#include <kactioncollection.h>
#include <kapplication.h>
#include <kconfig.h>
#include <kconfigdialog.h>
#include <kdirwatch.h>
#include <kfiledialog.h>
#include <kfilterbase.h>
#include <kfilterdev.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <kicon.h>
#include <kinputdialog.h>
#include <kcomponentdata.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kmimetype.h>
#include <kparts/componentfactory.h>
#include <kparts/genericfactory.h>
#include <kselectaction.h>
#include <kstandardaction.h>
#include <kservicetypetrader.h>
#include <KTemporaryFile>
#include <ktoggleaction.h>
#include <ktoolinvocation.h>
#include <kprogressdialog.h>
#include <kvbox.h>
#include <kxmlguifactory.h>


#include <cmath>
#include <cstdlib>
#include <Q3HBox>
#include <QApplication>
#include <QFileInfo>
#include <QLabel>
#include <QProgressDialog>
#include <QSplitter>
#include <QTimer>
#include <QtGlobal>
#include <QTextStream>
#include <QToolBox>
#include <Q3VBox>

#define MULTIPAGE_VERSION 3

#define DEBUG_KMULTIPAGE

typedef KParts::GenericFactory<LigaturePart> LigaturePartFactory;
K_EXPORT_COMPONENT_FACTORY(ligaturepart, LigaturePartFactory)

LigaturePart::LigaturePart(QWidget* parentWidget,
                     QObject* parent,
		     const QStringList& args)
  : KParts::ReadWritePart(parent)
{
  Q_UNUSED(args);

  //
  // Initialize the ligaturepart
  //
  setComponentData(LigaturePartFactory::componentData());
  setReadWrite(true);
  ReadWritePart::setModified(false);
  setXMLFile("ligaturepart.rc");
  pageCache = new DocumentPageCache();

  //
  // Initialize and connect members
  //
  timer_id = -1;
  exportDialog = 0;
  saveAction = 0;
  multiPage = 0;
  multiPageLibrary = QString::null;
  searchInProgress = false;
  tmpUnzipped = 0;
  pageChangeIsConnected = false;
  // The page size dialog is constructed on first usage -- saves some
  // memory when not used.
  _pageSizeDialog = 0;
  // dirwatch, used to reload the main file, when changed
  watch = KDirWatch::self();
  connect(watch, SIGNAL(dirty(const QString&)), this, SLOT(fileChanged(const QString&)));
  watch->startScan();

  //
  // What's that?
  //
  KGlobal::locale()->insertCatalog("ligature");

  // Generate shared data
  dataModel = new DataModel();
  connect(dataModel, SIGNAL(currentPageNumberChanged()), this, SLOT(setCurrentPageNumber()));
  connect(dataModel, SIGNAL(numberOfPagesChanged()), this, SLOT(setCurrentPageNumber()));
  connect(dataModel, SIGNAL(selectionChanged(bool)), this, SLOT(textSelected(bool)));
  connect(dataModel, SIGNAL(viewModeChanged()), this, SLOT(setViewMode()));

  pageCache->setupObservers(dataModel);


  //
  // Generate GUI, and connect Signals/Slots
  //

  mainWidget = parentWidget;
  mainWidget->setFocusPolicy(Qt::StrongFocus);

  // Create splitter
  {
    splitterWidget = new QSplitter(mainWidget);
    setWidget(splitterWidget);
    splitterWidget->setOpaqueResize(false);
    splitterWidget->setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::MinimumExpanding);
    splitterWidget->setCollapsible(splitterWidget->indexOf(sideBar), false);
    splitterWidget->setSizes(dataModel->preferences()->guiLayout());

    // Create SideBar
    {
      sideBar = new QToolBox(splitterWidget);
      sideBar->setMinimumWidth(160);
      sideBar->setMaximumWidth(350);
      splitterWidget->setStretchFactor(splitterWidget->indexOf(sideBar), 0);

      // Create ContentsList
      {
        contentsSplitter = new QSplitter(Qt::Vertical, sideBar);
        contentsSplitter->setOpaqueResize(true);
        contentsSplitter->setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::MinimumExpanding);
        contentsSplitter->setSizes(dataModel->preferences()->contentsLayout());
        sideBar->addItem(contentsSplitter, QIcon(SmallIcon("contents")), i18n("Contents"));
      }

      // Create Table of Contents
      {
        tableOfContents = new TableOfContents(contentsSplitter);
        tableOfContents->setupObservers(dataModel);
      }

      // Create List of Bookmarks
      {
        KVBox* bookmarkBox = new KVBox(contentsSplitter);
        bookmarkBox->setSpacing(5);
        KHBox* bookmarkTitleBox = new KHBox(bookmarkBox);
        bookmarkTitleBox->setSpacing(4);
        QLabel* bookmarkTitlePixmap = new QLabel(bookmarkTitleBox);
        bookmarkTitlePixmap->setPixmap(KIconLoader::global()->loadIcon("bookmark", K3Icon::NoGroup, K3Icon::SizeSmall));
        bookmarkTitlePixmap->setFixedSize(K3Icon::SizeSmall, K3Icon::SizeSmall);
        new QLabel(i18n("Bookmarks"), bookmarkTitleBox);
        bookmarkList = new BookmarkList(bookmarkBox);
	bookmarkList->setupObservers(dataModel);
      }

      // Create MarkList
      {
        _markList = new MarkList(sideBar, "marklist");
        _markList->setupObservers(dataModel);
	sideBar->addItem(_markList, QIcon(SmallIcon("thumbnail")), i18n("Thumbnails"));
        _markList->setPageCache(pageCache);
      }

      // Restore state of the sidebar
      sideBar->setCurrentWidget(sideBar->widget(dataModel->preferences()->sideBarItem()));
    }// end: Sidebar

    // Create PageView
    {
      _pageView = new PageView(splitterWidget);
      _pageView->setPageCache(pageCache);
      _pageView->setupObservers(dataModel);
      connect(pageView(), SIGNAL(viewSizeChanged(const QSize&)), this, SLOT(slotStartFitTimer()));
      connect(pageView(), SIGNAL(zoomIn()), this, SLOT(zoomIn()));
      connect(pageView(), SIGNAL(zoomOut()), this, SLOT(zoomOut()));

      // The next three signals are relayed from the DocumentWidgets.
      // TODO: find a better way to do this.
      connect(pageView(), SIGNAL(localLink(const Hyperlink&)), this, SLOT(handleLocalLink(const Hyperlink&)));
      connect(pageView(), SIGNAL(setStatusBarText(const QString&)), this, SIGNAL(setStatusBarText(const QString&)) );
      connect(pageView(), SIGNAL(clearSelection()), this, SLOT(clearSelection()));
      connect(this, SIGNAL(switchTool(int)), pageView(), SLOT(slotSwitchTool()));
    } // end: PageView
  } // end: Splitter

  // Create Search Panel
  {
    searchWidget = new SearchWidget(mainWidget);
    searchWidget->hide();
    connect(searchWidget, SIGNAL(findNextText()), this, SLOT(findNextText()));
    connect(searchWidget, SIGNAL(findPrevText()), this, SLOT(findPrevText()));
    connect(searchWidget, SIGNAL(stopSearch()), this, SLOT(stopSearch()));
  }


  //
  // Actions
  //

  // File Menu
  exportTextAction = actionCollection()->addAction("export_text");
  exportTextAction->setText(i18n("Text..."));
  connect(exportTextAction, SIGNAL(triggered(bool)), this, SLOT(doExportText()));
  saveAction = KStandardAction::save(this, SLOT(save()), actionCollection());
  saveAsAction = KStandardAction::saveAs(this, SLOT(slotSaveAs()), actionCollection());
  printAction = KStandardAction::print(this, SLOT(slotPrint()), actionCollection());

  // Edit Menu
  copyTextAction = actionCollection()->addAction(KStandardAction::Copy, "copy_text", this, SLOT(copyText()));
  copyTextAction->setEnabled(false);

  selectAllAction = actionCollection()->addAction(KStandardAction::SelectAll, "edit_select_all",
                                                  this, SLOT(doSelectAll()));

  deselectAction = actionCollection()->addAction(KStandardAction::Deselect, "edit_deselect_all",
                                                 this, SLOT(clearSelection()));
  deselectAction->setEnabled(false);

  findTextAction = actionCollection()->addAction(KStandardAction::Find, "find", this, SLOT(showFindTextDialog()));

  findNextAction = actionCollection()->addAction(KStandardAction::FindNext, "findnext", this, SLOT(findNextText()));
  findNextAction->setEnabled(false);
  connect(searchWidget, SIGNAL(searchEnabled(bool)), findNextAction, SLOT(setEnabled(bool)));

  findPrevAction = actionCollection()->addAction(KStandardAction::FindPrev, "findprev", this, SLOT(findPrevText()));
  findPrevAction->setEnabled(false);
  connect(searchWidget, SIGNAL(searchEnabled(bool)), findPrevAction, SLOT(setEnabled(bool)));

  // View Menu
  QStringList viewModes;
  viewModes.append(i18n("Single Page"));
  viewModes.append(i18n("Continuous"));
  viewModes.append(i18n("Continuous - Facing"));
  viewModes.append(i18n("Overview"));
  viewModeAction = actionCollection()->add<KSelectAction>("viewmode");
  viewModeAction->setText(i18n("View Mode"));
  viewModeAction->setItems(viewModes);
  viewModeAction->setCurrentItem(dataModel->preferences()->viewMode());
  connect(viewModeAction, SIGNAL(triggered(int)), dataModel, SLOT(setViewMode(int)));

  media = actionCollection()->addAction("view_media");
  media->setText(i18n("Set Default Paper &Size..."));
  connect(media, SIGNAL(triggered()), this, SLOT(slotMedia()));

  guessViewMode = actionCollection()->add<KToggleAction>("view_guess_viewmode", this, SLOT(slotShowSidebar()));
  guessViewMode->setText(i18n("&Guess View Mode When Opening A New File"));

  zoomInAct = KStandardAction::zoomIn (this, SLOT(zoomIn()), actionCollection());
  zoomOutAct = KStandardAction::zoomOut(this, SLOT(zoomOut()), actionCollection());

  fitPageAct = actionCollection()->add<KToggleAction>("view_fit_to_page");
  fitPageAct->setIcon(KIcon("view_fit_window"));
  fitPageAct->setText(i18n("&Fit to Page"));
  fitPageAct->setShortcut(Qt::Key_P);
  connect(fitPageAct, SIGNAL(toggled(bool)), this, SLOT(enableFitToPage(bool)));

  fitWidthAct = actionCollection()->add<KToggleAction>("view_fit_to_width");
  fitWidthAct->setIcon(KIcon("view_fit_width"));
  fitWidthAct->setText(i18n("Fit to Page &Width"));
  fitWidthAct->setShortcut(Qt::Key_W);
  connect(fitWidthAct, SIGNAL(toggled(bool)), this, SLOT(enableFitToWidth(bool)));

  fitHeightAct = actionCollection()->add<KToggleAction>("view_fit_to_height");
  fitHeightAct->setIcon(KIcon("view_fit_height"));
  fitHeightAct->setText(i18n("Fit to Page &Height"));
  fitHeightAct->setShortcut(Qt::Key_H);
  connect(fitHeightAct, SIGNAL(toggled(bool)), this, SLOT(enableFitToHeight(bool)));

  QActionGroup* viewFitGroup = new QActionGroup(widget());
  viewFitGroup->setExclusive(true);
  fitPageAct->setActionGroup(viewFitGroup);
  fitWidthAct->setActionGroup(viewFitGroup);
  fitHeightAct->setActionGroup(viewFitGroup);

  rotateRightAction = actionCollection()->addAction("rotate_right");
  rotateRightAction->setIcon(KIcon("rotate_cw"));
  rotateRightAction->setText(i18n("Rotate right"));
  connect(rotateRightAction, SIGNAL(triggered(bool)), SLOT(slotRotateRight()));
  rotateRightAction->setShortcut(Qt::ALT+Qt::Key_R);
  rotateLeftAction = actionCollection()->addAction("rotate_left");
  rotateLeftAction->setIcon(KIcon("rotate_ccw"));
  rotateLeftAction->setText(i18n("Rotate left"));
  connect(rotateLeftAction, SIGNAL(triggered(bool)), SLOT(slotRotateLeft()));
  rotateLeftAction->setShortcut(Qt::ALT+Qt::Key_L);

  presentationAction = actionCollection()->addAction("presentation");
  presentationAction->setIcon(KIcon("kpresenter_kpr"));
  presentationAction->setText(i18n("&Presentation"));
  connect(presentationAction, SIGNAL(triggered(bool)), SLOT(slotShowPresentation()));
  presentationAction->setShortcut(Qt::CTRL+Qt::SHIFT+Qt::Key_P);

  // Go Menu
  gotoAct = KStandardAction::gotoPage(this, SLOT(goToPage()), actionCollection());
  gotoAct->setShortcut(Qt::CTRL+Qt::Key_G);

  backAct = KStandardAction::prior(pageView(), SLOT(prevPage()), actionCollection());
  forwardAct = KStandardAction::next(pageView(), SLOT(nextPage()), actionCollection());

  startAct = KStandardAction::firstPage(pageView(), SLOT(firstPage()), actionCollection());
  endAct = KStandardAction::lastPage(pageView(), SLOT(lastPage()), actionCollection());

  readUpAct = actionCollection()->addAction("go_read_up");
  readUpAct->setIcon(KIcon("up"));
  readUpAct->setText(i18n("Read Up Document"));
  connect(readUpAct, SIGNAL(triggered(bool)), pageView(), SLOT(readUp()));
  readUpAct->setShortcut(Qt::SHIFT+Qt::Key_Space);
  readDownAct = actionCollection()->addAction("go_read_down");
  readDownAct->setIcon(KIcon("down"));
  readDownAct->setText(i18n("Read Down Document"));
  connect(readDownAct, SIGNAL(triggered(bool)), pageView(), SLOT(readDown()));
  readDownAct->setShortcut(Qt::Key_Space);

  backAction = actionCollection()->addAction("go_back");
  backAction->setIcon(KIcon("1leftarrow"));
  backAction->setText(i18n("&Back"));
  connect(backAction, SIGNAL(triggered(bool)), pageView(), SLOT(goBack()));
  backAction->setEnabled(false);
  connect(dataModel->history(), SIGNAL(backItem(bool)), backAction, SLOT(setEnabled(bool)));

  forwardAction = actionCollection()->addAction("go_forward");
  forwardAction->setIcon(KIcon("1rightarrow"));
  forwardAction->setText(i18n("&Forward"));
  connect(forwardAction, SIGNAL(triggered(bool)), pageView(), SLOT(goForward()));
  forwardAction->setEnabled(false);
  connect(dataModel->history(), SIGNAL(forwardItem(bool)), forwardAction, SLOT(setEnabled(bool)));

  // Tools Menu
  moveModeAction = actionCollection()->add<KToggleAction>("move_tool");
  moveModeAction->setIcon(KIcon("movetool"));
  moveModeAction->setText(i18n("&Move Tool"));
  moveModeAction->setShortcut(Qt::Key_F4);
  connect(moveModeAction, SIGNAL(activated()), this, SLOT(slotEnableMoveTool()));

  selectionModeAction = actionCollection()->add<KToggleAction>("selection_tool");
  selectionModeAction->setIcon(KIcon("selectiontool"));
  selectionModeAction->setText(i18n("&Selection Tool"));
  selectionModeAction->setShortcut(Qt::Key_F5);
  connect(selectionModeAction, SIGNAL(activated()), this, SLOT(slotEnableSelectionTool()));

  rectSelectToolAction = actionCollection()->add<KToggleAction>("rect_selection_tool");
  rectSelectToolAction->setIcon(KIcon("frame_edit"));
  rectSelectToolAction->setText(i18n("&Rectangle Selection"));
  rectSelectToolAction->setShortcut(Qt::Key_F6);
  connect(rectSelectToolAction, SIGNAL(activated()), this, SLOT(slotEnableRectSelectionTool()));

  rulerToolAction = actionCollection()->add<KToggleAction>("ruler_tool");
  rulerToolAction->setIcon(KIcon("measuretool"));
  rulerToolAction->setText(i18n("Measuring &Tool"));
  rulerToolAction->setShortcut(Qt::Key_F7);
  connect(rulerToolAction, SIGNAL(activated()), this, SLOT(slotEnableRulerTool()));

  QActionGroup* toolsGroup = new QActionGroup(widget());
  toolsGroup->setExclusive(true);
  moveModeAction->setActionGroup(toolsGroup);
  selectionModeAction->setActionGroup(toolsGroup);
  rectSelectToolAction->setActionGroup(toolsGroup);
  rulerToolAction->setActionGroup(toolsGroup);

  // Settings Menu
  showSidebar = actionCollection()->add<KToggleAction>("show_sidebar");
  showSidebar->setIcon(KIcon("show_side_panel"));
  showSidebar->setText(i18n("Show &Sidebar"));
  connect(showSidebar, SIGNAL(triggered(bool) ), SLOT(slotShowSidebar()));
  showSidebar->setCheckedState(KGuiItem(i18n("Hide &Sidebar")));

  scrollbarHandling = actionCollection()->add<KToggleAction>("scrollbarHandling");
  scrollbarHandling->setText(i18n("Show Scrollbars"));
  scrollbarHandling->setCheckedState(KGuiItem(i18n("Hide Scrollbars")));

  watchAct = actionCollection()->add<KToggleAction>("watch_file");
  watchAct->setText(i18n("&Watch File"));

  settingsAction = KStandardAction::preferences(this, SLOT(doSettings()), actionCollection());

  // ToolBar
  zoom_action = actionCollection()->add<KSelectAction>("view_zoom");
  zoom_action->setText(i18n("&Zoom"));
  zoom_action->setEditable(true);
  zoom_action->setItems(_zoomVal.zoomNames());

  connect(&_zoomVal, SIGNAL(zoomNamesChanged(const QStringList &)), this, SLOT(zoomNamesChanged(const QStringList &)));
  connect(&_zoomVal, SIGNAL(valNoChanged(int)), this, SLOT(setCurrentZoomItem(int)));
  connect(&_zoomVal, SIGNAL(zoomNameChanged(const QString &)), this, SIGNAL(zoomChanged(const QString &)) );
  connect(zoom_action, SIGNAL(triggered(const QString &)), this, SLOT(setZoomValue(const QString &)));


  // Movement actions
  QAction* scrollUpAction = actionCollection()->addAction("scroll_up");
  scrollUpAction->setText(i18n("Scroll Up"));
  scrollUpAction->setShortcut(Qt::Key_Up);
  pageView()->addAction(scrollUpAction);
  connect(scrollUpAction, SIGNAL(triggered(bool)), pageView(), SLOT(scrollUp()));

  QAction* scrollDownAction = actionCollection()->addAction("scroll_down");
  scrollDownAction->setText(i18n("Scroll Down"));
  scrollDownAction->setShortcut(Qt::Key_Down);
  pageView()->addAction(scrollDownAction);
  connect(scrollDownAction, SIGNAL(triggered(bool)), pageView(), SLOT(scrollDown()));

  QAction* scrollLeftAction = actionCollection()->addAction("scroll_left");
  scrollLeftAction->setText(i18n("Scroll Left"));
  scrollLeftAction->setShortcut(Qt::Key_Left);
  pageView()->addAction(scrollLeftAction);
  connect(scrollLeftAction, SIGNAL(triggered(bool)), pageView(), SLOT(scrollLeft()));

  QAction* scrollRightAction = actionCollection()->addAction("scroll_right");
  scrollRightAction->setText(i18n("Scroll Right"));
  scrollRightAction->setShortcut(Qt::Key_Right);
  pageView()->addAction(scrollRightAction);
  connect(scrollRightAction, SIGNAL(triggered(bool)), pageView(), SLOT(scrollRight()));

  QAction* scrollUpPageAction = actionCollection()->addAction("scroll_up_page");
  scrollUpPageAction->setText(i18n("Scroll Up Page"));
  scrollUpPageAction->setShortcut(Qt::SHIFT + Qt::Key_Up);
  pageView()->addAction(scrollUpPageAction);
  connect(scrollUpPageAction, SIGNAL(triggered(bool)), pageView(), SLOT(scrollUpPage()));

  QAction* scrollDownPageAction = actionCollection()->addAction("scroll_down_page");
  scrollDownPageAction->setText(i18n("Scroll Down Page"));
  scrollDownPageAction->setShortcut(Qt::SHIFT + Qt::Key_Down);
  pageView()->addAction(scrollDownPageAction);
  connect(scrollDownPageAction, SIGNAL(triggered(bool)), pageView(), SLOT(scrollDownPage()));

  QAction* scrollLeftPageAction = actionCollection()->addAction("scroll_left_page");
  scrollLeftPageAction->setText(i18n("Scroll Left Page"));
  scrollLeftPageAction->setShortcut(Qt::SHIFT + Qt::Key_Left);
  pageView()->addAction(scrollLeftPageAction);
  connect(scrollLeftPageAction, SIGNAL(triggered(bool)), pageView(), SLOT(scrollLeftPage()));

  QAction* scrollRightPageAction = actionCollection()->addAction("scroll_right_page");
  scrollRightPageAction->setText(i18n("Scroll Right Page"));
  scrollRightPageAction->setShortcut(Qt::SHIFT + Qt::Key_Right);
  pageView()->addAction(scrollRightPageAction);
  connect(scrollRightPageAction, SIGNAL(triggered(bool)), pageView(), SLOT(scrollRightPage()));

  QAction* nextViewModeAction = actionCollection()->addAction("next_view_mode_action");
  nextViewModeAction->setText(i18n("Next View Mode"));
  nextViewModeAction->setShortcut(Qt::CTRL+Qt::Key_M);
  pageView()->addAction(nextViewModeAction);
  connect(nextViewModeAction, SIGNAL(triggered(bool)), this, SLOT(slotNextViewMode()));

  QAction* previousViewModeAction = actionCollection()->addAction("previous_view_mode_action");
  previousViewModeAction->setText(i18n("Previous View Mode"));
  previousViewModeAction->setShortcut(Qt::CTRL+Qt::SHIFT+Qt::Key_M);
  pageView()->addAction(previousViewModeAction);
  connect(previousViewModeAction, SIGNAL(triggered(bool)), this, SLOT(slotPreviousViewMode()));

  //
  // Read Settings
  //
  readSettings();

  // ScrollBars
  connect(scrollbarHandling, SIGNAL(toggled(bool)),  pageView(), SLOT(slotShowScrollbars(bool)));

  // To register the tool icons in the statusbar we need to send signals to the ligature, since
  // we don't own a pointer to it. There is the complication that emitting signals does not work
  // in the constructor. To work around a problem we use a timer, so the registerTools() method
  // gets called by the eventloop outside of the constructor.
  QTimer::singleShot(10, this, SLOT(registerTools()));

  m_extension = new LigaturePartExtension(this);

  checkActions();
}


LigaturePart::~LigaturePart()
{
  writeSettings();

  // This stops the rendering thread. It is important to stop the
  // thread before we delete the multipage (... and thus the renderer)
  // in the next step
  pageCache->setRenderer(0);
  delete multiPage;

  delete tmpUnzipped;

  // The following objects hold pointers to the dataModel. It is
  // therefore very important that we delete them before we delete the
  // dataModel. Remaining objects/widgets will later automatically be
  // deleted by Qt.
  delete tableOfContents;
  delete bookmarkList;
  delete _markList;
  delete _pageView;
  delete pageCache;
  delete _pageSizeDialog;

  delete dataModel;
}


void LigaturePart::slotStartFitTimer()
{
  fitTimer.setSingleShot(true);
  fitTimer.start(100);
}


void LigaturePart::restoreDocument(const KUrl &url, int page)
{
  if (openUrl(url))
    dataModel->setCurrentPageNumber(Anchor(page));
}


void LigaturePart::saveDocumentRestoreInfo(KConfig* config)
{
  if (multiPage.isNull())
    return;

  config->writePathEntry("URL", url().url());
  if (dataModel->numberOfPages() > 0)
    config->writeEntry("Page", QString::number(dataModel->currentPageNumber()));
}


void LigaturePart::slotFileOpen()
{
  if ((!multiPage.isNull()) && (multiPage->isModified() == true))
  {
    int ans = KMessageBox::warningContinueCancel( 0,
                                         i18n("Your document has been modified. Do you really want to open another document?"),
                                         i18n("Warning - Document Was Modified"),KStandardGuiItem::open());
    if (ans == KMessageBox::Cancel)
      return;
  }

  KUrl url = KFileDialog::getOpenUrl(KUrl(), supportedMimeTypes().join(" "), mainWidget);

  if (!url.isEmpty())
    openUrl(url);
}

QStringList LigaturePart::supportedMimeTypes()
{
  QStringList supportedMimeTypes;

  // Search for service
  KService::List offers = KServiceTypeTrader::self()->query(
    QString::fromLatin1("Ligature/MultiPage"),
    QString("([X-KDE-MultiPageVersion] == %1)").arg(MULTIPAGE_VERSION)
  );

  if (!offers.isEmpty())
  {
    KService::List::ConstIterator iterator = offers.begin();
	KService::List::ConstIterator end = offers.end();
    QStringList::Iterator mimeType;

    for (; iterator != end; ++iterator)
    {
      KService::Ptr service = *iterator;

      QStringList mimeTypes = service->serviceTypes();
      for (mimeType = mimeTypes.begin(); mimeType != mimeTypes.end(); ++mimeType)
      {
        if (!(*mimeType).contains("Ligature"))
          supportedMimeTypes << *mimeType;
      }
    }
  }

  // The ligature is also able to read compressed files and to
  // uncompress them on the fly.

  // Check if this version of KDE supports bzip2
  KFilterBase* filter = KFilterBase::findFilterByMimeType( "application/x-bzip2" );

  const bool bzip2Available = filter != 0;
  delete filter;

  supportedMimeTypes << "application/x-gzip";

  if (bzip2Available)
    supportedMimeTypes << "application/x-bzip2";

  return supportedMimeTypes;
}

QStringList LigaturePart::fileFormats() const
{
  // Compile a list of the supported filename patterns

  // First we build a list of the mimetypes which are supported by the
  // currently installed ligaturePluginGUI-Plugins.
  QStringList supportedMimeTypes;
  QStringList supportedPattern;

  // Search for service
  KService::List offers = KServiceTypeTrader::self()->query(
      QString::fromLatin1("Ligature/MultiPage"),
      QString("([X-KDE-MultiPageVersion] == %1)").arg(MULTIPAGE_VERSION)
  );

  if (!offers.isEmpty())
  {
    KService::List::ConstIterator iterator = offers.begin();
	KService::List::ConstIterator end = offers.end();
    QStringList::Iterator mimeType;

    for (; iterator != end; ++iterator)
    {
      KService::Ptr service = *iterator;

      QStringList mimeTypes = service->serviceTypes();
      for (mimeType = mimeTypes.begin(); mimeType != mimeTypes.end(); ++mimeType)
      {
        if (!(*mimeType).contains("Ligature"))
        {
          QStringList pattern = KMimeType::mimeType(*mimeType)->patterns();
          while(!pattern.isEmpty())
          {
            supportedPattern.append(pattern.front().trimmed());
            pattern.pop_front();
          }
        }
      }
    }
  }

  // The ligature is also able to read compressed files and to
  // uncompress them on the fly. Thus, we modify the list of supported
  // file formats which we obtain from the multipages to include
  // compressed files like "*.dvi.gz". We add "*.dvi.bz2" if support
  // for bzip2 is compiled into KDE.

  // Check if this version of KDE supports bzip2
  KFilterBase* filter = KFilterBase::findFilterByMimeType( "application/x-bzip2" );
  const bool bzip2Available = filter != 0L;
  delete filter;

  QStringList compressedPattern;

  for(QStringList::Iterator it = supportedPattern.begin(); it != supportedPattern.end(); ++it )
  {
    if ((*it).indexOf(".gz", -3) == -1) // Paranoia safety check
      compressedPattern.append(*it + ".gz");

    if ((bzip2Available) && ((*it).indexOf(".bz2", -4) == -1)) // Paranoia safety check
      compressedPattern.append(*it + ".bz2");
  }

  while (!compressedPattern.isEmpty())
  {
    supportedPattern.append(compressedPattern.front());
    compressedPattern.pop_front();
  }

  kDebug(kvs::shell) << "Supported Pattern: " << supportedPattern << endl;

  return supportedPattern;
}


void LigaturePart::slotSetFullPage(bool fullpage)
{
  pageView()->setFullScreenMode(fullpage);
  if (fullpage)
    slotShowSidebar(false);

  // Restore normal view
  if (fullpage == false)
    {
    slotShowSidebar();
    pageView()->slotShowScrollbars(scrollbarHandling->isChecked());
  }
}


void LigaturePart::slotShowSidebar()
{
  bool show = showSidebar->isChecked();
  slotShowSidebar(show);
}


bool LigaturePart::openFile()
{
  closeUrl();

  KUrl tmpFileURL;

  // We try to be error-tolerant about filenames. If the user calls us
  // with something like "test", and we are using the DVI-part, we'll
  // also look for "testdvi" and "test.dvi".
  QFileInfo fi(m_file);
  m_file = fi.absoluteFilePath();

  if (!fi.exists())
  {
    QStringList supportedPatterns = fileFormats();
    QStringList endings;

    for (QStringList::Iterator it = supportedPatterns.begin(); it != supportedPatterns.end(); ++it)
    {
      // Only consider patterns starting with "*."
      if ((*it).indexOf("*.") == 0)
      {
        // Remove first Letter from string
        endings.append((*it).mid(2).trimmed());
      }
    }
    kDebug(kvs::shell) << "Supported Endings: " << endings << endl;

    // Now try to append the endings with and without "." to the given filename,
    // and see if that gives a existing file.
    for (QStringList::Iterator it = endings.begin(); it != endings.end(); ++it)
    {
      fi.setFile(m_file+(*it));
      if (fi.exists())
      {
        m_file = m_file+(*it);
        break;
      }
      fi.setFile(m_file+"."+(*it));
      if (fi.exists())
      {
        m_file = m_file+"."+(*it);
        break;
      }
    }

    // If we still have not found a file. Show an error message and return.
    if (!fi.exists())
    {
      KMessageBox::error(mainWidget, i18n("<qt>File <nobr><strong>%1</strong></nobr> does not exist.</qt>", m_file));
      emit setStatusBarText(QString::null);
      return false;
    }
    m_url.setPath(QFileInfo(m_file).absoluteFilePath());
  }

  // Set the window caption now, before we do any uncompression and generation of temporary files.
  tmpFileURL.setPath(m_file);
  emit setStatusBarText(i18n("Loading '%1'...", tmpFileURL.prettyUrl()));
  emit setWindowCaption( tmpFileURL.prettyUrl() ); // set Window caption WITHOUT the reference part!

  // Check if the file is compressed
  KMimeType::Ptr mimetype = KMimeType::findByPath( m_file );

  if (( mimetype->name() == "application/x-gzip" ) ||
      ( mimetype->name() == "application/x-bzip2" ) ||
      ( mimetype->parentMimeType() == "application/x-gzip" ) ||
      ( mimetype->parentMimeType() == "application/x-bzip2" ))
  {
    // The file is compressed. Make a temporary file, and store an uncompressed version there...
    if (tmpUnzipped != 0L)  // Delete old temporary file
      delete tmpUnzipped;

    tmpUnzipped = new KTemporaryFile;
    if (tmpUnzipped == 0L)
    {
      KMessageBox::error(mainWidget, i18n("<qt><strong>File Error</strong> Could not create "
                         "temporary file.</qt>"));
      emit setWindowCaption(QString::null);
      emit setStatusBarText(QString::null);
      return false;
    }
    tmpUnzipped->setAutoRemove(true);
    tmpUnzipped->open();
    if(tmpUnzipped->error() != QFile::NoError)
    {
      KMessageBox::error(mainWidget, i18n("<qt><strong>File Error</strong> Could not create temporary file "
                         "<nobr><strong>%1</strong></nobr>.</qt>", tmpUnzipped->errorString()));
      emit setWindowCaption(QString::null);
      emit setStatusBarText(QString::null);
      return false;
    }

    QIODevice* filterDev;
    if (( mimetype->parentMimeType() == "application/x-gzip" ) ||
        ( mimetype->parentMimeType() == "application/x-bzip2" ))
      filterDev = KFilterDev::deviceForFile(m_file, mimetype->parentMimeType());
    else
      filterDev = KFilterDev::deviceForFile(m_file);
    if (filterDev == 0L)
    {
      emit setWindowCaption(QString::null);
      emit setStatusBarText(QString::null);
      return false;
    }
    if(!filterDev->open(QIODevice::ReadOnly))
    {
      KMessageBox::detailedError(mainWidget, i18n("<qt><strong>File Error</strong> Could not open the file "
          "<nobr><strong>%1</strong></nobr> for uncompression. "
          "The file will not be loaded.</qt>", m_file),
          i18n("<qt>This error typically occurs if you do not have enough permissions to read the file. "
          "You can check ownership and permissions if you right-click on the file in the Konqueror "
          "file manager and then choose the 'Properties' menu.</qt>"));
      emit setWindowCaption(QString::null);
      delete filterDev;
      emit setStatusBarText(QString::null);
      return false;
    }

    KProgressDialog progress(0L,
                             i18n("Uncompressing..."),
                             i18n("<qt>Uncompressing the file <nobr><strong>%1</strong></nobr>. Please wait.</qt>", m_file));

    progress.progressBar()->setMaximum((int) fi.size()/1024);
    progress.progressBar()->setValue(0);
    progress.setMinimumDuration(250);

    QByteArray buf(1024, ' ');
    int read = 0, wrtn = 0;

    while ((read = filterDev->read(buf.data(), buf.size())) > 0)
    {
      qApp->processEvents();
      if (progress.wasCancelled())
        break;
      progress.progressBar()->setValue(progress.progressBar()->value()+1);

      wrtn = tmpUnzipped->write(buf.data(), read);
      if(read != wrtn)
        break;
    }
    delete filterDev;
    tmpUnzipped->flush();

    if (progress.wasCancelled()) {
      emit setStatusBarText(QString::null);
      return false;
    }

    if ((read != 0) || (tmpUnzipped->size() == 0))
    {
      KMessageBox::detailedError(mainWidget, i18n("<qt><strong>File Error</strong> Could not uncompress "
          "the file <nobr><strong>%1</strong></nobr>. The file will not be loaded.</qt>", m_file ),
          i18n("<qt>This error typically occurs if the file is corrupt. "
          "If you want to be sure, try to decompress the file manually using command-line tools.</qt>"));
      emit setWindowCaption(QString::null);
      emit setStatusBarText(QString::null);
      return false;
    }
    m_file = tmpUnzipped->fileName();
    tmpUnzipped->close();
  }

  // Now call the openUrl-method of the multipage and give it an URL
  // pointing to the downloaded file.
  tmpFileURL.setPath(m_file);
  // Pass the reference part of the URL through to the multipage
  tmpFileURL.setRef(m_url.ref());

  mimetype = KMimeType::findByUrl(tmpFileURL);

  // Build the mimetype filter for this file, used by saveAs()
  mimeTypeFilter = mimetype->patterns().join(" ") + "|" + mimetype->comment();

  // Search for a plugin that supports the needed mimetype.
  KService::List offers = KServiceTypeTrader::self()->query(
    QString::fromLatin1("Ligature/MultiPage"),
    QString("([X-KDE-MultiPageVersion] == %1)").arg(MULTIPAGE_VERSION)
  );

  KService::Ptr service;
  bool matchingPluginFound = false;

  if (!offers.isEmpty())
  {
    KService::List::ConstIterator iterator = offers.begin();
    KService::List::ConstIterator end = offers.end();
    QStringList::Iterator mimeTypeIterator;

    for (; iterator != end; ++iterator)
    {
      QStringList mimeTypes = (*iterator)->serviceTypes();
      for (mimeTypeIterator = mimeTypes.begin(); mimeTypeIterator != mimeTypes.end(); ++mimeTypeIterator)
      {
        // If one of the supported mimetypes of the plugin equals the mimetype we need,
        // we remember the service and stop the search.
        if (mimetype->name() == *mimeTypeIterator)
        {
          service = *iterator;
          matchingPluginFound = true;
          break;
        }
      }
      if (matchingPluginFound)
        break;
    }
  }

  if (offers.isEmpty() || !matchingPluginFound)
  {
    KMessageBox::detailedError(mainWidget, i18n("<qt>The document <b>%1</b> cannot be shown because "
                                                "its file type is not supported.</qt>", m_file),
                               i18n("<qt>The file has mime type <b>%1</b> which is not supported by "
                                    "any of the installed Ligature plugins.</qt>", mimetype->name()));
    emit setWindowCaption(QString::null);
    emit setStatusBarText(QString::null);
    return false;
  }

  // The the new multiPage is different then the currently loaded one.
  if (service->library() != multiPageLibrary)
  {
    // We write the settings before we load the new multipage, so
    // that the new multipage gets the same settings than the
    // currently loaded one.
    writeSettings();

    // Delete old config dialog
    KConfigDialog* configDialog = KConfigDialog::exists("ligature_config");
    delete configDialog;

    ligaturePluginGUI* oldMultiPage = multiPage;

    // Try to load the multiPage
    int error = 0;
    multiPage = static_cast<ligaturePluginGUI*>(KService::createInstance<KParts::Plugin>(service, mainWidget, QStringList(), &error));

    if (multiPage.isNull()) {
      QString reason;
      switch(error) {
      case KLibLoader::ErrNoServiceFound:
        reason = i18n("<qt>No service implementing the given mimetype and fullfilling the given constraint expression can be found.</qt>");
        break;
      case KLibLoader::ErrServiceProvidesNoLibrary:
        reason = i18n("<qt>The specified service provides no shared library.</qt>");
        break;
      case KLibLoader::ErrNoLibrary:
        reason = i18n("<qt><p>The specified library <b>%1</b> could not be loaded. The error message returned was:</p> <p><b>%2</b></p></qt>", service->library(), KLibLoader::self()->lastErrorMessage());
        break;
      case KLibLoader::ErrNoFactory:
        reason = i18n("<qt>The library does not export a factory for creating components.</qt>");
        break;
      case KLibLoader::ErrNoComponent:
        reason = i18n("<qt>The factory does not support creating components of the specified type.</qt>");
        break;
      }

      QString text = i18n("<qt><p><b>Problem:</b> The document <b>%1</b> cannot be shown.</p>"
                          "<p><b>Reason:</b> The software "
                          "component <b>%2</b> which is required to display files of type <b>%3</b> could "
                          "not be initialized. This could point to serious misconfiguration of your KDE "
                          "system, or to damaged program files.</p>"
                          "<p><b>What you can do:</b> You could try to re-install the software packages in "
                          "question. If that does not help, you could file an error report, either to the "
                          "provider of your software (e.g. the vendor of your Linux distribution), or "
                          "directly to the authors of the software. The entry <b>Report Bug...</b> in the "
                          "<b>Help</b> menu helps you to contact the KDE programmers.</p></qt>", m_file, service->library(), mimetype->name());
      QString caption = i18n("Error Initializing Software Component");
      if (reason.isEmpty())
        KMessageBox::error(mainWidget, text, caption);
      else
        KMessageBox::detailedError(mainWidget, text, reason, caption);
      emit setStatusBarText(QString::null);
      return false;
    }

    // Set parent widget for the multipage
    multiPage->setParentWidget(mainWidget);

    // Setup GUI that the multipage might wish to add

    // For some KPart hosts (kile for example) don't build their GUI with
    // KXMLGUIFactory, and factory() returns 0. In these cases we cannot
    // add gui items of the plugins.
    if (factory())
    {
      factory()->addClient(multiPage);
    }
    else
    {
      kError() << "factory() == NULL: cannot create plugin specific gui items." << endl;
    }

    // Setup shared data
    multiPage->setupObservers(dataModel);

    // Clear the page view.
    pageView()->clear();

    // Relay signals.
    connect(multiPage, SIGNAL(setStatusBarText(const QString& ) ), this, SLOT(setStatusBarTextFromMultiPage(const QString& ) ) );
    connect(multiPage, SIGNAL(documentHasBeenModified()), this, SLOT(setModified()));
    connect(multiPage, SIGNAL(renderModeChanged()), this, SLOT(renderModeChanged()));
    if (multiPage->getRenderer().isNull())
      kError(kvs::shell) << "LigaturePart::openFile(): multiPage->getRenderer() == 0, cannot connect renderer"  << endl;
    else
      connect(multiPage->getRenderer(), SIGNAL(setStatusBarText(const QString&)), this, SIGNAL(setStatusBarText(const QString&)));

    // Remember the name of the part. So only need to switch if really necessary.
    multiPageLibrary = service->library();

    pageView()->setMultiPage(multiPage);

    // Initialize documentPageCache.
    pageCache->setRenderer(multiPage->getRenderer());

    if (oldMultiPage != 0)
    {
      if (factory())
      {
        factory()->removeClient(oldMultiPage);
      }
      else
      {
        kError() << "factory() == NULL: cannot remove plugin specific gui items." << endl;
      }
      delete oldMultiPage;
    }

    // This makes sure that the GUI of the multipage is switched on and
    // off whenever the GUI of this part is switched on or off --for
    // instance, in konqueror when this ligaturepart is shown in one TAB,
    // and the used switches to another tab, both the GUI of this
    // ligaturepart, and the GUI of the multipage must vanish.
    insertChildClient( multiPage );

    readSettings();
  }


  // Load the URL
  dataModel->deselectText();
  dataModel->history()->clear();
  pageCache->clear();
  dataModel->removeAllBookmarks();
  emit setStatusBarText(i18n("Loading file %1", m_file));

  bool r = multiPage->openUrl(m_file, m_url);

  if (r) {
    dataModel->setCurrentPageNumber(Anchor(1));

    if (guessViewMode->isChecked()) {
      // Guess suitable view mode
      if ((multiPage->getRenderer()->totalPages() < 3) ||
          (multiPage->getRenderer()->sizeOfPage(1).isPortrait() != true))
        dataModel->setViewMode(KVSPrefs::EnumViewMode::Continuous);
      else
        dataModel->setViewMode(KVSPrefs::EnumViewMode::ContinuousFacing);
    } else
      // Set the multipage to the current viewmode.
      dataModel->setViewMode(dataModel->preferences()->viewMode());
  }

  // Clear Statusbar
  emit setStatusBarText(QString::null);

  updateZoomLevel(); // @@@@@@@@@@@@@

  // We disable the selection tool for plugins that dont support text.
  // Currently this is only the fax plugin.
  if (multiPage->supportsTextSearch())
  {
    selectionModeAction->setEnabled(true);
  }
  else
  {
    selectionModeAction->setEnabled(false);
    // Switch to the movetool when textselection is not supported
    if (dataModel->preferences()->tool() == KVSPrefs::EnumTool::Select)
    {
      moveModeAction->setChecked(true);
    }
  }

  if (r) {
    dataModel->loadDocumentInfo(m_file+".ligature");

    dataModel->setCurrentPageNumber(Anchor(1));

    QString reference = url().ref();
    if (!reference.isEmpty())
      dataModel->setCurrentPageNumber(multiPage->getRenderer()->parseReference(reference));

    // Set Table of Contents
    tableOfContents->setContents(multiPage->getRenderer()->getBookmarks());

    // Add the file to the watchlist
    watch->addFile( m_file );

    // Notify the ViewShell about the newly opened file.
    emit fileOpened();
  } else {
    m_url = KUrl();
    emit setWindowCaption(QString::null);
  }

  checkActions();
  emit zoomChanged(QString("%1%").arg((int)(_zoomVal.value()*100.0+0.5)));
  emit setStatusBarText(QString::null);
  return r;
}


void LigaturePart::fileChanged(const QString &file)
{
  if (file == m_file && watchAct->isChecked())
    reload();
}


bool LigaturePart::closeUrl()
{
  // Paranoid safety check
  if (multiPage.isNull())
    return true;

  abortLoad(); //just in case
  if ( isReadWrite() && isModified() )
    if (!queryClose())
      return false;

  bool wasModified = isReadWrite() && isModified();

  KParts::ReadOnlyPart::closeUrl();
  ReadWritePart::setModified(false);

  if( watch && !m_file.isEmpty() )
    watch->removeFile( m_file );

  if ( !wasModified && !m_file.isEmpty() )
    dataModel->saveDocumentInfo(m_file+".ligature");
  dataModel->removeAllBookmarks();
  dataModel->setNumberOfPages(0);

  // Clear navigation history.
  dataModel->history()->clear();

  // Clear Table of Contents
  tableOfContents->clear();

  // Clear the page cache
  pageCache->clear();

  multiPage->closeUrl();
  checkActions();
  emit setWindowCaption("");

  // Not modified => ok and delete temp file.
  return true;
}


void LigaturePart::slotMedia()
{
  // If the user has chosen "Custom paper size..", show the paper size
  // dialog. Construct it, if necessary. The paper size dialog will
  // know the address of userRequestedPaperSize and change this
  // member, if the user clicks ok/accept. The signal/slot mechanism
  // will then make sure that the necessary updates in the GUI are
  // done.
  if (_pageSizeDialog == 0) {
     _pageSizeDialog = new pageSizeDialog(mainWidget);
     _pageSizeDialog->setupObservers(dataModel);
    if (_pageSizeDialog == 0) {
      kError(kvs::shell) << "Could not construct the page size dialog!" << endl;
      return;
    }
  }

  // Reset the "preferred paper size" menu. We don't want to have the
  // "custom paper size" check if the user aborts the dialog.
  checkActions();

  // Set or update the paper size dialog to show the currently
  // selected value.
   _pageSizeDialog->setPageSize(dataModel->defaultPageSize().serialize());
   _pageSizeDialog->show();
}


void LigaturePart::goToPage()
{
  if (multiPage.isNull()) {
    kError(kvs::shell) << "LigaturePart::goToPage() called with multiPage == NULL" << endl;
    return;
  }

  bool ok = false;
  PageNumber p = KInputDialog::getInteger(i18n("Go to Page"), i18n("Page:"),
                                          dataModel->currentPageNumber(), 1, dataModel->numberOfPages(),
                                          1 /*step*/, &ok, mainWidget);
  if (ok)
    dataModel->setCurrentPageNumber(p);
}


void LigaturePart::disableZoomFit()
{
  switch(dataModel->preferences()->fitToPage())
  {
    case KVSPrefs::EnumFitToPage::FitToPage:
      fitPageAct->setChecked(false);
      enableFitToPage(false);
      break;
    case KVSPrefs::EnumFitToPage::FitToPageWidth:
      fitWidthAct->setChecked(false);
      enableFitToWidth(false);
      break;
    case KVSPrefs::EnumFitToPage::FitToPageHeight:
      fitHeightAct->setChecked(false);
      enableFitToHeight(false);
      break;
    default:
      // we are already in manual zoom mode
      return;
  }
}


void LigaturePart::zoomIn()
{
  disableZoomFit();

  float oldVal = _zoomVal.value();
  float newVal = _zoomVal.zoomIn();

  if (oldVal != newVal)
    _zoomVal.setZoomValue(setZoom(_zoomVal.zoomIn()));
}


void LigaturePart::zoomOut()
{
  disableZoomFit();

  float oldVal = _zoomVal.value();
  float newVal = _zoomVal.zoomOut();

  if (oldVal != newVal)
    _zoomVal.setZoomValue(setZoom(_zoomVal.zoomOut()));
}


void LigaturePart::updateZoomLevel()
{
  switch(dataModel->preferences()->fitToPage())
  {
    case KVSPrefs::EnumFitToPage::FitToPage:
      fitToPage();
      break;
    case KVSPrefs::EnumFitToPage::FitToPageWidth:
      fitToWidth();
      break;
    case KVSPrefs::EnumFitToPage::FitToPageHeight:
      fitToHeight();
      break;
    default:
      // manual zoom, nothing to do
      return;
  }
}

void LigaturePart::enableFitToPage(bool enable)
{
  if (enable)
  {
    dataModel->preferences()->setFitToPage(KVSPrefs::EnumFitToPage::FitToPage);
    fitToPage();
    connect(&fitTimer, SIGNAL(timeout()), SLOT(fitToPage()));
  }
  else
  {
    dataModel->preferences()->setFitToPage(KVSPrefs::EnumFitToPage::DontFit);
    disconnect(&fitTimer, SIGNAL(timeout()), this, SLOT(fitToPage()));
  }
}

void LigaturePart::enableFitToWidth(bool enable)
{
  if (enable)
  {
    dataModel->preferences()->setFitToPage(KVSPrefs::EnumFitToPage::FitToPageWidth);
    fitToWidth();
    connect(&fitTimer, SIGNAL(timeout()), SLOT(fitToWidth()));
  }
  else
  {
    dataModel->preferences()->setFitToPage(KVSPrefs::EnumFitToPage::DontFit);
    disconnect(&fitTimer, SIGNAL(timeout()), this, SLOT(fitToWidth()));
  }
}


void LigaturePart::enableFitToHeight(bool enable)
{
  if (enable)
  {
    dataModel->preferences()->setFitToPage(KVSPrefs::EnumFitToPage::FitToPageHeight);
    fitToHeight();
    connect(&fitTimer, SIGNAL(timeout()), SLOT(fitToHeight()));
  }
  else
  {
    dataModel->preferences()->setFitToPage(KVSPrefs::EnumFitToPage::DontFit);
    disconnect(&fitTimer, SIGNAL(timeout()), this, SLOT(fitToHeight()));
  }
}


void LigaturePart::fitToPage()
{
  double z = qMin(pageView()->calculateFitToHeightZoomValue(), pageView()->calculateFitToWidthZoomValue());

  // Check if the methods returned usable values. Values that are not
  // within the limits indicate that fitting to width or height is
  // currently not possible (e.g. because no document is
  // loaded). In that case, we abort.
  if ((z < ZoomLimits::MinZoom/1000.0) || (z > ZoomLimits::MaxZoom/1000.0))
    return;

  setZoom(z);
  _zoomVal.setZoomFitPage(z);
}


void LigaturePart::fitToHeight()
{
  double z = pageView()->calculateFitToHeightZoomValue();

  // Check if the method returned a usable value. Values that are not
  // within the limits indicate that fitting to height is currently
  // not possible (e.g. because no document is loaded). In that case,
  // we abort.
  if ((z < ZoomLimits::MinZoom/1000.0) || (z > ZoomLimits::MaxZoom/1000.0))
    return;

  setZoom(z);
  _zoomVal.setZoomFitHeight(z);
}


void LigaturePart::fitToWidth()
{
  double z = pageView()->calculateFitToWidthZoomValue();

  // Check if the method returned a usable value. Values that are not
  // within the limits indicate that fitting to width is currently not
  // possible (e.g. because no document is loaded). In that case, we
  // abort.
  if ((z < ZoomLimits::MinZoom/1000.0) || (z > ZoomLimits::MaxZoom/1000.0))
    return;

  setZoom(z);
  _zoomVal.setZoomFitWidth(z);
}


void LigaturePart::setZoomValue(const QString &sval)
{
  if (sval == i18n("Fit to Page Width"))
  {
    fitWidthAct -> setChecked(true);
    fitToWidth();
  }
  else if (sval == i18n("Fit to Page Height"))
  {
    fitHeightAct -> setChecked(true);
    fitToHeight();
  }
  else if (sval == i18n("Fit to Page"))
  {
    fitPageAct -> setChecked(true);
    fitToPage();
  }
  else
  {
    disableZoomFit();

    float fval = _zoomVal.value();
    _zoomVal.setZoomValue(sval);
    if (fval != _zoomVal.value())
      _zoomVal.setZoomValue(setZoom(_zoomVal.value()));
  }

  mainWidget->setFocus();
}


void LigaturePart::zoomNamesChanged(const QStringList& list)
{
  zoom_action->setItems(list);
}


void LigaturePart::setCurrentZoomItem(int index)
{
  zoom_action->setCurrentItem(index);
}


void LigaturePart::checkActions()
{
  int numberOfPages = 0;
  int currentPage = 0;
  bool doc = false;

  if (!multiPage.isNull()) {
    currentPage = dataModel->currentPageNumber();
    numberOfPages = dataModel->numberOfPages();
    doc = !url().isEmpty();
  }

  if (dataModel->preferences()->viewMode() == KVSPrefs::EnumViewMode::Overview) {
    int visiblePages = dataModel->preferences()->overviewModeColumns() * dataModel->preferences()->overviewModeRows();

    // firstVisiblePage is the smallest currently shown pagenumber.
    int firstVisiblePage = currentPage - (currentPage % visiblePages);

    backAct->setEnabled(doc && currentPage >= visiblePages);
    forwardAct->setEnabled(doc && firstVisiblePage <= numberOfPages - visiblePages);

    startAct->setEnabled(doc && firstVisiblePage > 1);
    endAct->setEnabled(doc && firstVisiblePage + visiblePages < numberOfPages);
  } else {
    backAct->setEnabled(doc && currentPage > 1);
    forwardAct->setEnabled(doc && currentPage < numberOfPages);

    startAct->setEnabled(doc && currentPage > 1);
    endAct->setEnabled(doc && currentPage < numberOfPages);
  }

  gotoAct->setEnabled(doc && numberOfPages > 1);
  readDownAct->setEnabled(doc);
  readUpAct->setEnabled(doc);

  zoomInAct->setEnabled(doc);
  zoomOutAct->setEnabled(doc);

  fitPageAct->setEnabled(doc);
  fitHeightAct->setEnabled(doc);
  fitWidthAct->setEnabled(doc);

  rotateRightAction->setEnabled(doc);
  rotateLeftAction->setEnabled(doc);

  presentationAction->setEnabled(doc);
  media->setEnabled(doc);

  printAction->setEnabled(doc);

  saveAction->setEnabled(isModified());

  saveAsAction->setEnabled(doc);

  bool textSearch = false;
  if (!multiPage.isNull())
    if (doc && multiPage->supportsTextSearch())
      textSearch = true;
  exportTextAction->setEnabled(textSearch);
  findTextAction->setEnabled(textSearch);
  selectAllAction->setEnabled(textSearch);

  media->setEnabled(!multiPage.isNull() && !multiPage->hasSpecifiedPageSizes());
}


void LigaturePart::slotPrint()
{
#ifdef __GNUC__
#warning TODO: REMOVE THIS METHOD
#endif
  multiPage->print();
}


void LigaturePart::readSettings()
{
  showSidebar->setChecked(dataModel->preferences()->pageMarks());
  slotShowSidebar();

  watchAct->setChecked(dataModel->preferences()->watchFile());

  // Read zoom value. Even if 'fitToPage' has been set above, there is
  // no widget available right now, so setting a good default value
  // from the configuration file is perhaps not a bad idea.
  float _zoom = dataModel->preferences()->zoom();
  if ( (_zoom < ZoomLimits::MinZoom/1000.0) || (_zoom > ZoomLimits::MaxZoom/1000.0)) {
    kWarning(kvs::shell) << "Illeagal zoom value of " << _zoom*100.0 << "% found in the preferences file. Setting zoom to 100%." << endl;
    _zoom = 1.0;
  }
  _zoomVal.setZoomValue(setZoom(_zoom));

  // The value 'fitToPage' has several meanings: 1 is 'fit to page
  // width', 2 is 'fit to page height', 3 is 'fit to page'. Other
  // values indicate 'no fit to page'. Note: at the time this code is
  // executed, the methods fitToWidth(), etc., do not work well at all
  // (perhaps some data is not initialized yet)? For that reason, we
  // do not call these methods, and load the last zoom-value from the
  // configuration file below. The hope is that this value is not
  // terribly wrong. If the user doesn't like it, it suffices to
  // resize the window just a bit...
  switch(dataModel->preferences()->fitToPage()) {
  case KVSPrefs::EnumFitToPage::FitToPage:
    fitPageAct->setChecked(true);
    _zoomVal.setZoomFitPage(_zoom);
    enableFitToPage(true);
    break;
  case KVSPrefs::EnumFitToPage::FitToPageWidth:
    fitWidthAct->setChecked(true);
    _zoomVal.setZoomFitWidth(_zoom);
    enableFitToWidth(true);
    break;
  case KVSPrefs::EnumFitToPage::FitToPageHeight:
    fitHeightAct->setChecked(true);
    _zoomVal.setZoomFitHeight(_zoom);
    enableFitToHeight(true);
    break;
  }

  // Set the currently active tool
  switch(dataModel->preferences()->tool())
  {
    case KVSPrefs::EnumTool::Move:
      moveModeAction->setChecked(true);
      break;
    case KVSPrefs::EnumTool::Select:
      selectionModeAction->setChecked(true);
      break;
    case KVSPrefs::EnumTool::RectSelect:
      rectSelectToolAction->setChecked(true);
      break;
    case KVSPrefs::EnumTool::Ruler:
      rulerToolAction->setChecked(true);
      break;
    default:
      moveModeAction->setChecked(true);
  }

  // Check if scrollbars should be shown.
  bool sbstatus = dataModel->preferences()->scrollbars();
  scrollbarHandling->setChecked(sbstatus);

  // Check if ligature should guess a good viewmode when opening a new
  // document
  guessViewMode->setChecked(dataModel->preferences()->guessViewMode());

  if (!multiPage.isNull())
    multiPage->readSettings();
}


void LigaturePart::writeSettings()
{
  // if loading the KPart failed - just exit now
  //if (!showSidebar)
  //  return;

  if (!multiPage.isNull())
    multiPage->writeSettings();

  // Save TOC layout
  tableOfContents->writeSettings();

  dataModel->preferences()->setGuiLayout(splitterWidget->sizes());
  dataModel->preferences()->setContentsLayout(contentsSplitter->sizes());
  // Save state of the sidebar
  dataModel->preferences()->setSideBarItem(sideBar->currentIndex());

  dataModel->preferences()->setPageMarks(showSidebar->isChecked());
  dataModel->preferences()->setWatchFile(watchAct->isChecked());
  dataModel->preferences()->setZoom(_zoomVal.value());
  dataModel->preferences()->setPaperFormat(dataModel->defaultPageSize().serialize());
  dataModel->preferences()->setScrollbars(scrollbarHandling->isChecked());
  dataModel->preferences()->setGuessViewMode(guessViewMode->isChecked());

  if (fitPageAct->isChecked())
    dataModel->preferences()->setFitToPage(KVSPrefs::EnumFitToPage::FitToPage);
  else if(fitWidthAct->isChecked())
    dataModel->preferences()->setFitToPage(KVSPrefs::EnumFitToPage::FitToPageWidth);
  else if (fitHeightAct->isChecked())
    dataModel->preferences()->setFitToPage(KVSPrefs::EnumFitToPage::FitToPageHeight);
  else
    dataModel->preferences()->setFitToPage(KVSPrefs::EnumFitToPage::DontFit);

  dataModel->preferences()->writeConfig();
}


void LigaturePart::connectNotify ( const char *sig )
{
  if (QString(sig).contains("pageChanged"))
    pageChangeIsConnected = true;
}


void LigaturePart::setStatusBarTextFromMultiPage( const QString &msg )
{
  if (msg.isEmpty())
  {
    if (pageChangeIsConnected)
      emit setStatusBarText(QString::null);
    else
    {
      int currentPage = dataModel->currentPageNumber();
      int numberOfPages = dataModel->numberOfPages();
      emit setStatusBarText(i18n("Page %1 of %2", currentPage, numberOfPages));
    }
  }
  else
    emit setStatusBarText(msg);
}


KAboutData* LigaturePart::createAboutData()
{
  return new KAboutData("ligaturepart", I18N_NOOP("Document Viewer Part"),
                        "0.6", "" /* I18N_NOOP("") */,
                        KAboutData::License_GPL,
                        I18N_NOOP("Copyright (c) 2005 Wilfried Huss"));
}


void LigaturePart::doSettings()
{
  if (KConfigDialog::showDialog("ligature_config"))
    return;

  KConfigDialog* configDialog = new KConfigDialog(mainWidget, "ligature_config", dataModel->preferences());

  optionDialogGUIWidget* guiWidget = new optionDialogGUIWidget(mainWidget);
  configDialog->addPage(guiWidget, i18n("User Interface"), "view_choose", i18n("User Interface Options"));

  optionDialogAccessibilityWidget* accWidget = new optionDialogAccessibilityWidget(mainWidget);
  configDialog->addPage(accWidget, i18n("Accessibility"), "access", i18n("Options for Accessibilty Mode"));

  DlgPresentation* presentationWidget = new DlgPresentation(mainWidget);
  configDialog->addPage(presentationWidget, i18n("Presentation"), "kpresenter_kpr", i18n("Options for Presentation Mode"));

  if (multiPage)
    multiPage->addConfigDialogs(configDialog);

  connect(configDialog, SIGNAL(settingsChanged( const QString &)), this, SLOT(preferencesChanged()));
  configDialog->show();
}

void LigaturePart::preferencesChanged()
{
  markList()->slotShowThumbnails();
  renderModeChanged();
  if (multiPage)
    multiPage->preferencesChanged();
  dataModel->setViewMode(dataModel->preferences()->viewMode());
}

void LigaturePart::partActivateEvent( KParts::PartActivateEvent *ev )
{
  QApplication::sendEvent( multiPage, ev );
}


void LigaturePart::guiActivateEvent( KParts::GUIActivateEvent *ev )
{
  QApplication::sendEvent( multiPage, ev );
}


void LigaturePart::registerTools()
{
  emit switchTool(dataModel->preferences()->tool());

  KIconLoader* iconLoader = KIconLoader::global();
  iconLoader->addAppDir("ligaturepart");
  emit registerTool(KVSPrefs::EnumTool::Move, iconLoader->loadIcon("movetool", K3Icon::Small, K3Icon::SizeSmall));
  emit registerTool(KVSPrefs::EnumTool::Select, iconLoader->loadIcon("selectiontool", K3Icon::Small, K3Icon::SizeSmall));
  emit registerTool(KVSPrefs::EnumTool::RectSelect, iconLoader->loadIcon("frame_edit", K3Icon::Small, K3Icon::SizeSmall));
  emit registerTool(KVSPrefs::EnumTool::Ruler, iconLoader->loadIcon("measuretool", K3Icon::Small, K3Icon::SizeSmall));
}

void LigaturePart::slotEnableMoveTool()
{
  // Safety Check
  if (multiPage.isNull())
    return;

  kDebug(kvs::shell) << "enable move tool" << endl;

  dataModel->preferences()->setTool(KVSPrefs::EnumTool::Move);
  emit switchTool(KVSPrefs::EnumTool::Move);
}


void LigaturePart::slotEnableRectSelectionTool()
{
  if (multiPage.isNull())
    return;

  kDebug(kvs::shell) << "enable rectangle selection tool" << endl;

  dataModel->preferences()->setTool(KVSPrefs::EnumTool::RectSelect);
  emit switchTool(KVSPrefs::EnumTool::RectSelect);
}


void LigaturePart::slotEnableSelectionTool()
{
  if (multiPage.isNull())
    return;

  kDebug(kvs::shell) << "enable selection tool" << endl;

  dataModel->preferences()->setTool(KVSPrefs::EnumTool::Select);
  emit switchTool(KVSPrefs::EnumTool::Select);
}


void LigaturePart::slotEnableRulerTool()
{
  if (multiPage.isNull())
    return;

  kDebug(kvs::shell) << "enable ruler tool" << endl;

  dataModel->preferences()->setTool(KVSPrefs::EnumTool::Ruler);
  emit switchTool(KVSPrefs::EnumTool::Ruler);
}


void LigaturePart::slotNextViewMode()
{
  int viewmode = dataModel->preferences()->viewMode();
  int newmode;
  switch (viewmode)
  {
    case KVSPrefs::EnumViewMode::SinglePage:
      newmode = KVSPrefs::EnumViewMode::Continuous;
      break;
    case KVSPrefs::EnumViewMode::Continuous:
      newmode = KVSPrefs::EnumViewMode::ContinuousFacing;
      break;
    case KVSPrefs::EnumViewMode::ContinuousFacing:
      newmode = KVSPrefs::EnumViewMode::Overview;
      break;
    default:
    case KVSPrefs::EnumViewMode::Overview:
      newmode = KVSPrefs::EnumViewMode::SinglePage;
      break;
  }
  dataModel->setViewMode(newmode);
}


void LigaturePart::slotPreviousViewMode()
{
  int viewmode = dataModel->preferences()->viewMode();
  int newmode;
  switch (viewmode)
  {
    case KVSPrefs::EnumViewMode::SinglePage:
      newmode = KVSPrefs::EnumViewMode::Overview;
      break;
    case KVSPrefs::EnumViewMode::Continuous:
      newmode = KVSPrefs::EnumViewMode::SinglePage;
      break;
    case KVSPrefs::EnumViewMode::ContinuousFacing:
      newmode = KVSPrefs::EnumViewMode::Continuous;
      break;
    default:
    case KVSPrefs::EnumViewMode::Overview:
      newmode = KVSPrefs::EnumViewMode::ContinuousFacing;
      break;
  }
  dataModel->setViewMode(newmode);
}


void LigaturePart::setViewMode()
{
  // Temporarely disable smooth scrolling during the viewmode change
  bool smooth = dataModel->preferences()->smoothScrolling();
  dataModel->preferences()->setSmoothScrolling(false);

  int viewmode = dataModel->preferences()->viewMode();
  viewModeAction->setCurrentItem(viewmode);
  pageView()->setViewMode();
  updateZoomLevel();

  dataModel->preferences()->setSmoothScrolling(smooth);
}


void LigaturePart::setModified()
{
 if (multiPage.isNull()) {
    kError(kvs::shell) << "LigaturePart::setModified() called with multiPage == NULL" << endl;
    return;
  }
  if (multiPage->getRenderer().isNull())
    return;

  dataModel->deselectText();
  dataModel->history()->clear();
  pageCache->clear();

  //TODO: test if this is really needed.
  pageView()->generateDocumentWidgets();

  // Set Table of Contents
  tableOfContents->setContents(multiPage->getRenderer()->getBookmarks());

  ReadWritePart::setModified();
  checkActions();
}


LigaturePartExtension::LigaturePartExtension(LigaturePart *parent)
  : KParts::BrowserExtension( parent)
{
}

bool LigaturePart::saveFile()
{
  if (multiPage.isNull()) {
    kError(kvs::shell) << "LigaturePart::saveFile: called with multiPage == NULL" << endl;
    return false;
  }
  return multiPage->slotSave(m_file);
}


void LigaturePart::slotSaveAs()
{
  if (multiPage.isNull()) {
    kError(kvs::shell) << "LigaturePart::slotSaveAs: called with multiPage == NULL" << endl;
    return;
  }

  QString fileName = KFileDialog::getSaveFileName(KUrl(), mimeTypeFilter, mainWidget, i18n("Save File As"));

  if (QFile(fileName).exists()) {
    int r = KMessageBox::warningContinueCancel (0, i18n("<qt>The file %1 exists. Do you want to overwrite that file?</qt>", fileName),
                                                i18n("Overwrite File"), KGuiItem(i18n("Overwrite")));
    if (r == KMessageBox::Cancel)
      return;
  }

  KUrl url;
  url.setPath(fileName);
  saveAs(url);
}


void LigaturePart::slotShowPresentation()
{
  if (!presentationWidget)
  {
    presentationWidget = new PresentationWidget(pageView(), pageCache);
    presentationWidget->setupObservers(dataModel);
    connect(presentationWidget, SIGNAL(httpLink(const Hyperlink&)), this, SLOT(handleLocalLink(const Hyperlink&)));
  }
}


bool LigaturePart::autoZoom() const
{
  return (dataModel->preferences()->fitToPage() != KVSPrefs::EnumFitToPage::DontFit);
}


void LigaturePart::slotRotateRight()
{
  bool rememberSmoothMove = dataModel->preferences()->smoothScrolling();
  dataModel->preferences()->setSmoothScrolling(false);

  switch (dataModel->preferences()->rotation()) {
  case KVSPrefs::EnumRotation::Portrait:
    dataModel->preferences()->setRotation(KVSPrefs::EnumRotation::Landscape);
    break;
  case KVSPrefs::EnumRotation::Landscape:
    dataModel->preferences()->setRotation(KVSPrefs::EnumRotation::Upsidedown);
    break;
  case KVSPrefs::EnumRotation::Upsidedown:
    dataModel->preferences()->setRotation(KVSPrefs::EnumRotation::Seascape);
    break;
  case KVSPrefs::EnumRotation::Seascape:
    dataModel->preferences()->setRotation(KVSPrefs::EnumRotation::Portrait);
    break;
  }

  PageNumber _currentPage = dataModel->currentPageNumber();
  if (autoZoom())
  {
    // Update zoomlevel in autozoom modes
    slotStartFitTimer();
  }
  else
  {
    // Update widget sizes immediately
    pageView()->resizeWidgets();
  }
  markList()->rebuildThumbnailWidgets();

  // return to the privious position in the document
  dataModel->setCurrentPageNumber(_currentPage);
  dataModel->preferences()->setSmoothScrolling(rememberSmoothMove);
}


void LigaturePart::slotRotateLeft()
{
  bool rememberSmoothMove = dataModel->preferences()->smoothScrolling();
  dataModel->preferences()->setSmoothScrolling(false);

  switch (dataModel->preferences()->rotation()) {
  case KVSPrefs::EnumRotation::Portrait:
    dataModel->preferences()->setRotation(KVSPrefs::EnumRotation::Seascape);
    break;
  case KVSPrefs::EnumRotation::Landscape:
    dataModel->preferences()->setRotation(KVSPrefs::EnumRotation::Portrait);
    break;
  case KVSPrefs::EnumRotation::Upsidedown:
    dataModel->preferences()->setRotation(KVSPrefs::EnumRotation::Landscape);
    break;
  case KVSPrefs::EnumRotation::Seascape:
    dataModel->preferences()->setRotation(KVSPrefs::EnumRotation::Upsidedown);
    break;
  }

  PageNumber _currentPage = dataModel->currentPageNumber();
  if (autoZoom())
  {
    // Update zoomlevel in autozoom modes
    slotStartFitTimer();
  }
  else
  {
    // Update widget sizes immediately
    pageView()->resizeWidgets();
  }
  markList()->rebuildThumbnailWidgets();

  // return to the privious position in the document
  dataModel->setCurrentPageNumber(_currentPage);
  dataModel->preferences()->setSmoothScrolling(rememberSmoothMove);
}


void LigaturePart::renderModeChanged(bool clearCache)
{
  if (clearCache)
    pageCache->clear();

  pageView()->repaintPages();
  markList()->rebuildThumbnailWidgets();
}

//TODO: split into several functions
void LigaturePart::setCurrentPageNumber()
{
#ifdef DEBUG_KMULTIPAGE
  kDebug(kvs::shell) << "ligaturePluginGUI::setCurrentPageNumber()" << endl;
#endif

  if (multiPage.isNull() || dataModel->numberOfPages() == 0)
  {
    if (pageChangeIsConnected)
    {
      emit pageChanged("");
      emit sizeChanged("");
    }
    else
      emit setStatusBarText("");

    return;
  }

  updateZoomLevel();

  // ATTN: The string here must be the same as in setPage() below
  QString pageString = i18n("Page %1 of %2", dataModel->currentPageNumber(), dataModel->numberOfPages());
  if (pageChangeIsConnected) {
    emit pageChanged(pageString);
    pageSize _pageSize(pageCache->sizeOfPage(dataModel->currentPageNumber()));
    emit sizeChanged(_pageSize.description());
  } else
    emit setStatusBarText(pageString);

  checkActions();
}


void LigaturePart::slotShowSidebar(bool show)
{
  if (sideBar == 0)
    return;

  if (show)
    sideBar->show();
  else
    sideBar->hide();
}


void LigaturePart::doExportText()
{
  if (multiPage.isNull()) {
    kError(kvs::shell) << "LigaturePart::scroll() called with multipage == 0" << endl;
    return;
  }
  // Generate a suggestion for a reasonable file name
  QString suggestedName = url().path();
  suggestedName = suggestedName.left(suggestedName.indexOf(".")) + ".txt";

  delete exportDialog;
  exportDialog = new ExportDialog(suggestedName, i18n("*.txt|Plain Text (Latin 1) (*.txt)"),
    pageView(), dataModel->selectedPages(), dataModel->numberOfPages());

  connect(exportDialog, SIGNAL(exportPages(const QString&, const QList<PageNumber>&)),
          this, SLOT(slotExportText(const QString&, const QList<PageNumber>&)));
}


void LigaturePart::handleLocalLink(const Hyperlink& link)
{
#ifdef DEBUG_SPECIAL
  kDebug(kvs::shell) << "hit: local link to " << link.linkText << endl;
#endif

  if (multiPage.isNull()) {
    kError(kvs::shell) << "ligaturePluginGUI::handleLocalLink( ... ) called, but multiPage==0" << endl;
    return;
  }

  if (link.anchor.isValid())
    dataModel->setCurrentPageNumber(link.anchor);
  else {
    if (link.linkText[0] != '#' ) {
      // We could in principle use KIO::Netaccess::run() here, but
      // it is perhaps not a very good idea to allow a DVI-file to
      // specify arbitrary commands, such as "rm -rvf /". Using
      // the kfmclient seems to be MUCH safer.
      QUrl Link_Url(m_file);
      Link_Url.setFragment(link.linkText);
      KToolInvocation::invokeBrowser(Link_Url.toString());
    }else
      kError(kvs::shell) << "ligaturePluginGUI::handleLocalLink( ... ) with link to " << link << endl;
  }
}


double LigaturePart::setZoom(double zoom)
{
#ifdef DEBUG_KMULTIPAGE
  kDebug(kvs::shell) << "ligaturePluginGUI::setZoom(" << zoom << ")" << endl;
#endif

  if (zoom < ZoomLimits::MinZoom/1000.0)
    zoom = ZoomLimits::MinZoom/1000.0;
  if (zoom > ZoomLimits::MaxZoom/1000.0)
    zoom = ZoomLimits::MaxZoom/1000.0;

  dataModel->setResolution(mainWidget->logicalDpiX()*zoom);
  return zoom;
}


void LigaturePart::findNextText()
{
  if (multiPage.isNull()) {
    kError(kvs::shell) << "LigaturePart::findNextText() called with multiPage == NULL" << endl;
    return;
  }

#ifdef KDVI_MULTIPAGE_DEBUG
  kDebug(kvs::shell) << "ligaturePluginGUI::findNextText() called" << endl;
#endif

  searchInProgress = true;

  // Used to remember if the documentPage we use is from the cache.
  // If not we need to delete it manually to avoid a memory leak.
  bool cachedPage = false;

  QString searchText = searchWidget->getText();

  if (searchText.isEmpty())
  {
    kError(kvs::shell) << "ligaturePluginGUI::findNextText() called when search text was empty" << endl;
    return;
  }

  bool case_sensitive = searchWidget->caseSensitive();

  // Find the page and text position on the page where the search will
  // start. If nothing is selected, we start at the beginning of the
  // current page. Otherwise, start after the selected text.  TODO:
  // Optimize this to get a better 'user feeling'
  quint16 startingPage;
  quint16 startingTextItem;

  TextSelection userSelection = dataModel->selectedText();
  if (userSelection.isEmpty())
  {
    startingPage     = dataModel->currentPageNumber();
    startingTextItem = 0;
  }
  else
  {
    startingPage     = userSelection.getPageNumber();
    startingTextItem = userSelection.getSelectedTextEnd()+1;
  }

  TextSelection foundSelection;

  RenderedDocumentPagePixmap* searchPage = 0;

  for(unsigned int i = 0; i < dataModel->numberOfPages(); i++)
  {
    unsigned int pageNumber = (i + startingPage - 1) % dataModel->numberOfPages() + 1;

    if (!searchInProgress)
    {
      // Interrupt the search
      setStatusBarText(i18n("Search interrupted"));
      if (!cachedPage)
        delete searchPage;
      return;
    }

    if (i != 0)
    {
      setStatusBarText(i18n("Search page %1 of %2", pageNumber, dataModel->numberOfPages()));
      qApp->processEvents();
    }

    // Check if we already have a rendered version of the page in the cache. As we are only interested in the
    // text we don't care about the page size.
    if (pageCache->isPageCached(pageNumber))
    {
      // If the last search page used was created locally, we need to destroy it
      if (!cachedPage)
        delete searchPage;

      searchPage = pageCache->getPage(pageNumber, false);
      cachedPage = true;
    }
    else
    {
      // If the page is not in the cache we draw a small version of it, since this is faster.

      // We only create a new searchPage if we need to, otherwise reuse the existing one.
      JobId id(pageNumber, 0.0, 0, false);

      cachedPage = false;
      searchPage = multiPage->getRenderer()->getText(id);
    }

    // If there is no text in the current page, try the next one.
    if (!searchPage || searchPage->textBoxList.size() == 0)
      continue;

    foundSelection = searchPage->find(searchText, startingTextItem, case_sensitive);

    if (foundSelection.isEmpty())
    {
      // In the next page, start search again at the beginning.
      startingTextItem = 0;
      clearSelection();

      if (pageNumber == dataModel->numberOfPages())
      {
        int answ = KMessageBox::questionYesNo(pageView(),
                   i18n("<qt>The search string <strong>%1</strong> could not be found by the "
                        "end of the document. Should the search be restarted from the beginning "
                        "of the document?</qt>", searchText),
                   i18n("Text Not Found"), KStandardGuiItem::cont(), KStandardGuiItem::cancel());

        if (answ != KMessageBox::Yes)
        {
          setStatusBarText(QString::null);
          searchInProgress = false;
          if (!cachedPage)
            delete searchPage;
          return;
        }
      }
    }
    else
    {
      dataModel->selectText(foundSelection);
      pageView()->gotoSelection(dataModel->selectedText());
      setStatusBarText(QString::null);
      searchInProgress = false;
      if (!cachedPage)
        delete searchPage;
      return;
    }
  }

  KMessageBox::sorry(pageView(), i18n("<qt>The search string <strong>%1</strong> could not be found.</qt>", searchText));
  setStatusBarText(QString::null);
  searchInProgress = false;
  if (!cachedPage)
    delete searchPage;
}


void LigaturePart::findPrevText()
{
  if (multiPage.isNull()) {
    kError(kvs::shell) << "LigaturePart::findPrevText() called with multiPage == NULL" << endl;
    return;
  }

#ifdef KDVI_MULTIPAGE_DEBUG
  kDebug(kvs::shell) << "ligaturePluginGUI::findPrevText() called" << endl;
#endif

  searchInProgress = true;

  // Used to remember if the documentPage we use is from the cache.
  // If not we need to delete it manually to avoid a memory leak.
  bool cachedPage = false;

  QString searchText = searchWidget->getText();

  if (searchText.isEmpty())
  {
    kError(kvs::shell) << "ligaturePluginGUI::findPrevText() called when search text was empty" << endl;
    return;
  }

  bool case_sensitive = searchWidget->caseSensitive();

  // Find the page and text position on the page where the search will
  // start. If nothing is selected, we start at the beginning of the
  // current page. Otherwise, start after the selected text.  TODO:
  // Optimize this to get a better 'user feeling'
  unsigned int startingPage;
  int startingTextItem;

  TextSelection userSelection = dataModel->selectedText();
  if (userSelection.isEmpty())
  {
    startingPage     = dataModel->currentPageNumber();
    startingTextItem = -1;
  }
  else
  {
    startingPage     = userSelection.getPageNumber();
    startingTextItem = userSelection.getSelectedTextStart()-1;
  }

  TextSelection foundSelection;

  RenderedDocumentPagePixmap* searchPage = 0;

  for(unsigned int i = 0; i < dataModel->numberOfPages(); i++)
  {
    int pageNumber = startingPage - i;
    if (pageNumber <= 0)
      pageNumber += dataModel->numberOfPages();

    if (!searchInProgress)
    {
      // Interrupt the search
      setStatusBarText(i18n("Search interrupted"));
      if (!cachedPage)
        delete searchPage;
      return;
    }

    if (i != 0)
    {
      setStatusBarText(i18n("Search page %1 of %2", pageNumber, dataModel->numberOfPages()));
      qApp->processEvents();
    }

    // Check if we already have a rendered version of the page in the cache. As we are only interested in the
    // text we don't care about the page size.
    if (pageCache->isPageCached(pageNumber))
    {
      // If the last search page used was created locally, we need to destroy it
      if (!cachedPage)
        delete searchPage;

      searchPage = pageCache->getPage(pageNumber, false);
      cachedPage = true;
    }
    else
    {
      // If the page is not in the cache we draw a small version of it, since this is faster.

      // We only create a new searchPage if we need to, otherwise reuse the existing one.
      JobId id(pageNumber, 0.0, 0, false);

      cachedPage = false;

      searchPage = multiPage->getRenderer()->getText(id);
    }

    // If there is no text in the current page, try the next one.
    if (!searchPage || searchPage->textBoxList.size() == 0)
      continue;

    foundSelection = searchPage->findRev(searchText, startingTextItem, case_sensitive);

    if (foundSelection.isEmpty())
    {
      // In the next page, start search again at the beginning.
      startingTextItem = -1;
      clearSelection();

      if (pageNumber == 1)
      {
        int answ = KMessageBox::questionYesNo(pageView(),
                  i18n("<qt>The search string <strong>%1</strong> could not be found by the "
                        "beginning of the document. Should the search be restarted from the end "
                        "of the document?</qt>", searchText),
                  i18n("Text Not Found"), KStandardGuiItem::cont(), KStandardGuiItem::cancel());

        if (answ != KMessageBox::Yes)
        {
          setStatusBarText(QString::null);
          searchInProgress = false;
          if (!cachedPage)
            delete searchPage;
          return;
        }
      }
    }
    else
    {
      dataModel->selectText(foundSelection);
      pageView()->gotoSelection(dataModel->selectedText());
      setStatusBarText(QString::null);
      searchInProgress = false;
      if (!cachedPage)
        delete searchPage;
      return;
    }
  }

  KMessageBox::sorry(pageView(), i18n("<qt>The search string <strong>%1</strong> could not be found.</qt>", searchText));
  setStatusBarText(QString::null);
  searchInProgress = false;
  if (!cachedPage)
    delete searchPage;
}


void LigaturePart::clearSelection()
{
  PageNumber page = dataModel->selectedText().getPageNumber();

  if (!page.isValid())
    return;

  // Clear selection
  dataModel->deselectText();

  // Now we need to update the widget which contained the selection
  pageView()->updatePage(page);
}


void LigaturePart::showFindTextDialog()
{
  if (multiPage.isNull()) {
    kError(kvs::shell) << "LigaturePart::showFindTextDialog() called with multiPage == NULL" << endl;
    return;
  }
  if (multiPage->getRenderer().isNull()) {
    kError(kvs::shell) << "LigaturePart::showFindTextDialog() called with renderer == NULL" << endl;
    return;
  }

  if (!multiPage->getRenderer()->supportsTextSearch())
    return;

  searchWidget->show();
  searchWidget->setFocus();
}


void LigaturePart::stopSearch()
{
  if (searchInProgress)
  {
    // stop the search
    searchInProgress = false;
  }
  else
    searchWidget->hide();
}


void LigaturePart::copyText()
{
  dataModel->copyText();
}


void LigaturePart::doSelectAll()
{
  switch(pageView()->numberOfWidgets()) {
  case 0:
    kError(kvs::shell) << "ligaturePluginGUI::doSelectAll() while widgetList is empty" << endl;
    break;
  case 1:
    pageView()->pageWidget(0)->selectAll();
    break;
  default:
    if (pageView()->numberOfWidgets() < dataModel->currentPageNumber())
      kError(kvs::shell) << "ligaturePluginGUI::doSelectAll() while widgetList.size()=" << pageView()->numberOfWidgets() << "and currentPageNumber()=" << dataModel->currentPageNumber() << endl;
    else
      pageView()->pageWidget(dataModel->currentPageNumber()-1)->selectAll();
  }
}


void LigaturePart::jumpToReference(const QString& reference)
{
  kDebug(kvs::shell) << "LigaturePart::jumpToReference(" << reference << ") called" << endl;

  if (multiPage.isNull()) {
    kError(kvs::shell) << "LigaturePart::jumpToReference() called with multiPage == NULL" << endl;
    return;
  }
  if (multiPage->getRenderer().isNull()) {
    kError(kvs::shell) << "LigaturePart::jumpToReference() called with renderer == NULL" << endl;
    return;
  }

  dataModel->setCurrentPageNumber(multiPage->getRenderer()->parseReference(reference));
}


void LigaturePart::reload()
{
#ifdef KMULTIPAGE_DEBUG
  kDebug(kvs::shell) << "Reload file " << m_file << endl;
#endif

  if (multiPage.isNull()) {
    kError(kvs::shell) << "LigaturePart::reload() called with multiPage == NULL" << endl;
    return;
  }
  if (multiPage->getRenderer().isNull()) {
    kError(kvs::shell) << "ligaturePluginGUI::reload() called, but no renderer was set" << endl;
    return;
  }

  if (multiPage->getRenderer()->isValidFile(m_file)) {
    pageCache->clear();
    dataModel->deselectText();
    dataModel->history()->clear();
    emit setStatusBarText(i18n("Reloading file %1", m_file));
    PageNumber pg = dataModel->currentPageNumber();

    killTimer(timer_id);
    timer_id = -1;
    bool r = multiPage->getRenderer()->setFile(m_file, m_url);

    // Set Table of Contents
    tableOfContents->setContents(multiPage->getRenderer()->getBookmarks());

    dataModel->setCurrentPageNumber(pg);
    dataModel->setNumberOfPages(multiPage->getRenderer()->totalPages());
    multiPage->setFile(r);
    emit setStatusBarText(QString::null);

    renderModeChanged();
  } else {
    if (timer_id == -1)
      timer_id = startTimer(1000);
  }
}


void LigaturePart::timerEvent( QTimerEvent * )
{
#ifdef KMULTIPAGE_DEBUG
  kDebug(kvs::shell) << "Timer Event " << endl;
#endif
  reload();
}


void LigaturePart::slotExportText(const QString& filename, const QList<PageNumber>& exportedPages)
{
  if (multiPage.isNull()) {
    kError(kvs::shell) << "LigaturePart::reload() called with multiPage == NULL" << endl;
    return;
  }
  if (multiPage->getRenderer().isNull()) {
    kError(kvs::shell) << "ligaturePluginGUI::reload() called, but no renderer was set" << endl;
    return;
  }

  delete exportDialog;
  exportDialog = 0;

  QFile textFile(filename);
  textFile.open(QIODevice::WriteOnly);
  QTextStream stream(&textFile);

  RenderedDocumentPagePixmap* dummyPage = 0;

  QProgressDialog progress(i18n("Exporting to text..."), i18n("Abort"), 0, exportedPages.count(), pageView());
  progress.setMinimumDuration(300);

  for(int i = 0; i < exportedPages.count(); i++)
  {
    PageNumber page = exportedPages[i];
    progress.setValue(i);
    qApp->processEvents();

    if (progress.wasCanceled())
      break;

    JobId id(page, 0.0, 0, false);
    // We gracefully ignore any errors (bad file, etc.)
    dummyPage = multiPage->getRenderer()->getText(id);

    for(int j=0; j < dummyPage->textBoxList.size(); j++)
    {
      // We try to detect newlines
      if (j > 0)
      {
        // Like all our textalgorithmns this currently assumes left to right text.
#ifdef __GNUC__
#warning TODO: make this more generic. But we first would need to guess the corrent orientation.
#endif
        if (dummyPage->textBoxList[j].box.top() > dummyPage->textBoxList[j-1].box.bottom() &&
            dummyPage->textBoxList[j].box.x() < dummyPage->textBoxList[j-1].box.x())
        {
          stream << "\n";
        }
      }
      stream << dummyPage->textBoxList[j].text;
    }

    // Send newline after each page.
    stream << "\n";
  }

  delete dummyPage;

  // Switch off the progress dialog, etc.
  progress.setValue(dataModel->numberOfPages());
  return;
}


void LigaturePart::textSelected(bool selected)
{
  copyTextAction->setEnabled(selected);
  deselectAction->setEnabled(selected);
}


#include "ligaturepart.moc"
