// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-
#ifndef LIGATUREPART_H
#define LIGATUREPART_H


#include <kparts/browserextension.h>
#include <QTimer>

#include "dataView.h"
#include "documentPageCache.h"
#include "history.h"
#include "hyperlink.h"
#include "marklist.h"
#include "pageSize.h"
#include "pageNumber.h"
#include "pageView.h"
#include "zoom.h"


class BookmarkList;
class ExportDialog;
class KAboutData;
class QAction;
class KConfig;
class KDirWatch;
class KComponentData;
class ligaturePluginGUI;
class KSelectAction;
class KToggleAction;
class KUrl;
class LigaturePartExtension;
class pageSizeDialog;
class PresentationWidget;
class QHBoxLayout;
class QSize;
class QSplitter;
class QToolBox;
class KTemporaryFile;
class SearchWidget;
class TableOfContents;

class LigaturePart: public KParts::ReadWritePart
{
  Q_OBJECT
  Q_CLASSINFO("Version", "Version 1 for KDE 4")

public:

  LigaturePart(QWidget* parentWidget, QObject* parent, const QStringList& args = QStringList());
  virtual ~LigaturePart();

  /** Methods which are associated with the DCOP functionality of the
     ligaturePluginGUI. This method can be implemented by the multipage,
     e.g. to jump to a certain location. */
  virtual void jumpToReference(const QString& reference);

  static KAboutData* createAboutData();

  /* Returns a list of mimetypes supported by the installed Ligature plugins. */
  virtual QStringList supportedMimeTypes();

  /** \brief saving to file

  Implementation of the purely virtual function defined in
  KParts::KReadWritePart */
  virtual bool saveFile();

signals:
  void zoomChanged(const QString &);
  void pageChanged(const QString &);
  void sizeChanged(const QString &);
  void fileOpened();

  /** This is emmited by @see registerTools() to inform the ligature
      which icons it should add as tool indicators to the statusbar. */
  void registerTool(int, QPixmap);

  /** This is emitted when the active tool changes. The parameter is the
      id of the new tool, as defined by the KVSPrefs object. */
  void switchTool(int);

public slots:
  void slotExportText(const QString& filename, const QList<PageNumber>&);

  /** This method may be called after the text search dialog
     'findDialog' has been set up, and the user has entered a search
     phrase. The method searches for the next occurence of the text,
     starting from the beginning of the current page, or after the
     currently selected text, if there is any. */
  void findNextText();

  /** This method may be called after the text search dialog
     'findDialog' has been set up, and the user has entered a search
     phrase. The method searches for the next occurence of the text,
     starting from the end of the current page, or before the
     currently selected text, if there is any. */
  void findPrevText();

  virtual void slotSetFullPage(bool fullpage);

  void slotFileOpen();

  /** Re-implementation of the closeUrl() method from the
      ReadWritePart */
  virtual bool closeUrl();

  void restoreDocument(const KUrl &url, int page);
  void saveDocumentRestoreInfo(KConfig* config);

private slots:
  void setStatusBarTextFromMultiPage(const QString &);

  void slotShowSidebar();
  void slotMedia();

  void goToPage();

  void disableZoomFit();
  void updateZoomLevel();

  void enableFitToPage(bool);
  void enableFitToHeight(bool);
  void enableFitToWidth(bool);

  void fitToPage();
  void fitToHeight();
  void fitToWidth();

  void slotPrint();

  void fileChanged(const QString&);

  // Connected to the QLineEdit in the toolbar.
  void setZoomValue(const QString &);

private:
  /** @returns true if one of the auto zoom modes (like fit to Page) is active. */
  bool autoZoom() const;

  KToggleAction *showSidebar, *scrollbarHandling;
  KSelectAction *zoom_action;

  PageView* pageView() { return _pageView; }

  MarkList* markList() { return _markList; }

  /** Re-implementation of the openFile() method from the
      ReadWritePart */
  virtual bool openFile();

  QStringList fileFormats() const;

  void connectNotify ( const char * );

  void partActivateEvent( KParts::PartActivateEvent *ev );
  void guiActivateEvent( KParts::GUIActivateEvent *ev );

private slots:
  void setCurrentPageNumber();

  // Add tool icons to the statusbar.
  void registerTools();

  void checkActions();

  void slotStartFitTimer();

  void doSettings();
  void preferencesChanged();

  void slotEnableMoveTool();
  void slotEnableSelectionTool();
  void slotEnableRectSelectionTool();
  void slotEnableRulerTool();

  void slotNextViewMode();
  void slotPreviousViewMode();

  void setViewMode();

  void textSelected(bool selected);

  /** Re-implementation of the setModified() method from the
      ReadWritePart

  This method also checks the actions and calles checkActions().
  */
  virtual void setModified();

  void slotSaveAs();

  /** Exports the document to a plain text file. */
  void doExportText();

  void slotShowPresentation();
  void slotRotateRight();
  void slotRotateLeft();

  void zoomIn();
  void zoomOut();

private:
  DataModel *dataModel;

  // This is set to true while a text search is performed.
  // If set to false the search is interrupted.
  bool searchInProgress;

  // This method reads the configuration file. It should only be
  // called when no document is loaded.
  void readSettings();
  void writeSettings();

  // The method openFile of this ligaturepart can be called even if
  // m_file points to a compressed file. In that case, the temporary
  // file tmpUnzipped will be created and a decompressed copy of the
  // file stored there.
  KTemporaryFile *tmpUnzipped;

  KDirWatch *watch;

  QAction *media;
  QAction* zoomInAct;
  QAction* zoomOutAct;
  QAction* backAct;
  QAction* forwardAct;
  QAction* startAct;
  QAction* endAct;
  QAction* gotoAct;
  QAction* saveAction;
  QAction* saveAsAction;
  QAction* printAction;
  QAction* readUpAct;
  QAction* readDownAct;
  QAction* backAction;
  QAction* forwardAction;
  QAction* settingsAction;

  KToggleAction* watchAct;
  KToggleAction* guessViewMode;
  KToggleAction* fitPageAct;
  KToggleAction* fitHeightAct;
  KToggleAction* fitWidthAct;

  KSelectAction* viewModeAction;

  QAction* presentationAction;

  KToggleAction* moveModeAction;
  KToggleAction* selectionModeAction;
  KToggleAction* rectSelectToolAction;
  KToggleAction* rulerToolAction;

  QAction* exportTextAction;

  QAction* copyTextAction;
  QAction* selectAllAction;
  QAction* deselectAction;
  QAction* findTextAction;
  QAction* findNextTextAction;
  QAction* findPrevAction;
  QAction* findNextAction;

  QAction* rotateRightAction;
  QAction* rotateLeftAction;

  QPointer<ligaturePluginGUI> multiPage;

  // Name of the library of the currently loaded multiPage.
  // Is used to check if it is really necessary to load a new MultiPage.
  QString multiPageLibrary;

  LigaturePartExtension *m_extension;

  bool pageChangeIsConnected;

  QWidget *mainWidget;
  QHBoxLayout* mainLayout;

  /** stores the current zoom value */
  Zoom _zoomVal;
  pageSizeDialog *_pageSizeDialog;

  QTimer fitTimer;

  ExportDialog* exportDialog;

  //=============
  QSplitter* splitterWidget;
  QSplitter* contentsSplitter;
  PageView *_pageView;
  SearchWidget* searchWidget;
  QToolBox* sideBar;

  MarkList* _markList;
  TableOfContents* tableOfContents;
  BookmarkList* bookmarkList;

  QPointer<PresentationWidget> presentationWidget;

  /* The pageCache caches already drawn "documentPages" and invokes
     the renderer if the needed page is not available in the cache. */
  DocumentPageCache *pageCache;


  /* For internal use by the reload()-method. See the comments in
     ligaturePluginGUI.cpp, right before the timerEvent function. */
  int  timer_id;

  /* This contains the mime type filter of the current file, in the format
     supported by KFileDialog.

     see KFileDialog::setFilter() for details. */
  QString mimeTypeFilter;

 private slots:
  /** For internal use the reload()-method. This is a dreadful
     hack. The problem we adress with this timer event is the
     following: the ligature has a KDirWatch object which looks at
     the DVI file and calls reload() when the object has changed. That
     works very nicely in principle, but in practise, when TeX runs
     for several seconds over a complicated file, this does not work
     at all. First, most of the time, while TeX is still writing, the
     file is invalid. Thus, reload() is very often called when the DVI
     file is bad. We solve this problem by checking the file first. If
     the file is bad, we do not reload. Second, when the file finally
     becomes good, it very often happens that KDirWatch does not
     notify us anymore. Whether this is a bug or a side effect of a
     feature of KDirWatch, I dare not say. We remedy that problem by
     using a timer: when reload() was called on a bad file, we
     automatically come back (via the timerEvent() function) every
     second and check if the file became good. If so, we stop the
     timer. It may well happen that KDirWatch calls us several times
     while we are waiting for the file to become good, but that does
     not do any harm. */
  void timerEvent( QTimerEvent *e );

  /** This methos is similar to openFile(). It is used when the "Watch
     file" option is activated, and the file has changed on disk. It
     differs from openFile() in two aspects: first, the file is
     checked for validity with DVIRenderer.isValidFile(m_file) if the
     file is invalid, a timer event is used to call the method again
     after a brief pause. Secondly, when the GUI is updated, the
     implementation does not jump to the first page, but tries to keep
     the current page. */
  virtual void reload();

  void doSelectAll();

  /** Shows the "text search" dialog, if text search is supported by
     the renderer. Otherwise, the method returns immediately. */
  void showFindTextDialog();

  void copyText();

  /** Interrupts a search if one is currently performed, otherwise
     hide the search panel */
  void stopSearch();

  void clearSelection();

  /** sets a zoom factor. The multipage implementation might refuse to
     use a given zoom factor, even if it falls within the bounds given
     by the constants MinZoom and MaxZoom which are defined in
     zoomlimits.h. In that case, the multipage implementation chooses a
     different zomm factor. The implementation returns the factor which
     has actually been used. A default implementation is provided. */
  double setZoom(double z);

  /** Update the items of the zoom combobox in the toolbar. */
  void zoomNamesChanged(const QStringList&);

  /** Set the current item of the zoom combobox in the toolbar. */
  void setCurrentZoomItem(int index);

  void handleLocalLink(const Hyperlink& link);

  /** Show or hide the sidebar widget. */
  void slotShowSidebar(bool);

  /** Initialize/Update PageWidgets, thumbnail list and bookmark list

  This slot is called after a document was loaded, when the
  document was modified (e.g. when pages were deleted), or the
  rendering mode is changed (e.g. when a different accessibility
  viewmode is selected). The following is done

  - The page cache is cleared, if clearCache == true.

  - all page widgets as well as the thumbnails are updated.
  */
  void renderModeChanged(bool clearCache = true);
};


class LigaturePartExtension : public KParts::BrowserExtension
{
  Q_OBJECT
  friend class LigaturePart;

public:

  LigaturePartExtension(LigaturePart *parent);
  virtual ~LigaturePartExtension() {}

};


#endif
