// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-

/***************************************************************************
 *   Copyright (C) 2004-2006 by Wilfried Huss                              *
 *   Wilfried.Huss@gmx.at                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.          *
 ***************************************************************************/

#ifndef MARKLIST_H
#define MARKLIST_H

#include "dataView.h"
#include "pageNumber.h"
#include "smoothScrollView.h"

#include <QList>
#include <QPixmap>
#include <QVector>
#include <QMap>
#include <QTimer>

class DocumentPageCache;
class MarkList;
class MarkListWidget;
class QAction;
class QCheckBox;
class QLabel;
class QMenu;


/****** ThumbnailWidget ******/

class ThumbnailWidget : public QWidget, public DataView
{
  Q_OBJECT

public:
  ThumbnailWidget(MarkListWidget* parent_, const PageNumber& _pageNumber, DocumentPageCache*);

  virtual void setupObservers(DataModel*);

  void setPageNumber(const PageNumber&);

  int thumbnailWidth();

private:
  virtual void paintEvent(QPaintEvent*);

private:
  PageNumber pageNumber;

  DocumentPageCache* pageCache;

  MarkListWidget* parent;
};


/****** MarkListWidget ******/


class MarkListWidget : public QWidget, public DataView
{
  Q_OBJECT

public:
  MarkListWidget(QWidget* _parent, MarkList*, const PageNumber& _pageNumber, DocumentPageCache*, bool _showThumbnail = true);

  bool isChecked() const;

  /** @returns true if the widget is visible in the thumbnail list. */
  bool isVisible();

  /** @returns true if the whole widget is visible in the thumbnail list. */
  bool isCompletelyVisible();

  PageNumber getPageNumber() const { return pageNumber; }

  virtual void setupObservers(DataModel*);

  int thumbnailWidth() { if (thumbnailWidget) return thumbnailWidget->thumbnailWidth(); else return 0; }

public slots:
  void toggle();

  void addBookmark();
  void setBookmarkLabel(const QString& label);
  void removeBookmark();

  bool isPageBookmarked() const;

  void setChecked( bool checked );

  void setSelected( bool selected );

  int setNewWidth(int width);

  void setThumbnail();

  void setPageNumber(const PageNumber&);

signals:
  /** Emitted when the Page is selected in the ThumbnailView. */
  void selected(const PageNumber&);

  /** Emitted on right click. */
  void showPopupMenu(const PageNumber& pageNumber, const QPoint& position);

  void selectionToggled(const PageNumber& page, bool on);

protected:
  virtual void mousePressEvent(QMouseEvent*);
  virtual void paintEvent(QPaintEvent*);

private slots:
  void toggleSelection(bool on);

private:

  bool showThumbnail;
  bool _selected;
  
  ThumbnailWidget* thumbnailWidget;
  QCheckBox* checkBox;
  QLabel* pageLabel;
  QLabel* bookmarkLabel;
  QColor _backgroundColor;

  PageNumber pageNumber;

  DocumentPageCache* pageCache;

  static const int margin = 5;

  MarkList* markList;
};


/****** MarkList ******/


class MarkList: public SmoothScrollView
{
    Q_OBJECT

public:
  MarkList(QWidget* parent = 0, const char* name = 0);
  virtual ~MarkList();

  void setPageCache(DocumentPageCache*);

  virtual QSize sizeHint() const { return QSize(); }

  virtual void viewportPaintEvent(QPaintEvent* e);

  virtual void setupObservers(DataModel*);

  bool isPageBookmarked(const PageNumber& pageNumber) { return dataModel->isPageBookmarked(pageNumber); }

public slots:
  void rebuildThumbnailWidgets();

  void thumbnailSelected(const PageNumber&);

  void clear();

  void slotShowThumbnails();

  /** @brief Called whenever the rendering of a thumbnail has been finished. */
  void slotSetThumbnail(PageNumber);

protected:
  virtual void viewportResizeEvent(QResizeEvent*);
  virtual void showEvent(QShowEvent*);
  virtual void mousePressEvent(QMouseEvent*);

  virtual bool isSmoothScrollDistance(double scrollDistance);

private slots:
  void setCurrentPage();

  void selectionToggled(const PageNumber& page, bool on);

  /** @brief Add a bookmark.

      @note connected to the corresponding signal from @ref DataModel */
  void bookmarkAdded(const PageNumber& page, const QString& label);

  /** @brief Rename a bookmark.

      @note connected to the corresponding signal from @ref DataModel */
  void bookmarkRenamed(const PageNumber& page, const QString& label);

  /** @brief Remove a bookmark.

      @note connected to the corresponding signal from @ref DataModel */
  void bookmarkRemoved(const PageNumber& page);

  /** @brief Remove all bookmarks.

      @note connected to the corresponding signal from @ref DataModel */
  void allBookmarksRemoved();

  /** @brief Select all pages.

      @note connected to the corresponding signal from @ref DataModel */
  void pageSelected(const PageNumber&);

  /** @brief Deselect all pages.

      @note connected to the corresponding signal from @ref DataModel */
  void pageDeselected(const PageNumber&);

  void showPopupMenu(const PageNumber& pageNumber, const QPoint& position);

  void selectAll();
  void selectEven();
  void selectOdd();
  void toggleSelection();
  void removeSelection();

  /** @brief Create MarkListWidgets.

      Calling this functions creates all MarkListWidgets that are visible
      in the current viewport. Widgets that are outside of the viewport
      are deleted, so that we only have a bounded number of widgets at all
      times, which is not depending on the number of pages in the document.
  */
  void slotCreateWidgets();
  void slotCreateWidgets(int x, int y);

  void slotStartFitTimer();

  /** @brief Resizes all MarkListWidgets to fit into the width of the viewport. */
  void fitToWidth();

signals:
  void viewSizeChanged(const QSize& size);

private:
  /** @brief creates the thumbnail widget for a page

      Calling this function creates a thumbnail widget. The widget
      is placed into the scrollview at the right place. Also the
      state of the widget (selection, bookmarks) are restored as
      according to the DataModel.

      If a widget for the needed page already exists, it is returned
      instead of a new one.
  */
  MarkListWidget* createWidget(const PageNumber& pageNumber);

  /** @brief calculate the vertical positions of all Thumbnail widgets.

      The positions are stored in the list widgetPositionList.
  */
  void recalculateWidgetPositions();

private:
  // This widget is only used to calculate the height of a MarkListWidget
  // for a given size. It is hidden. TODO: find a better way to do this.
  MarkListWidget* testWidget;

  // The widgets in this list are currently not in use.
  // Because creating and deleting MarkListWidgets is slow
  // we use this to hold widgets that are temporarely not needed.
  QVector<MarkListWidget*> widgetCache;

  QVector<unsigned int> widgetPositionList;

  QMap<PageNumber, MarkListWidget*> widgetMap;

  PageNumber currentPage;

  PageNumber clickedThumbnail;

  DocumentPageCache* pageCache;

  QMenu* contextMenu;

  QAction *Action_SelectCurrentPage;
  QAction *Action_SelectAllPages;
  QAction *Action_SelectEvenPages;
  QAction *Action_SelectOddPages;
  QAction *Action_InverSelection;
  QAction *Action_DeselectAllPages;
  QAction *Action_BookmarkPage;

  QTimer fitTimer;
};

#endif
