// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-

/***************************************************************************
 *   Copyright (C) 2006 by Wilfried Huss                                   *
 *   Wilfried.Huss@gmx.at                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.          *
 ***************************************************************************/

#include <QApplication>

#include "kvs_debug.h"
#include "renderedDocumentPagePixmap.h"
#include "renderQueue.h"

//#define DEBUG_RENDERQUEUE


RenderQueue::RenderQueue(QObject* parent)
  : QObject(parent), mutex(QMutex::Recursive)
{
}

RenderQueue::~RenderQueue()
{
  clear();
}

void RenderQueue::clear()
{
  QMutexLocker locker(&mutex);

  renderQueue.clear();
}

void RenderQueue::addRenderJob(const JobId& id)
{
  // TODO: Manage Priorities
  QMutexLocker locker(&mutex);

  if (!renderQueue.contains(id))
  {
    renderQueue.prepend(id);

    wakeRenderThread();
  }
  else
  {
    // If the page is already pending, move it to the front of the queue.
    renderQueue.removeAt(renderQueue.indexOf(id));
    renderQueue.prepend(id);
  }

  // Don't let the render queue become to large
  // TODO: Filter rendering requests that are not needed anymore, because
  // the widget of this page is now hidden.
  // FIXME: this sometimes seems to remove render requests that are still needed.
  //if (renderQueue.size() >= 60)
  //{
  //  renderQueue.pop_back();
  //}
}

void RenderQueue::addFinishedPage(RenderedDocumentPagePixmap* page)
{
  QMutexLocker locker(&mutex);

  PageNumber pageNumber = page->getPageNumber();

#ifdef DEBUG_RENDERQUEUE
  kDebug(kvs::shell) << "add finished page " << pageNumber << endl;
#endif

  renderQueue.removeAll(page->getId());

  emitRenderingFinished(page);
}

JobId RenderQueue::requestRenderJob()
{
  QMutexLocker locker(&mutex);
  if (renderQueue.isEmpty())
  {
    // Return a JobId with invalid page number, to indicate the  no rendering requests
    // are in the queue.
    return JobId();
  }

  return renderQueue.front();
}

void RenderQueue::emitRenderingFinished(RenderedDocumentPagePixmap* page)
{
  RenderingFinishedEvent* ce = new RenderingFinishedEvent(page);
  QApplication::postEvent(parent(), ce);  // Qt will delete it when done
}


void RenderQueue::waitForItems(QMutex* m)
{
  itemAdded.wait(m);
}


void RenderQueue::wakeRenderThread()
{
#ifdef DEBUG_RENDERQUEUE
  kDebug(kvs::shell) << "awake render thread" << endl;
#endif
  itemAdded.wakeAll();
}

#include "renderQueue.moc"
