// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-

/***************************************************************************
 *   Copyright (C) 2005 by Wilfried Huss                                   *
 *   Wilfried.Huss@gmx.at                                                  *
 *                                                                         *
 *   Copyright (C) 2006 by Stefan Kebekus                                  *
 *   kebekus@kde.org                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.          *
 ***************************************************************************/

#include "renderThread.h"

#include "renderQueue.h"
#include "documentRenderer.h"
#include "kvs_debug.h"
#include "renderedDocumentPage.h"

//#define RENDERTTHREAD_DEBUG

RenderThread::RenderThread(RenderQueue* queue)
  : QThread(), stopRequested(false), renderQueue(queue)
{
}

RenderThread::~RenderThread()
{
  stop();
}


void RenderThread::setRenderer(DocumentRenderer* _renderer)
{
  // Stop the renderer, if it is already running
  stop();
  
  // Set the renderer, and start the thread
  renderer = _renderer;
  if (!renderer.isNull())
    start();
}


void RenderThread::stop()
{
  stopRequested = true;

  // Awake the thread, so it can really exit
  renderQueue->wakeRenderThread();

  // Wait until the thread has stopped.
  wait();

  stopRequested = false;
}


void RenderThread::run()
{
  QMutex renderMutex;

  while (true)
  {
    renderMutex.lock();

    if (stopRequested)
    {
      renderMutex.unlock();
      return;
    }

    JobId id = renderQueue->requestRenderJob();

    if (!id.pageNumber.isValid())
    {
#ifdef RENDERTTHREAD_DEBUG
      kDebug(kvs::shell) << "render thread wait" << endl;
#endif
      renderQueue->waitForItems(&renderMutex);
    }
    else
    {
      // Safety check
      if (renderer.isNull())
      {
        kError(kvs::shell) << "RenderThread : renderer is null" << endl
                           << "rendering thread terminated abnormally." << endl;
        renderMutex.unlock();
        return;
      }
      RenderedDocumentPagePixmap* page = renderer->drawPage(id);
      if (page)
        renderQueue->addFinishedPage(page);
    }
    renderMutex.unlock();
  }
#ifdef RENDERTTHREAD_DEBUG
  kDebug(kvs::shell) << "rendering thread finished" << endl;
#endif
}
