// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-
/* This file is part of the KDE project
   Copyright (C) 2005-2006 Wilfried Huss <Wilfried.Huss@gmx.at>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include <config.h>

#include "renderedDocumentPagePixmap.h"
#include "kvs_debug.h"
#include "kvsprefs.h"

#include <kimageeffect.h>

#include <QImage>
#include <QPixmap>

//#define RenderedDocumentPagePixmap_debug

bool JobId::operator == (const JobId& id) const
{
  return (pageNumber == id.pageNumber && resolution == id.resolution && rotation == id.rotation && isThumbnail == id.isThumbnail);
}


bool JobId::operator != (const JobId& id) const
{
  return !(*this == id);
}


QString JobId::key()
{
  QString key = QString::number(pageNumber) + ":" + QString::number(resolution) + ":" + QString::number(rotation);
  if (isThumbnail)
    key += ":t";

  return key;
}


RenderedDocumentPagePixmap::RenderedDocumentPagePixmap(JobId _id)
  : currentRotation(0), currentRotationA(0), id(_id)
{
  RenderedDocumentPage::setPageNumber(id.pageNumber);
}


RenderedDocumentPagePixmap::~RenderedDocumentPagePixmap()
{
}


void RenderedDocumentPagePixmap::setupObservers(DataModel* _dataModel)
{
  DataView::setupObservers(_dataModel);
}


void RenderedDocumentPagePixmap::setPageNumber(const PageNumber& _pageNumber)
{
  RenderedDocumentPage::setPageNumber(_pageNumber);
  id.pageNumber = _pageNumber;
}

void RenderedDocumentPagePixmap::resizePixmaps()
{
  pagePixmap = QPixmap(sizeInPixels);
}

void RenderedDocumentPagePixmap::resize(int width, int height)
{
  sizeInPixels.setWidth(width);
  sizeInPixels.setHeight(height);
}

void RenderedDocumentPagePixmap::resize(const QSize& size)
{
  resize(size.width(), size.height());
}

QPixmap RenderedDocumentPagePixmap::pixmap()
{
  return pagePixmap;
}


void RenderedDocumentPagePixmap::transformImage()
{
  if (dataModel->preferences()->changeColors())
  {
    switch (dataModel->preferences()->renderMode())
    {
      case KVSPrefs::EnumRenderMode::Inverted:
        // Invert image pixels using QImage internal function
        _image.invertPixels();
        break;
      case KVSPrefs::EnumRenderMode::Recolor:
        // Recolor image using KImageEffect::flatten with dither:0
        KImageEffect::flatten(_image, dataModel->preferences()->recolorForeground(), dataModel->preferences()->recolorBackground());
        break;
      case KVSPrefs::EnumRenderMode::BlackWhite:
        // Manual Gray and Contrast
        unsigned int* data = (unsigned int*)_image.bits();
        int val;
        int pixels = _image.width() * _image.height();
        int con = dataModel->preferences()->bWContrast();
        int thr = 255 - dataModel->preferences()->bWThreshold();

        for( int i = 0; i < pixels; ++i )
        {
          val = qGray(data[i]);
          if (val > thr)
            val = 128 + (127 * (val - thr)) / (255 - thr);
          else if ( val < thr )
            val = (128 * val) / thr;
          if (con > 2)
          {
            val = con * (val - thr) / 2 + thr;
            if (val > 255)
              val = 255;
            else if (val < 0)
              val = 0;
          }
          data[i] = qRgba(val, val, val, 255);
        }
        break;
    }
  }

  if (id.rotation != 0)
  {
    QMatrix rotation;
    rotation.rotate(id.rotation * 90.0);
    _image = _image.transformed(rotation);
  }
}


void RenderedDocumentPagePixmap::setImage(QImage image)
{
  if (image.format() == QImage::Format_ARGB32 || image.format() == QImage::Format_ARGB32_Premultiplied)
  {
    // The image is of format ARGB32 or ARGB32_Premultiplied, which for some reason when converted into a
    // QPixmap can not be painted efficiently by Qt, and every paint operation that involves this pixmap,
    // slows X to a crawl (which means all open applications will have a extreme slow redrawing speed).
    // To avoid this nice Qt "feature" we manually convert the image to RGB32.
    _image = image.convertToFormat(QImage::Format_RGB32);
  }
  else
  {
    _image = image;
  }

  transformImage();
}


void RenderedDocumentPagePixmap::paintImage()
{
  if (!_image.isNull())
  {
#ifdef RenderedDocumentPagePixmap_debug
    kDebug(kvs::shell) << "RenderedDocumentPagePixmap::paintImage()" << endl;
#endif
    pagePixmap = QPixmap::fromImage(_image);
    //Free the memory used by the image.
    _image = QImage();
  }
}


unsigned int RenderedDocumentPagePixmap::memory()
{
  return size().height() * size().width() * pagePixmap.depth() / 8;
}

#include "renderedDocumentPagePixmap.moc"
