// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-
// zoom.cpp
//
// Part of LIGATURE - A framework for multipage text/gfx viewers
//
// (C) 2002 Stefan Kebekus
// Distributed under the GPL

#include <config.h>

#include "zoom.h"
#include "zoomlimits.h"

#include <klocale.h>

#include <cmath>


const float zoomVals[] = {0.20f, 0.33f, 0.50f, 0.75f, 1.00f, 1.25f, 1.50f, 2.00f, 2.50f, 0.0f};

Zoom::Zoom()
{
  valueNames << i18n("Fit to Page Width");
  valueNames << i18n("Fit to Page Height");
  valueNames << i18n("Fit to Page");

  _zoomValue = 1.0;
  valNo      = 6; // 1.0 is 6rd entry in the list
  for(int i=0; zoomVals[i] != 0; i++)
    valueNames << QString("%1%").arg(zoomVals[i]);
}


void Zoom::setZoomValue(const QString &cval)
{
  float fval;

  // Remove a trailing '%', if any
  QString val = cval.trimmed();
  if (val.right(1) == "%")
    val = val.left(val.length()-1).trimmed();

  bool ok;
  fval = val.toFloat(&ok)/100.0;

  if (ok == true)
    setZoomValue(fval);
  else {
    emit(zoomNamesChanged(valueNames));
    emit(valNoChanged(valNo));
    emit(zoomNameChanged(QString("%1%").arg((int)(_zoomValue*100.0+0.5))));
  }
}

void Zoom::setZoomFitWidth(float zoom)
{
  // Make sure _zoomValue is in the permissible range!
  const float zoom_min =  ZoomLimits::MinZoom / 1000.0f;
  const float zoom_max =  ZoomLimits::MaxZoom / 1000.0f;
  if (zoom < zoom_min)
    zoom = zoom_min;
  if (zoom > zoom_max)
    zoom = zoom_max;
  _zoomValue = zoom;

  valNo = 0;
  emit(valNoChanged(valNo));
  emit(zoomNameChanged(QString("%1%").arg((int)(_zoomValue*100.0+0.5))));
}

void Zoom::setZoomFitHeight(float zoom)
{
  // Make sure _zoomValue is in the permissible range!
  const float zoom_min =  ZoomLimits::MinZoom / 1000.0f;
  const float zoom_max =  ZoomLimits::MaxZoom / 1000.0f;
  if (zoom < zoom_min)
    zoom = zoom_min;
  if (zoom > zoom_max)
    zoom = zoom_max;
  _zoomValue = zoom;

  valNo = 1;
  emit(valNoChanged(valNo));
  emit(zoomNameChanged(QString("%1%").arg((int)(_zoomValue*100.0+0.5))));
}

void Zoom::setZoomFitPage(float zoom)
{
  // Make sure _zoomValue is in the permissible range!
  const float zoom_min =  ZoomLimits::MinZoom / 1000.0f;
  const float zoom_max =  ZoomLimits::MaxZoom / 1000.0f;
  if (zoom < zoom_min)
    zoom = zoom_min;
  if (zoom > zoom_max)
    zoom = zoom_max;
  _zoomValue = zoom;

  valNo = 2;
  emit(valNoChanged(valNo));
  emit(zoomNameChanged(QString("%1%").arg((int)(_zoomValue*100.0+0.5))));
}

void Zoom::setZoomValue(float f)
{
  // Make sure _zoomValue is in the permissible range!
  const float f_min =  ZoomLimits::MinZoom / 1000.0f;
  const float f_max =  ZoomLimits::MaxZoom / 1000.0f;
  if (f < f_min)
    f = f_min;
  if (f > f_max)
    f = f_max;
  _zoomValue = f;


  valueNames.clear();

  valueNames << i18n("Fit to Page Width");
  valueNames << i18n("Fit to Page Height");
  valueNames << i18n("Fit to Page");

  bool flag = false;
  for(int i = 0; zoomVals[i] != 0; i++) {
    if ((_zoomValue <= zoomVals[i]) && (flag == false)) {
      flag  = true;
      valNo = i + 3;
      if (fabs(_zoomValue-zoomVals[i]) > 0.01)
        valueNames << QString("%1%").arg((int)(_zoomValue*100.0+0.5));
    }
    valueNames << QString("%1%").arg((int)(zoomVals[i]*100.0+0.5));
  }

  if (flag == false) {
    valNo = int(valueNames.size());
    valueNames << QString("%1%").arg((int)(_zoomValue*100.0+0.5));
  }

  emit(zoomNamesChanged(valueNames));
  emit(valNoChanged(valNo));
  emit(zoomNameChanged(QString("%1%").arg((int)(_zoomValue*100.0+0.5))));
}

float Zoom::zoomIn()
{
  int i;
  for(i=0; zoomVals[i] != 0; i++)
    if (zoomVals[i] > _zoomValue)
      return zoomVals[i];

  return zoomVals[i-1];
}

float Zoom::zoomOut()
{
  float result = zoomVals[0];

  for(int i=0; zoomVals[i] != 0; i++)
    if (_zoomValue > zoomVals[i])
      result = zoomVals[i];

  return result;
}

#include "zoom.moc"
