// -*- c-basic-offset: 2 -*-
/*
 *  This file is part of the KDE libraries
 *  Copyright (C) 2003 Apple Computer, Inc.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "xmlhttprequest.h"
#include "xmlhttprequest.lut.h"
#include "kjs_window.h"
#include "kjs_events.h"


#include "dom/dom_doc.h"
#include "dom/dom_exception.h"
#include "dom/dom_string.h"
#include "misc/loader.h"
#include "html/html_documentimpl.h"
#include "xml/dom2_eventsimpl.h"

#include "khtml_part.h"
#include "khtmlview.h"

#include <kio/scheduler.h>
#include <kio/job.h>
#include <qobject.h>
#include <kdebug.h>

#ifdef APPLE_CHANGES
#include "KWQLoader.h"
#else
#include <kio/netaccess.h>
using KIO::NetAccess;
#endif

#define BANNED_HTTP_HEADERS "authorization,proxy-authorization,"\
                            "content-length,host,connect,copy,move,"\
                            "delete,head,trace,put,propfind,proppatch,"\
                            "mkcol,lock,unlock,options,via"

using namespace KJS;
using khtml::Decoder;

////////////////////// XMLHttpRequest Object ////////////////////////

/* Source for XMLHttpRequestProtoTable.
@begin XMLHttpRequestProtoTable 7
  abort			XMLHttpRequest::Abort			DontDelete|Function 0
  getAllResponseHeaders	XMLHttpRequest::GetAllResponseHeaders	DontDelete|Function 0
  getResponseHeader	XMLHttpRequest::GetResponseHeader	DontDelete|Function 1
  open			XMLHttpRequest::Open			DontDelete|Function 5
  send			XMLHttpRequest::Send			DontDelete|Function 1
  setRequestHeader	XMLHttpRequest::SetRequestHeader	DontDelete|Function 2
@end
*/

namespace KJS {

KJS_DEFINE_PROTOTYPE(XMLHttpRequestProto)
KJS_IMPLEMENT_PROTOFUNC(XMLHttpRequestProtoFunc)
KJS_IMPLEMENT_PROTOTYPE("XMLHttpRequest", XMLHttpRequestProto,XMLHttpRequestProtoFunc)


XMLHttpRequestQObject::XMLHttpRequestQObject(XMLHttpRequest *_jsObject)
{
  jsObject = _jsObject;
}

#ifdef APPLE_CHANGES
void XMLHttpRequestQObject::slotData( KIO::Job* job, const char *data, int size )
{
  jsObject->slotData(job, data, size);
}
#else
void XMLHttpRequestQObject::slotData( KIO::Job* job, const QByteArray &data )
{
  jsObject->slotData(job, data);
}
#endif

void XMLHttpRequestQObject::slotFinished( KJob* job )
{
  jsObject->slotFinished(job);
}

void XMLHttpRequestQObject::slotRedirection( KIO::Job* job, const KUrl& url)
{
  jsObject->slotRedirection( job, url );
}

XMLHttpRequestConstructorImp::XMLHttpRequestConstructorImp(ExecState *, DOM::DocumentImpl* d)
    : ObjectImp(), doc(d)
{
}

bool XMLHttpRequestConstructorImp::implementsConstruct() const
{
  return true;
}

ObjectImp *XMLHttpRequestConstructorImp::construct(ExecState *exec, const List &)
{
  return new XMLHttpRequest(exec, doc.get());
}

const ClassInfo XMLHttpRequest::info = { "XMLHttpRequest", 0, &XMLHttpRequestTable, 0 };


/* Source for XMLHttpRequestTable.
@begin XMLHttpRequestTable 7
  readyState		XMLHttpRequest::ReadyState		DontDelete|ReadOnly
  responseText		XMLHttpRequest::ResponseText		DontDelete|ReadOnly
  responseXML		XMLHttpRequest::ResponseXML		DontDelete|ReadOnly
  status		XMLHttpRequest::Status			DontDelete|ReadOnly
  statusText		XMLHttpRequest::StatusText		DontDelete|ReadOnly
  onreadystatechange	XMLHttpRequest::Onreadystatechange	DontDelete
  onload		XMLHttpRequest::Onload			DontDelete
@end
*/

bool XMLHttpRequest::getOwnPropertySlot(ExecState *exec, const Identifier& propertyName, PropertySlot& slot)
{
  return getStaticValueSlot<XMLHttpRequest, DOMObject>(exec, &XMLHttpRequestTable, this, propertyName, slot);
}

ValueImp *XMLHttpRequest::getValueProperty(ExecState *exec, int token) const
{
  switch (token) {
  case ReadyState:
    return Number(state);
  case ResponseText:
    return ::getStringOrNull(DOM::DOMString(response));
  case ResponseXML:
    if (state != XHRS_Loaded) {
      return Undefined();
    }
    if (!createdDocument) {
      QString mimeType = "text/xml";

      ValueImp *header = getResponseHeader("Content-Type");
      if (header->type() != UndefinedType) {
	mimeType = header->toString(exec).qstring().split(";")[0].trimmed();
      }

      if (mimeType == "text/xml" || mimeType == "application/xml" || mimeType == "application/xhtml+xml") {
	responseXML = doc->implementation()->createDocument();

	responseXML->open();
	responseXML->setURL(url.url());
	responseXML->write(response);
	responseXML->finishParsing();
	responseXML->close();

	typeIsXML = true;
      } else {
	typeIsXML = false;
      }
      createdDocument = true;
    }

    if (!typeIsXML) {
      return Null();
    }

    return getDOMNode(exec,responseXML.get());
  case Status:
    return getStatus();
  case StatusText:
    return getStatusText();
  case Onreadystatechange:
   if (onReadyStateChangeListener && onReadyStateChangeListener->listenerObj()) {
     return onReadyStateChangeListener->listenerObj();
   } else {
     return Null();
   }
  case Onload:
   if (onLoadListener && onLoadListener->listenerObj()) {
     return onLoadListener->listenerObj();
   } else {
     return Null();
   }
  default:
    kWarning() << "XMLHttpRequest::getValueProperty unhandled token " << token << endl;
    return 0;
  }
}

void XMLHttpRequest::put(ExecState *exec, const Identifier &propertyName, ValueImp *value, int attr)
{
  lookupPut<XMLHttpRequest,DOMObject>(exec, propertyName, value, attr, &XMLHttpRequestTable, this );
}

void XMLHttpRequest::putValueProperty(ExecState *exec, int token, ValueImp *value, int /*attr*/)
{
  switch(token) {
  case Onreadystatechange:
    if (onReadyStateChangeListener) onReadyStateChangeListener->deref();
    onReadyStateChangeListener = Window::retrieveActive(exec)->getJSEventListener(value, true);
    if (onReadyStateChangeListener) onReadyStateChangeListener->ref();
    break;
  case Onload:
    if (onLoadListener) onLoadListener->deref();
    onLoadListener = Window::retrieveActive(exec)->getJSEventListener(value, true);
    if (onLoadListener) onLoadListener->ref();
    break;
  default:
    kWarning() << "XMLHttpRequest::putValue unhandled token " << token << endl;
  }
}

XMLHttpRequest::XMLHttpRequest(ExecState *exec, DOM::DocumentImpl* d)
  : qObject(new XMLHttpRequestQObject(this)),
    doc(d),
    async(true),
    contentType(QString()),
    job(0),
    state(XHRS_Uninitialized),
    onReadyStateChangeListener(0),
    onLoadListener(0),
    decoder(0),
    response(QString::fromLatin1("")),
    createdDocument(false),
    aborted(false)
{
  setPrototype(XMLHttpRequestProto::self(exec));
}

XMLHttpRequest::~XMLHttpRequest()
{
  if (onLoadListener) 
      onLoadListener->deref();
  if (onReadyStateChangeListener) 
      onReadyStateChangeListener->deref();
  delete qObject;
  qObject = 0;
  delete decoder;
  decoder = 0;
}

void XMLHttpRequest::changeState(XMLHttpRequestState newState)
{
  if (state != newState) {
    state = newState;
    ProtectedPtr<ObjectImp> ref(this);

    if (onReadyStateChangeListener != 0 && doc->view() && doc->view()->part()) {
      DOM::Event ev = doc->view()->part()->document().createEvent("HTMLEvents");
      ev.initEvent("readystatechange", true, true);
      onReadyStateChangeListener->handleEvent(ev);
    }

    if (state == XHRS_Loaded && onLoadListener != 0 && doc->view() && doc->view()->part()) {
      DOM::Event ev = doc->view()->part()->document().createEvent("HTMLEvents");
      ev.initEvent("load", true, true);
      onLoadListener->handleEvent(ev);
    }
  }
}

bool XMLHttpRequest::urlMatchesDocumentDomain(const KUrl& _url) const
{
  // No need to do work if _url is not valid...
  if (!_url.isValid())
    return false;

  KUrl documentURL(doc->URL());

  // a local file can load anything
  if (documentURL.protocol().toLower() == "file") {
    return true;
  }

  // but a remote document can only load from the same port on the server
  if (documentURL.protocol().toLower() == _url.protocol().toLower() &&
      documentURL.host().toLower() == _url.host().toLower() &&
      documentURL.port() == _url.port()) {
    return true;
  }

  return false;
}

void XMLHttpRequest::open(const QString& _method, const KUrl& _url, bool _async)
{
  abort();
  aborted = false;

  // clear stuff from possible previous load
  requestHeaders.clear();
  responseHeaders.clear();
  response = QString::fromLatin1("");
  createdDocument = false;
  responseXML = 0;

  if (!urlMatchesDocumentDomain(_url)) {
    return;
  }

  method = _method.toLower();
  url = _url;
  async = _async;

  changeState(XHRS_Open);
}

void XMLHttpRequest::send(const QString& _body)
{
  aborted = false;

  if (method == "post") {
    QString protocol = url.protocol().toLower();

    // Abondon the request when the protocol is other than "http",
    // instead of blindly changing it to a "get" request.
    if (!protocol.startsWith("http") && !protocol.startsWith("webdav"))
    {
      abort();
      return;
    }

    // FIXME: determine post encoding correctly by looking in headers
    // for charset.
    QByteArray buf = _body.toUtf8();

    job = KIO::http_post( url, buf, false );
    if(contentType.isNull())
      job->addMetaData( "content-type", "Content-type: text/plain" );
    else
      job->addMetaData( "content-type", contentType );

  }
  else {
    job = KIO::get( url, false, false );
  }

  if (!requestHeaders.isEmpty()) {
    QString rh;
    QMap<QString, QString>::ConstIterator begin = requestHeaders.begin();
    QMap<QString, QString>::ConstIterator end = requestHeaders.end();
    for (QMap<QString, QString>::ConstIterator i = begin; i != end; ++i) {
      if (i != begin)
        rh += "\r\n";
      rh += i.key() + ": " + i.value();
    }

    job->addMetaData("customHTTPHeader", rh);
  }

  job->addMetaData("PropagateHttpHeader", "true");

  // Set the default referrer if one is not already supplied
  // through setRequestHeader. NOTE: the user can still disable
  // this feature at the protocol level (kio_http).
  if (requestHeaders.find("Referer") == requestHeaders.end()) {
    KUrl documentURL(doc->URL());
    documentURL.setPass(QString());
    documentURL.setUser(QString());
    job->addMetaData("referrer", documentURL.url());
    // kDebug() << "Adding referrer: " << documentURL << endl;
  }

  if (!async) {
    QByteArray data;
    KUrl finalURL;
    QString headers;

#ifdef APPLE_CHANGES
    data = KWQServeSynchronousRequest(khtml::Cache::loader(), doc->docLoader(), job, finalURL, headers);
#else
    QMap<QString, QString> metaData;
    if ( NetAccess::synchronousRun( job, 0, &data, &finalURL, &metaData ) ) {
      headers = metaData[ "HTTP-Headers" ];
    }
#endif
    job = 0;
    processSyncLoadResults(data, finalURL, headers);
    return;
  }

  qObject->connect( job, SIGNAL( result( KJob* ) ),
		    SLOT( slotFinished( KJob* ) ) );
#ifdef APPLE_CHANGES
  qObject->connect( job, SIGNAL( data( KIO::Job*, const char*, int ) ),
		    SLOT( slotData( KIO::Job*, const char*, int ) ) );
#else
  qObject->connect( job, SIGNAL( data( KIO::Job*, const QByteArray& ) ),
		    SLOT( slotData( KIO::Job*, const QByteArray& ) ) );
#endif
  qObject->connect( job, SIGNAL(redirection(KIO::Job*, const KUrl& ) ),
		    SLOT( slotRedirection(KIO::Job*, const KUrl&) ) );

#ifdef APPLE_CHANGES
  KWQServeRequest(khtml::Cache::loader(), doc->docLoader(), job);
#else
  KIO::Scheduler::scheduleJob( job );
#endif
}

void XMLHttpRequest::abort()
{
  if (job) {
    job->kill();
    job = 0;
  }
  delete decoder;
  decoder = 0;
  aborted = true;
  changeState(XHRS_Uninitialized);
}

void XMLHttpRequest::setRequestHeader(const QString& _name, const QString &value)
{
  QString name = _name.toLower().trimmed();

  // Content-type needs to be set separately from the other headers
  if(name == "content-type") {
    contentType = "Content-type: " + value;
    return;
  }

  // Sanitize the referrer header to protect against spoofing...
  if(name == "referer") {
    KUrl referrerURL(value);
    if (urlMatchesDocumentDomain(referrerURL))
      requestHeaders[name] = referrerURL.url();
    return;
  }

  // Sanitize the request headers below and handle them as if they are
  // calls to open. Otherwise, we will end up ignoring them all together!
  // TODO: Do something about "put" which kio_http sort of supports and
  // the webDAV headers such as PROPFIND etc...
  if (name == "get"  || name == "post") {
    KUrl reqURL (doc->URL(), value.trimmed());
    open(name, reqURL, async);
    return;
  }

  // Reject all banned headers. See BANNED_HTTP_HEADERS above.
  // kDebug() << "Banned HTTP Headers: " << BANNED_HTTP_HEADERS << endl;
  QStringList bannedHeaders = QString::fromLatin1(BANNED_HTTP_HEADERS).split(',');

  if (bannedHeaders.contains(name))
    return;   // Denied

  requestHeaders[name] = value.trimmed();
}

ValueImp *XMLHttpRequest::getAllResponseHeaders() const
{
  if (responseHeaders.isEmpty()) {
    return Undefined();
  }

  int endOfLine = responseHeaders.indexOf("\n");

  if (endOfLine == -1) {
    return Undefined();
  }

  return String(responseHeaders.mid(endOfLine + 1) + "\n");
}

ValueImp *XMLHttpRequest::getResponseHeader(const QString& name) const
{
  if (responseHeaders.isEmpty()) {
    return Undefined();
  }

  QRegExp headerLinePattern(name + ":", Qt::CaseInsensitive);

  int matchLength;
  int headerLinePos = headerLinePattern.indexIn(responseHeaders, 0);
  matchLength = headerLinePattern.matchedLength();
  while (headerLinePos != -1) {
    if (headerLinePos == 0 || responseHeaders[headerLinePos-1] == '\n') {
      break;
    }

    headerLinePos = headerLinePattern.indexIn(responseHeaders, headerLinePos + 1);
    matchLength = headerLinePattern.matchedLength();
  }


  if (headerLinePos == -1) {
    return Null();
  }

  int endOfLine = responseHeaders.indexOf("\n", headerLinePos + matchLength);

  return String(responseHeaders.mid(headerLinePos + matchLength, endOfLine - (headerLinePos + matchLength)).trimmed());
}

static ValueImp *httpStatus(const QString& response, bool textStatus = false)
{
  if (response.isEmpty()) {
    return Undefined();
  }

  int endOfLine = response.indexOf("\n");
  QString firstLine = (endOfLine == -1) ? response : response.left(endOfLine);
  int codeStart = firstLine.indexOf(" ");
  int codeEnd = firstLine.indexOf(" ", codeStart + 1);

  if (codeStart == -1 || codeEnd == -1) {
    return Undefined();
  }

  if (textStatus) {
    QString statusText = firstLine.mid(codeEnd + 1, endOfLine - (codeEnd + 1)).trimmed();
    return String(statusText);
  }

  QString number = firstLine.mid(codeStart + 1, codeEnd - (codeStart + 1));

  bool ok = false;
  int code = number.toInt(&ok);
  if (!ok) {
    return Undefined();
  }

  return Number(code);
}

ValueImp *XMLHttpRequest::getStatus() const
{
  return httpStatus(responseHeaders);
}

ValueImp *XMLHttpRequest::getStatusText() const
{
  return httpStatus(responseHeaders, true);
}

void XMLHttpRequest::processSyncLoadResults(const QByteArray &data, const KUrl &finalURL, const QString &headers)
{
  if (!urlMatchesDocumentDomain(finalURL)) {
    abort();
    return;
  }

  responseHeaders = headers;
  changeState(XHRS_Sent);
  if (aborted) {
    return;
  }

#ifdef APPLE_CHANGES
  const char *bytes = (const char *)data.data();
  int len = (int)data.size();

  slotData(0, bytes, len);
#else
  slotData(0, data);
#endif

  if (aborted) {
    return;
  }

  slotFinished(0);
}

void XMLHttpRequest::slotFinished(KJob *)
{
  if (decoder) {
    response += decoder->flush();
  }

  // make sure to forget about the job before emitting completed,
  // since changeState triggers JS code, which might e.g. call abort.
  job = 0;
  changeState(XHRS_Loaded);

  delete decoder;
  decoder = 0;
}

void XMLHttpRequest::slotRedirection(KIO::Job*, const KUrl& url)
{
  if (!urlMatchesDocumentDomain(url)) {
    abort();
  }
}

#ifdef APPLE_CHANGES
void XMLHttpRequest::slotData( KIO::Job*, const char *data, int len )
#else
void XMLHttpRequest::slotData(KIO::Job*, const QByteArray &_data)
#endif
{
  if (state < XHRS_Sent ) {
    responseHeaders = job->queryMetaData("HTTP-Headers");

    // NOTE: Replace a 304 response with a 200! Both IE and Mozilla do this.
    // Problem first reported through bug# 110272.
    int codeStart = responseHeaders.indexOf("304");
    if ( codeStart != -1) {
      int codeEnd = responseHeaders.indexOf("\n", codeStart+3);
      if (codeEnd != -1)
        responseHeaders.replace(codeStart, (codeEnd-codeStart), "200 OK");
    }

    changeState(XHRS_Sent);
  }

#ifndef APPLE_CHANGES
  const char *data = (const char *)_data.data();
  int len = (int)_data.size();
#endif

  if ( decoder == NULL ) {
    int pos = responseHeaders.indexOf(QLatin1String("content-type:"), 0, Qt::CaseInsensitive);

    if ( pos > -1 ) {
      pos += 13;
      int index = responseHeaders.indexOf('\n', pos);
      QString type = responseHeaders.mid(pos, (index-pos));
      index = type.indexOf(';');
      if (index > -1)
        encoding = type.mid( index+1 ).remove(QRegExp("charset[ ]*=[ ]*", Qt::CaseInsensitive)).trimmed();
    }

    decoder = new Decoder;
    if (!encoding.isNull())
      decoder->setEncoding(encoding.toLatin1().constData(), Decoder::EncodingFromHTTPHeader);
    else {
      // FIXME: Inherit the default encoding from the parent document?
    }
  }
  if (len == 0)
    return;

  if (len == -1)
    len = strlen(data);

  QString decoded = decoder->decode(data, len);

  response += decoded;

  if (!aborted) {
    changeState(XHRS_Receiving);
  }
}

ValueImp *XMLHttpRequestProtoFunc::callAsFunction(ExecState *exec, ObjectImp *thisObj, const List &args)
{
  if (!thisObj->inherits(&XMLHttpRequest::info)) {
    return throwError(exec, TypeError);
  }

  XMLHttpRequest *request = static_cast<XMLHttpRequest *>(thisObj);
  switch (id) {
  case XMLHttpRequest::Abort:
    request->abort();
    return Undefined();
  case XMLHttpRequest::GetAllResponseHeaders:
    if (args.size() != 0) {
    return Undefined();
    }

    return request->getAllResponseHeaders();
  case XMLHttpRequest::GetResponseHeader:
    if (args.size() != 1) {
    return Undefined();
    }

    return request->getResponseHeader(args[0]->toString(exec).qstring());
  case XMLHttpRequest::Open:
    {
      if (args.size() < 2 || args.size() > 5) {
        return Undefined();
      }

      QString method = args[0]->toString(exec).qstring();
      KHTMLPart *part = qobject_cast<KHTMLPart*>(Window::retrieveActive(exec)->part());
      if (!part)
        return Undefined();
      KUrl url = KUrl(part->document().completeURL(args[1]->toString(exec).qstring()).string());

      bool async = true;
      if (args.size() >= 3) {
	async = args[2]->toBoolean(exec);
      }

      if (args.size() >= 4) {
	url.setUser(args[3]->toString(exec).qstring());
      }

      if (args.size() >= 5) {
	url.setPass(args[4]->toString(exec).qstring());
      }

      request->open(method, url, async);

      return Undefined();
    }
  case XMLHttpRequest::Send:
    {
      if (args.size() > 1) {
        return Undefined();
      }

      if (request->state != XHRS_Open) {
	return Undefined();
      }

      QString body;
      if (args.size() >= 1) {
        DOM::NodeImpl* docNode = toNode(args[0]);
        if (docNode && docNode->isDocumentNode()) {
          DOM::DocumentImpl *doc = static_cast<DOM::DocumentImpl *>(docNode);
          
          try {
            body = doc->toString().string();
            // FIXME: also need to set content type, including encoding!
  
          } catch(DOM::DOMException& e) {
            return throwError(exec, GeneralError, "Exception serializing document");
          }
        } else {
          body = args[0]->toString(exec).qstring();
        }
      }

      request->send(body);

      return Undefined();
    }
  case XMLHttpRequest::SetRequestHeader:
    if (args.size() != 2) {
      return Undefined();
    }

    request->setRequestHeader(args[0]->toString(exec).qstring(), args[1]->toString(exec).qstring());

    return Undefined();
  }

  return Undefined();
}

} // end namespace

#include "xmlhttprequest.moc"
