/*
 *  This file is part of the KDE libraries
 *  Copyright (c) 2001 Michael Goffioul <kdeprint@swing.be>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License version 2 as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 **/

#include "kprinterpropertydialog.h"
#include "kprintdialogpage.h"
#include "kmfactory.h"
#include "kmuimanager.h"
#include "kmvirtualmanager.h"
#include "kmprinter.h"
#include "driver.h"

#include <kmessagebox.h>
#include <klocale.h>

#include <QtGui/QTabWidget>

KPrinterPropertyDialog::KPrinterPropertyDialog(KMPrinter *p, QWidget *parent)
: KDialog(parent, Qt::Dialog ),
  m_printer(p), m_driver(0), m_current(0)
{
	setButtons( KDialog::Ok | KDialog::Cancel | KDialog::User1 );
	setButtonGuiItem( KDialog::User1, KStandardGuiItem::save() );
	setModal(true);
	showButtonSeparator(false);
	setDefaultButton(KDialog::Ok);
	// set a margin
	m_tw = new QTabWidget(this);
	connect(m_tw,SIGNAL(currentChanged(int)),SLOT(slotCurrentChanged(int)));
	connect(this,SIGNAL(user1Clicked()),SLOT(slotUser1()));
	connect(this,SIGNAL(okClicked()),SLOT(slotOk()));
	setMainWidget(m_tw);

	if (m_printer)
		m_options = (m_printer->isEdited() ? m_printer->editedOptions() : m_printer->defaultOptions());
}

KPrinterPropertyDialog::~KPrinterPropertyDialog()
{
	qDeleteAll(m_pages);
	m_pages.clear();

	delete m_driver;
}

void KPrinterPropertyDialog::slotCurrentChanged(int index)
{
	if (m_current) m_current->getOptions(m_options,true);
	m_current = (KPrintDialogPage*)m_tw->widget(index);
	if (m_current) m_current->setOptions(m_options);
}

void KPrinterPropertyDialog::addPage(KPrintDialogPage *page)
{
	m_tw->addTab(page,page->title());
	m_pages.append(page);
	// QTabWidget does not emit currentChanged on inserting the first page
	if (m_pages.count() == 1) slotCurrentChanged(0);
}

bool KPrinterPropertyDialog::synchronize()
{
	if (m_current) m_current->getOptions(m_options,true);
	QString	msg;
	QListIterator<KPrintDialogPage*> it(m_pages);
	while (it.hasNext())
	{
		KPrintDialogPage *page = it.next();
		page->setOptions(m_options);
		if (!page->isValid(msg))
		{
			KMessageBox::error(this, msg.prepend("<qt>").append("</qt>"), i18n("Printer Configuration"));
			return false;
		}
	}
	return true;
}

void KPrinterPropertyDialog::setOptions(const QMap<QString,QString>& opts)
{
	// merge the 2 options sets
	for (QMap<QString,QString>::ConstIterator it=opts.begin(); it!=opts.end(); ++it)
		m_options[it.key()] = it.value();
	// update all existing pages
	QListIterator<KPrintDialogPage*> it2(m_pages);
	while (it2.hasNext())
		it2.next()->setOptions(m_options);
}

void KPrinterPropertyDialog::getOptions(QMap<QString,QString>& opts, bool incldef)
{
	collectOptions(opts, incldef);
}

void KPrinterPropertyDialog::collectOptions(QMap<QString,QString>& opts, bool incldef)
{
	QListIterator<KPrintDialogPage*> it(m_pages);
	while (it.hasNext())
		it.next()->getOptions(opts,incldef);
}

void KPrinterPropertyDialog::slotOk()
{
	if (!synchronize())
		return;
}

void KPrinterPropertyDialog::slotUser1()
{
	if (m_printer && synchronize())
	{
		QMap<QString,QString>	opts;
		collectOptions(opts, false);
		m_printer->setDefaultOptions(opts);
		m_printer->setEditedOptions(QMap<QString,QString>());
		m_printer->setEdited(false);
		KMFactory::self()->virtualManager()->triggerSave();
	}
}

void KPrinterPropertyDialog::enableSaveButton(bool state)
{
	showButton(KDialog::User1, state);
}

void KPrinterPropertyDialog::setupPrinter(KMPrinter *pr, QWidget *parent)
{
	KPrinterPropertyDialog	dlg(pr,parent);
	dlg.setObjectName("PropertyDialog");
	KMFactory::self()->uiManager()->setupPropertyDialog(&dlg);
	if (dlg.m_pages.count() == 0)
		KMessageBox::information(parent,i18n("No configurable options for that printer."),i18n("Printer Configuration"));
	else if (dlg.exec())
	{
		QMap<QString,QString>	opts;
		dlg.collectOptions(opts, false);
		pr->setEditedOptions(opts);
		pr->setEdited(true);
	}
}
#include "kprinterpropertydialog.moc"
