#include "tags.h"
#include "tagsgetter.h"

#include <noatun/global.h>
#include <noatun/player.h>
#include <noatun/playlist.h>

#include <qslider.h>
#include <qspinbox.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qtimer.h>
#include <qcheckbox.h>

#include <klocale.h>
#include <kconfig.h>
#include <kglobal.h>

using namespace Noatun;

TagsGetter *Tags::getter=0;

TagsGetter::TagsGetter(Global *global) : QObject(0), PlaylistNotifier(global)
{
	mTimerID = 0;
	new TagsPreferencesPage(this);
	connect(global->player(), SIGNAL(newSong()), SLOT(newSong()));
}

int TagsGetter::interval() const
{
	KGlobal::config()->setGroup("Tags");
	return KGlobal::config()->readEntry("interval", 250);
}

bool TagsGetter::loadAuto() const
{
	KGlobal::config()->setGroup("Tags");
	return KGlobal::config()->readEntry("LoadAuto", true);
}

void TagsGetter::added(PlaylistItem &i)
{
	items += i;
	if (mTimerID) killTimer(mTimerID), mTimerID=0;
	mTimerID=startTimer(interval());
}

void TagsGetter::removed(PlaylistItem &i)
{
	items.removeAll(i);
}

void TagsGetter::getSongs()
{
	items = Global::self()->playlist()->select("Tags::tagged_", "", -1, true, true);
	if (mTimerID) killTimer(mTimerID), mTimerID=0;
	mTimerID=startTimer(interval());
}

void TagsGetter::timerEvent(QTimerEvent *)
{
	if (!items.size())
	{
		if (mTimerID) killTimer(mTimerID), mTimerID=0;
		return;
	}

	PlaylistItem item=items.first();
	for (QList<Tags*>::Iterator i=tags.begin(); i != tags.end(); ++i)
	{
		if ((*i)->update(item))
			item.setProperty("Tags::tagged_", "1");
	}

	items.erase(items.begin());
}

void TagsGetter::newSong()
{
	PlaylistItem item = Global::self()->player()->current();
	if (item.isNull())
		return;

	for (QList<Tags*>::Iterator i=tags.begin(); i != tags.end(); ++i)
	{
		if ((*i)->update(item))
			item.setProperty("Tags::tagged_", "1");
	}

	items.removeAll(item);
}

void TagsGetter::setInterval(int ms)
{
	if (mTimerID) killTimer(mTimerID), mTimerID=0;
	mTimerID=startTimer(ms);

	KGlobal::config()->setGroup("Tags");
	KGlobal::config()->writeEntry("interval", ms);
	KGlobal::config()->sync();
}

void TagsGetter::setLoadAuto(bool eh)
{

	KGlobal::config()->setGroup("Tags");
	KGlobal::config()->writeEntry("LoadAuto", eh);
	KGlobal::config()->sync();

	if (mTimerID) killTimer(mTimerID), mTimerID=0;

	if (eh) mTimerID=startTimer(interval());
}

void TagsGetter::associate(Tags *t)
{
	tags.append(t);
	sortPriority();
//	getSongs();
	QTimer::singleShot(interval(), this, SLOT(getSongs()));
}

bool TagsGetter::lowerPrio(Tags* t1, Tags* t2)
{
	return t1->mPriority < t2->mPriority;
}

void TagsGetter::sortPriority()
{
	// find the lowest one, since it comes first

	qSort(tags.begin(), tags.end(), lowerPrio);
}

bool TagsGetter::unassociate(Tags *t)
{
	tags.removeAll(t);
	if (tags.count()==0)
	{
		delete this;
		return true;
	}
	return false;
}


// ----------------------------------------------------------------------------


Tags::Tags(int priority) : mPriority(priority)
{
	if (!getter)
		getter = new TagsGetter(Global::self()); // TODO: remove singleton access
	getter->associate(this);
}

Tags::~Tags()
{
	if (getter->unassociate(this))
		getter=0;
}


// ----------------------------------------------------------------------------


TagsPreferencesPage::TagsPreferencesPage(TagsGetter *parent)
	: PreferencesPage(Global::self(), parent, i18n("Tagging"),
		i18n("Settings for Tag Loaders"), "edit")
{
	tagsgetter = parent;

	QFrame *parentWidget = frame();
	QVBoxLayout *l = new QVBoxLayout(parentWidget);
	onPlay = new QCheckBox(i18n("Load tags &automatically"), parentWidget);
	l->addWidget(onPlay);
	QWidget *intervalLine = new QWidget(parentWidget);
	QHBoxLayout *llll = new QHBoxLayout(intervalLine);
	l->addWidget(intervalLine);
	l->addStretch();
	llll->addWidget(new QLabel(i18n(
//		"The time between each time noatun scans for a new file"
//		", and updates tags (e.g., ID3)",
		"Interval:"), intervalLine));
	llll->addWidget(slider = new QSlider(Qt::Horizontal, intervalLine));
	slider->setRange(0, 2000);
	llll->addWidget(spin = new QSpinBox(intervalLine));
	spin->setRange(0, 2000);
	spin->setSuffix(i18n(" ms"));
	connect(slider, SIGNAL(valueChanged(int)), spin, SLOT(setValue(int)));
	connect(spin, SIGNAL(valueChanged(int)), slider, SLOT(setValue(int)));
	connect(onPlay, SIGNAL(toggled(bool)), intervalLine, SLOT(setEnabled(bool)));
	connect(slider, SIGNAL(valueChanged(int)), this, SLOT(changed()));
	connect(spin, SIGNAL(valueChanged(int)), this, SLOT(changed()));
	connect(onPlay, SIGNAL(toggled(bool)), this, SLOT(changed()));
}

void TagsPreferencesPage::save()
{
	tagsgetter->setInterval(slider->value());
	tagsgetter->setLoadAuto(onPlay->isChecked());
	emit changed(false);
	emit saved();
}

void TagsPreferencesPage::load()
{
	onPlay->setChecked(tagsgetter->loadAuto());
	slider->setValue(tagsgetter->interval());
	emit changed(false);
}

void TagsPreferencesPage::defaults()
{
	onPlay->setChecked(true);
	slider->setValue(250);
	emit changed(true);
}


#include "tagsgetter.moc"
