
#include "metatag.h"
#include "metatag.moc"
#include "edit.h"

#include <noatun/stdaction.h>
#include <noatun/global.h>
#include <noatun/player.h>

#include <qfile.h>

#include <klocale.h>
#include <kaction.h>
#include <kconfig.h>
#include <kfilemetainfo.h>
#include <kgenericfactory.h>

#ifdef HAVE_TAGLIB
#include <fileref.h>
#include <tag.h>
#include <mpegfile.h>
#include <vorbisfile.h>
using namespace TagLib;
#endif


using namespace Noatun;

K_EXPORT_COMPONENT_FACTORY(noatun_metatag, PluginFactory<MetaTagLoader>("noatun_metatag"))

MetaTagLoader::MetaTagLoader(const KComponentData &instance, Noatun::Global *parent, const char* name)
	: Plugin(instance, parent, name)
{
	mAction = new KAction(i18n("&Tag Editor..."), "edit", Qt::CTRL+Qt::Key_T,
		this, SLOT(editTag()), actionCollection(), "edit_edittag");
	setXMLFile("noatun_metatag.rc");
	setPlugInto(Plugin::PlugInPlaylist);

	mThread = new ReaderThread(this);

	connect(global()->playlist(), SIGNAL(newCurrent()), this, SLOT(slotNewCurrent()));
	slotNewCurrent();
}


MetaTagLoader::~MetaTagLoader()
{
	kDebug(66666) << k_funcinfo << "waiting for TagReaderThread to finish..." << endl;
	mThread->wait();
	kDebug(66666) << k_funcinfo << "TagReaderThread exited." << endl;
	delete mThread;
}


Interface *MetaTagLoader::getInterface(const QString &)
{
	return 0;
}


void MetaTagLoader::slotNewCurrent()
{
	PlaylistItem item = global()->player()->current();
	mAction->setEnabled(item && !item.streamable());
}


void MetaTagLoader::editTag()
{
	PlaylistItem i = global()->player()->current();
	if(!i)
		return;

	Editor *e = new Editor();
	e->open(i);
	e->show();

	connect(e, SIGNAL(saved(PlaylistItem &)),
		SLOT(update(PlaylistItem &)));
}


bool MetaTagLoader::update(PlaylistItem & item)
{
	if (!item.url().isLocalFile()) // non-local files only have url() set, we cannot read tags from there
		return false;

	mListMutex.lock();
	mList.append(item);
	mListMutex.unlock();
	if (!mThread->running())
		mThread->start(QThread::LowestPriority);
	return false;
}


void MetaTagLoader::customEvent(QCustomEvent * e)
{
	if (e->type() == (QEvent::User + 1))
	{
		TagEvent* tagEv = static_cast<TagEvent *>(e);

		if (tagEv)
		{
			PlaylistItem item = tagEv->item();
			TagBundle t = tagEv->tag();
			propMap.clear();

			storeProperty(item, "title", t.title);
			storeProperty(item, "author", t.artist);
			storeProperty(item, "album", t.album);
			storeProperty(item, "comment", t.comment);
			storeProperty(item, "genre", t.genre);
			storeProperty(item, "track", t.trackno);
			storeProperty(item, "date", t.year);

			int playLength = t.length;
			if (playLength > 0)
				item.setLength(playLength * 1000);
			storeProperty(item, "bitrate", t.bitrate);
			storeProperty(item, "samplerate", t.samplerate);
			storeProperty(item, "channels", t.channels);

			// Hack to implement async tagging
			propMap.insert("Tags::tagged_", "1");
			item.setProperties(propMap);
		}
	}
}


bool MetaTagLoader::storeProperty(PlaylistItem &item, const QString &key, const QString &value)
{
	if (value.stripWhiteSpace().isEmpty())
		item.clearProperty(key);
	else
		propMap.insert(key, value);
	return true;
}


// ----------------------------------------------------------------------------


ReaderThread::ReaderThread(MetaTagLoader *loader)
{
	mLoader = loader;
}


void ReaderThread::run()
{
	bool processNext = true;

	while(processNext)
	{
		mLoader->mListMutex.lock();
		processNext = !mLoader->mList.empty();
		if (processNext)
		{
			PlaylistItem item = mLoader->mList.first();
			mLoader->mList.remove(item);
			mLoader->mListMutex.unlock(); // unlock asap
			readTag(item);
		}
		else
		{
			mLoader->mListMutex.unlock();
		}
	}
}


void ReaderThread::readTag(PlaylistItem &item)
{
	TagBundle tag;

#ifdef HAVE_TAGLIB
	if(item.mimetype() == "audio/x-mp3" || item.mimetype() == "audio/vorbis")
	{
		FileRef ref;
		QCString fileName = QFile::encodeName(item.url().path());

		bool stripWhitespaces = false;
		if(item.mimetype() == "audio/x-mp3")
		{
			MPEG::File *mfile = new MPEG::File(fileName.data());
			if (!mfile)
				return;
			stripWhitespaces = (mfile->ID3v1Tag(false) != 0);
 			ref = FileRef(mfile);
		}
		else if(item.mimetype() == "audio/vorbis")
		{
			ref = FileRef(new Vorbis::File(fileName.data()));
		}

		if (ref.isNull() || !ref.tag())
			return;

		tag.title = convertTString(ref.tag()->title(), stripWhitespaces);
		tag.artist = convertTString(ref.tag()->artist(), stripWhitespaces);
		tag.album = convertTString(ref.tag()->album(), stripWhitespaces);
		tag.comment = convertTString(ref.tag()->comment(), stripWhitespaces);
		tag.genre = convertTString(ref.tag()->genre(), stripWhitespaces);
		tag.trackno = convertInt(ref.tag()->track());
		tag.year = convertInt(ref.tag()->year());

		if (ref.audioProperties())
		{
			tag.length = ref.audioProperties()->length();
			tag.bitrate = convertInt(ref.audioProperties()->bitrate());
			tag.samplerate = convertInt(ref.audioProperties()->sampleRate());
			tag.channels = convertInt(ref.audioProperties()->channels());
		}
	}
	else // KFileMetaInfo as fallback
#endif
	{
		/*kDebug(66666) << k_funcinfo <<
			"fallback to KFileMetaInfo for '" << item.file() << "'" << endl;*/

		KFileMetaInfo file_info(item.url(), item.mimetype());
		if (!file_info.isValid()) // Ack, no file info :(
			return;

		// Now map the audio properties over
		if(item.length() == -1) // no value set, set almost correct id3tag time
		{
			KFileMetaInfoItem length_item = file_info.item("Length");
			if(length_item.isValid())
				tag.length = length_item.value().toInt();
		}

		tag.title      = getProperty(file_info, "Title");
		tag.artist     = getProperty(file_info, "Artist");
		tag.album      = getProperty(file_info, "Album");
		tag.comment    = getProperty(file_info, "Comment");
		tag.genre      = getProperty(file_info, "Genre");
		tag.trackno    = getProperty(file_info, "Tracknumber");
		tag.year       = getProperty(file_info, "Date");
		tag.bitrate    = getProperty(file_info, "Bitrate");
		tag.samplerate = getProperty(file_info, "Sample Rate");
		tag.channels   = getProperty(file_info, "Channels");
	}

	TagEvent *ev = new TagEvent(item, tag);
	QApplication::postEvent(mLoader, ev);
}


#ifdef HAVE_TAGLIB
QString ReaderThread::convertTString(const TagLib::String s, bool stripWhiteSpace)
{
	QString ret = TStringToQString(s);
	if (stripWhiteSpace)
		return ret.stripWhiteSpace();
	return ret;
}

QString ReaderThread::convertInt(int val)
{
	if (val != 0)
		return QString::number(val);
	return QString::null;
}
#endif


QString ReaderThread::getProperty(KFileMetaInfo &info, const QString &key)
{
	KFileMetaInfoItem info_item = info.item(key);
	if (info_item.isValid())
	{
		if (!info_item.value().toString().stripWhiteSpace().isEmpty())
		{
			// The item is valid and non-empty
			return info_item.value().toString();
		}
		else
		{
			return QString::null;
		}
	}
	return QString::null;
}
