/***************************************************************************
                     A simple database-driven playlist for Noatun
                    ----------------.-----------------------------
    begin                : 02.06.2005
    copyright            : (C) 2005 by Stefan Gehn
    email                : Stefan Gehn <mETz81@web.de>
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "itemadder.h"
#include "simpleplaylist.h"
#include "playlisthandler.h"

#include <qtimer.h>
#include <kdebug.h>
#include <kio/job.h>
#include <kfileitem.h>

#include <noatun/global.h>
#include <noatun/player.h>
#include <noatun/playlistsaver.h>


ItemAdder::ItemAdder(SimplePlaylist *pl, const KURL &url, int afterId, bool autoplay)
	: QObject(pl), playlist(pl), addAfterId(afterId), firstAddedId(0),
	recursingDirs(false), autoPlay(autoplay), listJob(0)
{
	urls.append(url);
	mediaMimeTypes = playlist->global()->player()->mimeTypes();
}

ItemAdder::ItemAdder(SimplePlaylist *pl, const KURL::List &urlList, int afterId, bool autoplay)
	: QObject(pl), playlist(pl), addAfterId(afterId), firstAddedId(0),
	recursingDirs(false), autoPlay(autoplay), listJob(0)
{
	urls = urlList;
	mediaMimeTypes = playlist->global()->player()->mimeTypes();
}

/*int ItemAdder::firstId() const // public
{
	return firstAddedId;
}*/

void ItemAdder::start() // private
{
	kDebug(66666) << k_funcinfo << "BEGIN" <<  endl;

	KURL::List::Iterator it;
	for (it = urls.begin(); it != urls.end(); ++it)
		addUnknown(*it);

	if (!recursingDirs)
		QTimer::singleShot(0, this, SLOT(finish()));

	kDebug(66666) << k_funcinfo << "END" <<  endl;
}


void ItemAdder::finish()
{
	kDebug(66666) << k_funcinfo << endl;

	if (autoPlay && firstAddedId != 0)
	{
		Noatun::PlaylistItem item(playlist->getById(firstAddedId));
		if (!item.isNull())
			playlist->global()->player()->play(item);
	}

	emit finished();
}

void ItemAdder::addUnknown(const KURL& url) // private
{
	int retId = -1;
	KFileItem fileItem(KFileItem::Unknown, KFileItem::Unknown, url);
	QString mimetype = fileItem.mimetype();
	QString extension = url.filename().lower().section('.', -1);

	/*kDebug(66666) << k_funcinfo <<
		"url = '" << url.prettyURL() <<
		"', mime = " << mimetype <<
		"', ext = " << extension << endl;*/

	if(fileItem.isLocalFile())
	{
		if (Noatun::PlaylistSaver::mimetypes().contains(mimetype)) // it's a playlist
		{
			retId = addPlaylist(url);
		}
		else // it's a local file or directory and no playlist
		{
			if(fileItem.isDir())
			{
				kDebug(66666) << k_funcinfo << "Local directory" << endl;
				addDirectory(url);
			}
			else if(mediaMimeTypes.contains(mimetype))
			{
				// Local media-file, add right away
				retId = playlist->addFile(url, -1, Noatun::PropertyMap(), addAfterId);
			}
			else
			{
				kDebug(66666) << k_funcinfo << "Unsupported LOCAL mimetype, file: " << url.prettyURL() << endl;
			}
		}
	}
	else // non-local file, we cannot check for the mimetype :(
	{
		if(Noatun::PlaylistSaver::extensions().contains(extension)) // it's probably a playlist
		{
			kDebug(66666) << k_funcinfo << "Remote playlist" << endl;
			retId = addPlaylist(url);
		}
		else // let's pray it's a stream
		{
			kDebug(66666) << k_funcinfo << "Remote file" << endl;
			retId = playlist->addFile(url, -1, Noatun::PropertyMap(), addAfterId);
		}
	}


	if (retId > 0)
	{
		// remember last item we added, the next item has to be added after it
		addAfterId = retId;

		// also remember id of the first item we added
		if (firstAddedId == 0)
			firstAddedId = retId;
	}
}

int ItemAdder::addPlaylist(const KURL &url) // private
{
	kDebug(66666) << k_funcinfo << "Adding playlist from file '" << url.fileName() << "'" << endl;

	PlaylistHandler plHandler(playlist, addAfterId);
	if (plHandler.load(url))
	{
		if (plHandler.getFirst() > 0 && firstAddedId == 0)
			firstAddedId = plHandler.getFirst();
		return plHandler.getLast();
	}
	return addAfterId; // no new "last id"
}


void ItemAdder::addDirectory(const KURL &dir)
{
	//kDebug(66666) << k_funcinfo << "dir='" << dir.prettyURL() << "'" << endl;
	recursingDirs = true;

	if (dir.upURL().equals(currentJobURL, true))
	{
		// We are a subdir of our currentJobURL and need to get listed next,
		// NOT after all the other dirs that are on the same level as
		// currentJobURL!
		kDebug(66666) << "INSERTING DIR " << dir.prettyURL() << endl;
		lastAddedSubDirectory = pendingAddDirectories.insert(lastAddedSubDirectory, dir);
		lastAddedSubDirectory++;
	}
	else
	{
		kDebug(66666) << "APPENDING DIR " << dir.prettyURL() << endl;
		pendingAddDirectories.append(dir);
	}

	addNextPendingDirectory();
}

// starts a new listJob if there is no active but work to do
void ItemAdder::addNextPendingDirectory()
{
	if (listJob)
		return;

	KURL::List::Iterator pendingIt = pendingAddDirectories.begin();
	if (pendingIt == pendingAddDirectories.end())
	{
		//nothing left, we're done
		kDebug(66666) << "DONE, no pending directories left" << endl;
		QTimer::singleShot(0, this, SLOT(finish()));
	}
	else
	{
		currentJobURL= *pendingIt;
		kDebug(66666) << "START " << currentJobURL.prettyURL() << endl;

		listJob = KIO::listDir(currentJobURL, false, false);

		connect(listJob, SIGNAL(entries(KIO::Job*, const KIO::UDSEntryList&)),
			this, SLOT(slotEntries(KIO::Job*, const KIO::UDSEntryList&)));
		connect(listJob, SIGNAL(result(KJob *)),
			this, SLOT(slotResult(KJob *)));
		connect(listJob, SIGNAL(redirection(KIO::Job *, const KURL &)),
			this, SLOT(slotRedirection(KIO::Job *, const KURL &)));

		pendingAddDirectories.remove(pendingIt);
		lastAddedSubDirectory = pendingAddDirectories.begin();
	}
}

void ItemAdder::slotResult(KJob *job)
{
	kDebug(66666) << "END   " << currentJobURL.prettyURL() << endl;
	if (job && job->error())
		job->showErrorDialog(playlist->mainWindow());
	listJob = 0;

	addNextPendingDirectory();
}

void ItemAdder::slotEntries(KIO::Job *, const KIO::UDSEntryList &entries)
{
	QMap<QString, KURL> sortedEntries; // temp list to sort entries

	KIO::UDSEntryListConstIterator it = entries.begin();
	KIO::UDSEntryListConstIterator end = entries.end();

	// create a "path -> url" map to sort files by path and not only by name
	for (; it != end; ++it)
	{
		KFileItem file(*it, currentJobURL, false, true);
		sortedEntries.insert(file.url().path(), file.url());
	}

	QMap<QString, KURL>::Iterator listIter;
	for(listIter = sortedEntries.begin(); listIter != sortedEntries.end(); ++listIter)
		addUnknown(listIter.data());
}

void ItemAdder::slotRedirection(KIO::Job *, const KURL & url)
{
	currentJobURL = url;
}


#include "itemadder.moc"
