/* This file is part of Noatun

  Copyright 2000-2006 Charles Samuels <charles@kde.org>
  Copyright 2000-2001 Neil Stevens <neil@qualityassistant.com>
  Copyright 2002 by Ryan Cumming <ryan@kde.org>
  Copyright 2003-2007 by Stefan Gehn <mETz81@web.de>

  Copyright (c) from the patches of:
                2001 Klas Kalass <klas.kalass@gmx.de>
                2001 Anno v. Heimburg <doktor.dos@gmx.de>

  Licensed under the "Artistic License"
*/

// Abandon All Hope, Ye Who Enter Here

#include <q3header.h>
#include <qlayout.h>
#include <qmap.h>
#include <qregexp.h>
#include <qtextstream.h>
#include <qpainter.h>

#include <kaction.h>
#include <kactioncollection.h>
#include <kdebug.h>
#include <kfiledialog.h>
#include <kfileitem.h>
#include <kjobuidelegate.h>
#include <kio/job.h>
#include <kio/netaccess.h>
#include <klocale.h>
#include <kpushbutton.h>
#include <ksimpleconfig.h>
#include <kstandarddirs.h>
#include <kstandardaction.h>
#include <kedittoolbar.h>
#include <k3tempfile.h>
#include <kmessagebox.h>
#include <kiconloader.h>

#include <noatun/player.h>
#include <noatun/playlistsaver.h>
#include <noatun/global.h>

#include "playlist.h"
#include "view.h"
#include "find.h"

using namespace Noatun;

#define SPL SplitPlaylist::SPL()

SafeListViewItem::SafeListViewItem(List *parent, Q3ListViewItem *after, const KUrl &url)
	: Q3CheckListItem(parent, 0, Q3CheckListItem::CheckBox) , removed(false)
{
	setProperty("url", url.url());

	static_cast<K3ListView*>(parent)->moveItem(this, 0, after);
	setOn(true);

	// is this really needed, it makes the listview too wide for me :(
//	setText(0,text.filename());

//	mProperties.setAutoDelete(true);

	PlaylistItem pli(SPL->item(new ItemData(SPL, this)));
	pli.added(SPL->Plugin::global());
}

SafeListViewItem::SafeListViewItem(List *parent, Q3ListViewItem *after, const QMap<QString,QString> &props)
	: Q3CheckListItem(parent, 0, Q3CheckListItem::CheckBox), removed(false)
{
	setOn(true);

	// A version of setProperty that assumes a key is unique,
	// and doesn't call modified for every new key.
	// Ugly, but this function is a very hot path on playlist loading
	for (QMap<QString,QString>::ConstIterator i=props.begin(); i!=props.end(); ++i )
	{
		const QString &key = i.key();
		const QString &val = i.value();

		if (key == "enabled")
		{
			setOn(val != "false" && val != "0");
		}
		else
		{
			Property p = {key, val};
			mProperties += p;
		}
	}

	static_cast<K3ListView*>(parent)->moveItem(this, 0, after);
	modified();

	PlaylistItem pli(SPL->item(new ItemData(SPL, this)));
	pli.added(SPL->Plugin::global());
}

SafeListViewItem::~SafeListViewItem()
{
	//kDebug(66666) << k_funcinfo << "For item " << text(0) << endl;
}

static void pad(QString &str)
{
	int len=str.length();
	int at = 0;
	int blocklen=0;

	static const int paddingsize=12;

	// not static for reason
	const QChar chars[paddingsize] =
	{
		QChar('0'), QChar('0'), QChar('0'), QChar('0'),
		QChar('0'), QChar('0'), QChar('0'), QChar('0'),
		QChar('0'), QChar('0'), QChar('0'), QChar('0')
	};

	for (int i=0; i < len; i++)
	{
		if (str[i].isNumber())
		{
			if (!blocklen)
				at = i;
			blocklen++;
		}
		else if (blocklen)
		{
			int pads=paddingsize;
			pads -= blocklen;
			str.insert(at, chars, pads);
			i += pads;
			blocklen = 0;
		}
	}
	if (blocklen)
	{
		int pads=paddingsize;
		pads -= blocklen;
		str.insert(at, chars, pads);
	}
}

int SafeListViewItem::compare(Q3ListViewItem * i, int col, bool) const
{
	QString text1 = text(col);
	QString text2 = i->text(col);

	pad(text1);
	pad(text2);
	return text1.compare(text2);
}

QString SafeListViewItem::property(const QString &n, const QString &def) const
{
	for (QList<Property>::ConstIterator i=mProperties.begin(); i != mProperties.end(); ++i)
	{
		if ((*i).key==n)
			return (*i).value;
	}

	if (n == "enabled")
		return (isOn() ? "true" : "false");
	return def;
}

void SafeListViewItem::setProperty(const QString &n, const QString &val)
{
	if (n == "enabled")
	{
		setOn( val != "false" && val != "0");
	}
	else
	{
		if ( property(n,"") == val )
			return;

		clearProperty(n);
		Property p={n,val};
		mProperties += p;
	}
	modified();
}

void SafeListViewItem::clearProperty(const QString &n)
{
	if (n == "enabled")
	{
		setOn(true);
		modified();
	}
	else
	{
		QList<Property>::Iterator      i(mProperties.begin());
		QList<Property>::ConstIterator end(mProperties.end());
		for ( ; i != end; ++i)
		{
			if ((*i).key == n)
			{
				mProperties.erase(i);
				modified();
				break;
			}
		}
	}
}

QStringList SafeListViewItem::properties() const
{
	QStringList list;
	for (QList<Property>::ConstIterator i=mProperties.begin(); i != mProperties.end(); ++i)
		list += (*i).key;
	list += "enabled";
	return list;
}

bool SafeListViewItem::hasProperty(const QString &n) const
{
	for (QList<Property>::ConstIterator i=mProperties.begin(); i != mProperties.end(); ++i)
	{
		if ((*i).key==n)
			return true;
	}
	return n == "enabled";
}

void SafeListViewItem::modified()
{
	bool widthChangeNeeded = false;

	PlaylistItem pli(SPL->item(new ItemData(SPL, this)));
	QString titlestring = SPL->Plugin::global()->player()->title(pli);

	if (text(0) != titlestring)
	{
		setText(0, titlestring);
		widthChangeNeeded = true;
	}

	if (pli.length()!=-1 && text(1)!=pli.lengthString())
	{
		setText(1, pli.lengthString());
		widthChangeNeeded = true;
	}

	if (widthChangeNeeded)
		widthChanged(-1);

	pli.modified(SPL->Plugin::global());
}


void SafeListViewItem::stateChange(bool s)
{
	// if you uncheck this, uncheck the others that
	// are selected too
	QList<Q3ListViewItem*> list = static_cast<K3ListView *>(listView())->selectedItems();

	// but not if I'm not selected
	if (list.contains(this))
	{
		Q3ListViewItem *i;
		foreach(i, list)
			static_cast<Q3CheckListItem*>(i)->setOn(s);
	}
	else
	{
		Q3CheckListItem::stateChange(s);
	}
}


void SafeListViewItem::paintCell(QPainter *p, const QColorGroup &cg, int column, int width, int align)
{
	Q3CheckListItem::paintCell(p, cg, column, width, align);

	ItemData *item = SPL->item(SPL->current());
	if (item && item->item() == this)
	{
		p->save();

		// p->setRasterOp(XorROP); // TODO KDE4 XXX
		p->setOpacity(0.75);
		p->fillRect(0, 0, width, height(), Qt::white);

		p->restore();
	}
}


void SafeListViewItem::remove()
{
	//kDebug(66666) << k_funcinfo << "For item " << text(0) << endl;
	removed = true;

	if (listView())
	{
		if (SPL->item(SPL->current())->item() == this)
			SPL->setCurrent(itemBelow(), false);

		listView()->takeItem(this);
	}
	else if (SPL->item(SPL->current())->item() == this)
	{
		SPL->setCurrent(0, false);
	}

#ifdef __GNUC__
#warning TODO
#endif
	//PlaylistItem pli(listView()->item(this));
	//pli.removed(Noatun::Global::self());
}

void SafeListViewItem::showPlaying(bool p)
{
	if (p)
	{
		setPixmap(0, ::SmallIcon("player_play"));
	}
	else
	{
		setPixmap(0, QPixmap());
	}

	QRect rect(listView()->itemRect(this));
	rect.setWidth(listView()->viewport()->width());
	listView()->viewport()->update(rect);
}


List::List(View *parent)
	: K3ListView(parent), recursiveAddAfter(0), listJob(0)
{
	addColumn(i18n("File"));
	addColumn(i18n("Time"));
	setAcceptDrops(true);
	setSorting(-1);
	setDropVisualizer(true);
	setDragEnabled(true);
	setItemsMovable(true);
	setSelectionMode(Q3ListView::Extended);
	connect(this, SIGNAL(dropped(QDropEvent*, Q3ListViewItem*)), SLOT(dropEvent(QDropEvent*, Q3ListViewItem*)));
	connect(this, SIGNAL(moved()), SLOT(move()));
	connect(this, SIGNAL(aboutToMove()), parent, SLOT(setNoSorting()));
	connect(this, SIGNAL(deleteCurrentItem()), parent, SLOT(deleteSelected()));
}


List::~List()
{
}


void List::move()
{
	emit modified();
}


bool List::acceptDrag(QDropEvent *event) const
{
	if (KUrl::List::canDecode(event->mimeData()))
	{
		event->setAccepted(true);
		return true;
	}

	return K3ListView::acceptDrag(event);
}


void List::dropEvent(QDropEvent *event, Q3ListViewItem *after)
{
	if (!KUrl::List::canDecode(event->mimeData()))
		return;

	KUrl::List urlList = KUrl::List::fromMimeData(event->mimeData());
	if (!urlList.isEmpty())
	{
		static_cast<View*>(parent())->setNoSorting();

		foreach(KUrl u, urlList)
		{
			after = addFile(u, false, after);
		}
		emit modified();
	}
}


void List::keyPressEvent(QKeyEvent *e)
{
	if (e->key()==Qt::Key_Enter || e->key()==Qt::Key_Return)
	{
		if (currentItem())
		{
			emit K3ListView::executed(currentItem());
		}

		return;
	}

	if (e->key()==Qt::Key_Delete)
	{
		if (currentItem())
		{
			emit deleteCurrentItem();
		}

		return;
	}

	K3ListView::keyPressEvent(e);


}

/**
 * use this only once!!!
 **/
class NoatunSaver : public PlaylistSaver
{
	List *mList;
	SafeListViewItem *after, *mFirst;
public:
	NoatunSaver(List *l, Q3ListViewItem *after=0)
		: mList(l)
	{
		this->after = static_cast<SafeListViewItem*>(after);
		mFirst = 0;
	}

	Q3ListViewItem *getAfter() { return after; }
	Q3ListViewItem *getFirst() { return mFirst; }

protected:
	virtual void readItem(const Noatun::PropertyMap &properties)
	{
		after = new SafeListViewItem(mList, after, properties);
		if (mFirst==0)
			mFirst = after;
	}

	virtual PlaylistItem writeItem()
	{
		if (!after)
		{
			after = mList->firstChild();
		}
		else
		{
			after = after->nextSibling();
		}

		if (after)
			return SPL->item(new ItemData(SPL, after));
		else
			return PlaylistItem();
	}
};


bool View::saveToURL(const KUrl &url)
{
	NoatunSaver saver(list);
	if(saver.save(url))
	{
		return true;
	}
	else
	{
		KMessageBox::error(this, i18n("Could not write to %1.", url.prettyUrl()));
		return false;
	}
}

void View::exportTo(const KUrl &url)
{
/*	KTempFile tmpFile(QString::null, ".m3u");
	QTextStream *t = tmpFile.textStream();

	// navigate the list
	for (SafeListViewItem *i = listView()->firstChild(); i != 0; i = i->itemBelow())
	{
		PlaylistItem item = SPL->item(new ItemData(SPL, i));
		KUrl u = item.url();
		if (u.isLocalFile())
			(*t) << u.path() << '\n';
		else
			(*t) << u.url() << '\n';
	}
	if (!tmpFile.close())
		return;

	KIO::NetAccess::upload(tmpFile.name(), url, this);

	tmpFile.unlink();
	*/
}

Q3ListViewItem *List::openGlobal(const KUrl &u, Q3ListViewItem *after)
{
	clear();
	NoatunSaver saver(this, after);
	saver.load(u);

	return saver.getAfter();
}

// for m3u files
Q3ListViewItem *List::importGlobal(const KUrl &u, Q3ListViewItem *after)
{
	NoatunSaver saver(this, after);
	if (!saver.load(u))
	{
		after = new SafeListViewItem(this, after, u);
		return after;
	}

	// return the first item added from this playlist
	// that way noatun can start playing the first item
	if (saver.getFirst())
		return saver.getFirst();

	// failsafe in case nothing was added, getFirst() may return 0
	return saver.getAfter();
}


Q3ListViewItem *List::addFile(const KUrl& url, bool play, Q3ListViewItem *after)
{
	// when a new item is added, we don't want to sort anymore
	//SPL->view->setNoSorting();
	static_cast<View*>(parent())->setNoSorting();


	if (
			url.path().right(4).toLower()==".m3u"
			|| url.path().right(4).toLower()==".pls"
			|| url.protocol().toLower()=="http"
		)
	{
		// a playlist is requested
		Q3ListViewItem *i = importGlobal(url, after);
		if (play)
			SPL->listItemSelected(i);
		return i;
	}
	else
	{
		if (!after)
			after = lastItem();

		KFileItem fileItem(KFileItem::Unknown,KFileItem::Unknown,url);
		if (fileItem.isDir())
		{
			addDirectoryRecursive(url, after);
			return after; // don't (and can't) know better!?
		}
		else
		{
			Q3ListViewItem *i = new SafeListViewItem(this, after, url);
			if (play)
				SPL->listItemSelected(i);
			return i;
		}
	}
}


void List::addDirectoryRecursive(const KUrl &dir, Q3ListViewItem *after)
{
	recursiveAddAfter = (after ? after : lastItem());

	if (dir.upUrl().equals(currentJobURL, KUrl::CompareWithoutTrailingSlash))
	{
		// We are a subdir of our currentJobURL and need to get listed next,
		// NOT after all the other dirs that are on the same level as
		// currentJobURL!
		lastAddedSubDirectory = pendingAddDirectories.insert(lastAddedSubDirectory, dir);
		lastAddedSubDirectory++;
	}
	else
	{
		pendingAddDirectories.append(dir);
	}
	addNextPendingDirectory();
}


// starts a new listJob if there is no active but work to do
void List::addNextPendingDirectory()
{
	KUrl::List::Iterator      pendingIt  = pendingAddDirectories.begin();
	KUrl::List::ConstIterator pendingEnd = pendingAddDirectories.end();

	if (!listJob && (pendingIt != pendingEnd))
	{
		currentJobURL = *pendingIt;
		listJob = KIO::listDir(currentJobURL, false, false);
		connect(
				listJob, SIGNAL(entries(KIO::Job*, const KIO::UDSEntryList&)),
				SLOT(slotEntries(KIO::Job*, const KIO::UDSEntryList&))
			);
		connect(
				listJob, SIGNAL(result(KJob *)),
				SLOT(slotResult(KJob *))
			);
		connect(
				listJob, SIGNAL(redirection(KIO::Job *, const KUrl &)),
				SLOT(slotRedirection(KIO::Job *, const KUrl &))
			);
		pendingAddDirectories.erase(pendingIt);
		lastAddedSubDirectory = pendingAddDirectories.begin();
	}
}

void List::slotResult(KJob *job)
{
	listJob = 0;
	if (job && job->error() && job->uiDelegate())
		job->uiDelegate()->showErrorMessage();
	addNextPendingDirectory();
}

void List::slotEntries(KIO::Job *, const KIO::UDSEntryList &entries)
{
	QMap<QString,KUrl> __list; // temp list to sort entries

	foreach (const KIO::UDSEntry entry, entries)
	{
		KFileItem file(entry, currentJobURL, false, true);
		// "prudhomm:
		// insert the path + url in the map to sort automatically by path
		// note also that you use audiocd to rip your CDs then it will be sorted the right way
		// now it is an easy fix to have a nice sort BUT it is not the best
		// we should sort based on the tracknumber"
		// - copied over from old kdirlister hack <hans_meine@gmx.de>
		__list.insert(file.url().path(), file.url());
	}

	QMap<QString,KUrl>::Iterator __it;
	for( __it = __list.begin(); __it != __list.end(); ++__it )
	{
		recursiveAddAfter= addFile(__it.value(), false, recursiveAddAfter);
	}
}

void List::slotRedirection(KIO::Job *, const KUrl & url)
{
	currentJobURL = url;
}


///////////////////////////////////////////////////////////////////////////////

View::View(Plugin *pl)
	: Noatun::NMainWindow(pl, false, false, 0, Qt::WType_TopLevel)
{
	kDebug(66666) << k_funcinfo << "BEGIN" << endl;
	setStandardToolBarMenuEnabled(true);

	mFinder = new Finder(this);

	list = new List(this);
	setCentralWidget(list);

	connect(list, SIGNAL(modified(void)), this, SLOT(setModified(void)) );
	// connect the click on the header with sorting
	connect(list->header(),SIGNAL(clicked(int)),this,SLOT(headerClicked(int)) );


	// --- "Playlist" menu ---
	mOpenNew = KStandardAction::openNew(this, SLOT(openNew()), actionCollection());
	mOpenpl  = KStandardAction::open(this, SLOT(open()), actionCollection());
	mSave    = KStandardAction::save(this, SLOT(save()), actionCollection());
	mSaveAs  = KStandardAction::saveAs(this, SLOT(saveAs()), actionCollection());

	KAction *addFiles = new KAction(actionCollection());
	addFiles->setText(i18n("Add &Files..."));
	addFiles->setIcon(KIcon("add"));
	connect(addFiles, SIGNAL(triggered()), SLOT(addFiles()));
	actionCollection()->addAction("file_add_files", addFiles);

	KAction *addFld = new KAction(actionCollection());
	addFld->setText(i18n("Add Fol&der..."));
	addFld->setIcon(KIcon("folder")); //TODO: folder_add icon needed
	connect(addFld, SIGNAL(triggered()), SLOT(addDirectory()));
	actionCollection()->addAction("file_add_dir", addFld);

	KStandardAction::close(this, SLOT(close()), actionCollection());

	// --- "Edit" menu ---

	mDelete = new KAction(actionCollection());
	mDelete->setText(i18n("Delete"));
	mDelete->setIcon(KIcon("editdelete"));
	connect(mDelete, SIGNAL(triggered()), SLOT(deleteSelected()));
	actionCollection()->addAction("edit_delete", mDelete);

	KStandardAction::find(this, SLOT(find()), actionCollection());
	KStandardAction::clear(list, SLOT(clear()), actionCollection());

	KAction *shuf = new KAction(actionCollection());
	shuf->setText(i18n("Shuffle"));
	shuf->setIcon(KIcon("misc"));
	connect(shuf, SIGNAL(triggered()), SPL, SLOT(randomize()));
	actionCollection()->addAction("shuffle", shuf);
	// ---


	//TODO: use standard name for rc-file
	setupGUI(ToolBar | Save | Create, "noatun/splui.rc");

	applyMainWindowSettings(KGlobal::config().data(), "SPL Window");
	list->setFocus();

	kDebug(66666) << k_funcinfo << "END" << endl;
}


View::~View()
{
	kDebug(66666) << k_funcinfo << "BEGIN" << endl;
	saveState();
	delete list;
	kDebug(66666) << k_funcinfo << "END" << endl;
}


void View::init()
{
	//kDebug(66666) << k_funcinfo << "BEGIN" << endl;

	KUrl internalURL;
	internalURL.setPath(KStandardDirs::locateLocal("appdata", "splitplaylist.xml"));
	list->openGlobal(internalURL);

	KConfig &config = *KGlobal::config();
	config.setGroup("splitplaylist");

	// this has to come after openGlobal, since openGlobal emits modified()
	setModified(config.readEntry<bool>("modified", false));
	QString path = config.readPathEntry("file");
	// don't call setPath with an empty path, that would make the url "valid"
	if (!path.isEmpty())
		mPlaylistFile.setPath(path);

	SPL->reset();
	int saved = config.readEntry<int>("current", 0);

	PlaylistItem item = SPL->getFirst();
	for(int i = 0 ; i < saved ; i++)
	{
		item = SPL->getAfter(item);
	}

	if (item)
		SPL->setCurrent(item);
	//kDebug(66666) << k_funcinfo << "END" << endl;
}


void View::find()
{
	mFinder->show();
	connect(mFinder, SIGNAL(search(Finder*)), SLOT(findIt(Finder*)));
}


static bool testWord(PlaylistItem &item, const QString &finder)
{
	if (item.property("author").contains(finder, Qt::CaseInsensitive))
		return true;
	if (item.property("title").contains(finder, Qt::CaseInsensitive))
		return true;
	if (item.property("album").contains(finder, Qt::CaseInsensitive))
		return true;
	if (item.property("comment").contains(finder, Qt::CaseInsensitive))
		return true;
	if (item.url().path().contains(finder, Qt::CaseInsensitive))
		return true;

	return false;
}


static bool testWord(PlaylistItem &item, const QRegExp &finder)
{
	if (item.property("author").contains(finder))
		return true;
	if (item.property("title").contains(finder))
		return true;
	if (item.property("album").contains(finder))
		return true;
	if (item.property("comment").contains(finder))
		return true;
	if (item.url().path().contains(finder))
		return true;

	return false;
}


void View::findIt(Finder *f)
{
	SafeListViewItem *search = static_cast<SafeListViewItem*>(list->currentItem());

	if (search)
	{
		if (f->isForward())
			search = search->itemBelow();
		else
			search = search->itemAbove();
	}
	else
	{
		if (f->isForward())
			search = list->firstChild();
		else
			search = static_cast<SafeListViewItem*>(list->lastChild());
	}


	while (search)
	{
		PlaylistItem item = SPL->item(new ItemData(SPL, search));
		if (f->regexp())
		{
			if (testWord(item, QRegExp(f->string(), Qt::CaseInsensitive)))
				break;
		}
		else
		{
			if (testWord(item, f->string()))
				break;
		}

		if (f->isForward())
			search = search->itemBelow();
		else
			search = search->itemAbove();

		if (!search)
		{
			if (f->isForward())
			{
				if (KMessageBox::questionYesNo(this, i18n("End of playlist reached. Continue searching from beginning?")) == KMessageBox::Yes)
					search=list->firstChild();
			}
			else
			{
				if (KMessageBox::questionYesNo(this, i18n("Beginning of playlist reached. Continue searching from end?")) == KMessageBox::Yes)
					search = static_cast<SafeListViewItem*>(list->lastChild());
			}
		}
	}

	if (search)
	{
		{ // select none
			QList<Q3ListViewItem*> sel=list->selectedItems();
			Q3ListViewItem *i;
			foreach (i, sel)
				list->setSelected(i, false);
		}
		list->setSelected(search, true);
		list->setCurrentItem(search);
		list->ensureItemVisible(search);
	}
}


void View::save()
{
	if (mPlaylistFile.isEmpty() || !mPlaylistFile.isValid())
	{
		saveAs();
		return;
	}

	if (saveToURL(mPlaylistFile))
		setModified(false);
}


void View::saveAs()
{
	KUrl u = KFileDialog::getSaveUrl(KUrl(), "*.xml splitplaylistdata *.pls *.m3u\n*", this, i18n("Save Playlist"));
	if(!u.isValid())
		return;
	mPlaylistFile = u;
	save();
}


void View::open()
{
	KUrl u=KFileDialog::getOpenUrl(KUrl(), "*.xml splitplaylistdata *.pls *.m3u\n*", this, i18n("Open Playlist"));
	if(!u.isValid())
		return;
	mPlaylistFile = u;
	list->openGlobal(u);
	setModified(false);
}


void View::openNew()
{
	mPlaylistFile = "";
	listView()->clear();
}


void List::clear()
{
	kDebug(66666) << k_funcinfo << "Clearing all items" << endl;
	SPL->setCurrent(0, false);

	// We cannot use QListView::clear() in here as this would delete all
	// PlaylistItemData and produce PlaylistItem objects with dangling mData
	// pointers around!!!
	clearSelection();

	SafeListViewItem *currentItem = 0;
	SafeListViewItem *nextItem = static_cast<SafeListViewItem*>(firstChild());
	while (nextItem)
	{
		currentItem = nextItem;
		nextItem = static_cast<SafeListViewItem*>(currentItem->itemBelow());
		currentItem->remove();
	}

	triggerUpdate();
}

void View::deleteSelected()
{
	QList<Q3ListViewItem*> items(list->selectedItems());

	bool stopped=false;
	// noatun shouldn't play files for now
	SafeListViewItem *afterLast=0;

	for (QList<Q3ListViewItem*>::Iterator it(items.begin()); it != items.end(); ++it)
	{
		SafeListViewItem *i = static_cast<SafeListViewItem*>(*it);
		if (!stopped && SPL->item(SPL->current())->item() == i)
		{
			SPL->Plugin::global()->player()->stop();
			SPL->setCurrent(0, false);
			stopped = true;
		}
		i->remove();

		afterLast = i->itemBelow();
	}

	if (stopped)
		SPL->setCurrent(afterLast, false);

	setModified(true);
}


void View::addFiles()
{
	const QStringList types = Noatun::PlaylistSaver::mimetypes() +
		SPL->Plugin::global()->player()->mimeTypes();

	kDebug(66666) << k_funcinfo << endl;

	KFileDialog dlg(KUrl("kfiledialog:///splitplaylistdir"), QString()/*types.join(" ")*/, this);
	dlg.setOperationMode(KFileDialog::Opening);
	dlg.setMimeFilter(types);
	dlg.setWindowTitle(i18n("Select File(s) to Add"));
	dlg.setMode(KFile::Files);
	dlg.okButton()->setText(i18n("&Add"));

	kDebug(66666) << k_funcinfo << "Executing KFileDialog..." << endl;

	dlg.exec();

	foreach(const KUrl &u, dlg.selectedUrls())
		addFile(u, false);

	setModified(true);
}

void View::addDirectory()
{
	QString folder = KFileDialog::getExistingDirectory(KUrl(), this,
		i18n("Select folder to add"));

	if (folder.isEmpty())
		return;

	KUrl url;
	url.setPath(folder);
	list->addDirectoryRecursive(url);

	setModified(true);
}

void View::setModified(bool b)
{
	modified = b;
	setCaption(i18n("Playlist"), modified);
}

void View::setModified(void)
{
	setModified(true);
}

void View::saveState()
{
	KConfig &config = *KGlobal::config();
	config.setGroup("splitplaylist");

	// remember if the playlist has been saved
	config.writeEntry("modified", modified);

	// real location of playlist file
	config.writePathEntry("file", mPlaylistFile.path());

	// save playlist in state-file
	saveToURL(KStandardDirs::locateLocal("appdata", "splitplaylist.xml"));

	unsigned int i;
	PlaylistItem item = SPL->getFirst();
	for(i = 0; item && item != SPL->current(); )
		item=SPL->getAfter(item), i++;

	config.writeEntry("current", i);
	saveMainWindowSettings(KGlobal::config().data(), "SPL Window");

	config.sync();
}


void View::configureToolBars()
{
	saveMainWindowSettings(KGlobal::config().data(), "SPL Window");
	KEditToolbar dlg(actionCollection(), "splui.rc");
	connect(&dlg, SIGNAL(newToolbarConfig()), SLOT(newToolBarConfig()));
	dlg.exec();
}


void View::newToolBarConfig()
{
	createGUI("splui.rc");
	applyMainWindowSettings(KGlobal::config().data(), "SPL Window");
}


// turns the sorting on or off
void View::setSorting(bool on, int column)
{
	if (on)
	{
		list->setSorting(column,true);
		list->setShowSortIndicator(true);
	}
	else
	{
		list->setShowSortIndicator(false);
		list->setSorting(-1);
	}
}


void View::headerClicked(int column)
{
	// this is to avoid that if we already have it sorted,
	// we sort it again ascendingly this way, clicking on
	// the header a second time will correctly toggle
	// ascending/descending sort
	if (list->showSortIndicator())
		return;
	setSorting(true,column);
}

#include "view.moc"

