/* NMM - Network-Integrated Multimedia Middleware
 *
 * Copyright (C) 2006
 *               Motama GmbH, Saarbruecken, Germany
 *
 * Maintainer: Bernhard Fuchshumer <fub@motama.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License only.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */


#include "audiooutput.h"
#include "backend.h"

#include <nmm/base/ProxyApplication.hpp>
#include <nmm/interfaces/device/audio/IAudioDevice.hpp>
#include <nmm/interfaces/file/ITrackImpl.hpp>
#include <nmm/base/ProxyObject.hpp>
#include <kdebug.h>

namespace Phonon
{
namespace nmm
{
AudioOutput::AudioOutput( Backend* backend, QObject* parent )
	: AbstractAudioOutput( parent )
	, m_backend( backend )
	, m_sink( 0 )
	, m_audioDevice( 0 )
	, m_endTrackListener(this, &AudioOutput::endTrackCb)
	//, m_endTrackCBEnabled(false)
{
    kDebug()<< k_funcinfo << endl;
}

AudioOutput::~AudioOutput()
{
    kDebug()<< k_funcinfo << endl;
	if (m_sink) delete m_sink;
}

float AudioOutput::volume() const
{
	if( m_audioDevice && m_device > 10000 ) // ALSAPlaybackNode
		return m_audioDevice->getVolume() * 0.01f;
	return m_volume;
}

int AudioOutput::outputDevice() const
{
	return m_device;
}

void AudioOutput::setVolume( float newVolume )
{
	if( m_audioDevice && m_device > 10000 ) // ALSAPlaybackNode
		m_audioDevice->setVolume( qRound( newVolume * 100.0f ) );
	m_volume = newVolume;
	emit volumeChanged( m_volume );
}

bool AudioOutput::setOutputDevice( int newDevice )
{
	Q_ASSERT( newDevice >= 10000 );
	Q_ASSERT( newDevice <= 10002 );
	m_device = newDevice;

    kDebug()<< k_funcinfo << endl;
	list<NMM::Response> node_response;
    NMM::NodeDescription sink_nd;
	if( newDevice > 10000 ) // ALSA
    {
        sink_nd=NMM::NodeDescription("ALSAPlaybackNode");
    }
    else //OSS
    {
        sink_nd=NMM::NodeDescription("PlaybackNode");
    }
    m_sink=m_backend->requestNode(sink_nd);
        
    if (!m_sink) {
        kError() << "unable to create audio output device node" << endl;
        return false;
    } else {
        if ( sink_nd.getSharingType() != NMM::NodeDescription::SHARED ) {
            if (sink_nd.getNodeEvent(NMM::INode::CONSTRUCTED)) {
                m_sink->reachConstructed();
                m_sink->sendEvent(*sink_nd.getNodeEvent(NMM::INode::CONSTRUCTED));
            }
        }

        m_sink->reachInitialized();
    }

    return true;
}

NMM::INode* AudioOutput::sinkNode( )
{
	if (!m_sink) setOutputDevice(m_device);
	NMM::INode* ret=m_sink;
	if (ret) 
		ret->getParentObject()->registerEventListener(NMM::ITrack::endTrack_event, &m_endTrackListener);
	m_sink=0;
	//m_endTrackCBEnabled=true;
	return ret;
}

NMM::Result AudioOutput::endTrackCb()
{
	kDebug()<<k_funcinfo<<endl;
	//if (!m_endTrackCBEnabled)
	//{
	//	return NMM::SUCCESS;
	//}
	emit endTrack();
	//m_endTrackCBEnabled=false;
	return NMM::SUCCESS;
}

}} //namespace Phonon::nmm

#include "audiooutput.moc"
// vim: sw=4 ts=4 tw=80 noet
