/* NMM - Network-Integrated Multimedia Middleware
 *
 * Copyright (C) 2006
 *               Motama GmbH, Saarbruecken, Germany
 *
 * Maintainer: Bernhard Fuchshumer <fub@motama.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License only.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */ 
#include "hostlistitem.h"

#include <Qt>
#include <kicon.h>


HostListItem::HostListItem(bool editable, QString hostname, bool audio, bool video, QString audioSink, QString videoSink, int audioId, int volume /*= -1*/, QString status /*= "UNKNOWN"*/):
	m_hostname(hostname),
	m_audio(audio),
	m_video(video),
	m_audioSink(audioSink),
	m_videoSink(videoSink),
	m_audioId(audioId),
	m_volume(volume),
	m_status(status),
	m_editableAudio(editable),
	m_editableVideo(editable),
	m_modifiedAudio(false),
	m_modifiedVideo(false)
{  
	loadPics();
}

HostListItem::HostListItem(bool editable, QString hostname, QString audioSink, int audioId, int volume/* = -1*/, QString status /*= "UNKNOWN"*/):
	m_hostname(hostname),
	m_audio(true),
	m_video(false),
	m_audioSink(audioSink),
	m_videoSink(DEFAULT_VIDEO_SINK),
	m_audioId(audioId),
	m_volume(volume),
	m_status(status),
	m_editableAudio(editable),
	m_editableVideo(true),
	m_modifiedAudio(false),
	m_modifiedVideo(false)
{
	loadPics();
}

HostListItem::HostListItem(bool editable, QString hostname, QString videoSink, QString status/* = "UNKNOWN"*/):
	m_hostname(hostname),
	m_audio(false),
	m_video(true),
	m_audioSink(DEFAULT_AUDIO_SINK),
	m_videoSink(videoSink),
	m_audioId(0),
	m_volume(-1),
	m_status(status),
	m_editableAudio(true),
	m_editableVideo(editable),
	m_modifiedAudio(false),
	m_modifiedVideo(false)
{
	loadPics();
}

HostListItem::~HostListItem()
{
}

void HostListItem::loadPics()
{
	this->yes = new KIcon( QLatin1String( "ok" ) );
	this->no = new KIcon( QLatin1String( "no" ) );
	this->info = new KIcon( QLatin1String( "info" ) );
}

QVariant HostListItem::getColumn(Column col, int role) const
{
	QVariant* retval = 0;

	if (role == Qt::DecorationRole)
	{
		switch (col)
		{
			case Hostname:
				retval = new QVariant();
				break;
			case Video:
				if (this->m_video)
					retval = new QVariant(*yes);
				else
					retval = new QVariant(*no);
				break;
			case Audio:
				if (this->m_audio)
					retval = new QVariant(*yes);
				else
					retval = new QVariant(*no);
				break;
			case AudioSink:
				retval = new QVariant();
				break;
			case VideoSink:
				retval = new QVariant();
				break;
			case AudioStream:
				retval = new QVariant();
				break;
			case Volume:
				retval = new QVariant();
				break;
			case Status:
				retval = new QVariant(*info);
				break;
			default:
				retval = new QVariant();
		}
	}
	else if (role == Qt::DisplayRole)
	{
		switch (col)
		{
			case Hostname:
				retval = new QVariant(this->m_hostname);
				break;
			case Video:
				retval = new QVariant();
				break;
			case Audio:
				retval = new QVariant();
				break;
			case AudioSink:
				retval = new QVariant(this->m_audioSink);
				break;
			case VideoSink:
				retval = new QVariant(this->m_videoSink);
				break;
			case AudioStream:
				retval = new QVariant(this->m_audioId);
				break;
			case Volume:
				retval = new QVariant(this->m_volume);
				break;
			case Status:
				retval = new QVariant(this->m_status);
				break;
			default:
				retval = new QVariant();
		}
	}
	else if (role == Qt::EditRole)
	{
		switch (col)
		{
			case Hostname:
				retval = new QVariant(this->m_hostname);
				break;
			case Video:
				retval = new QVariant(this->m_video);
				break;
			case Audio:
				retval = new QVariant(this->m_audio);
				break;
			case AudioSink:
				retval = new QVariant(this->m_audioSink);
				break;
			case VideoSink:
				retval = new QVariant(this->m_videoSink);
				break;
			case AudioStream:
				retval = new QVariant(this->m_audioId);
				break;
			case Volume:
				retval = new QVariant(this->m_volume);
				break;
			case Status:
				retval = new QVariant(this->m_status);
				break;
			default:
				retval = new QVariant();
		}
	}
	else
		return QVariant();

	

	return *retval;
}

bool HostListItem::setColumn(Column col, const QVariant& val)
{
	switch (col)
	{
		case Hostname:
			if (val.type() != QVariant::String)
				return false;
			this->m_hostname = val.toString();
			this->m_modifiedAudio = true;
			this->m_modifiedVideo = true;
			break;
		case Video:
			if (val.type() != QVariant::Bool)
				return false;
			this->m_video = val.toBool();
			this->m_modifiedVideo = true;
			break;
		case Audio:
			if (val.type() != QVariant::Bool)
				return false;
			this->m_audio = val.toBool();
			this->m_modifiedAudio = true;
			break;
		case AudioSink:
			if (val.type() != QVariant::String)
				return false;
			this->m_audioSink = val.toString();
			this->m_modifiedAudio = true;
			break;
		case VideoSink:
			if (val.type() != QVariant::String)
				return false;
			this->m_videoSink = val.toString();
			this->m_modifiedVideo = true;
			break;
		case AudioStream:
			if (val.type() != QVariant::Int)
				return false;
			this->m_audioId = val.toInt();
			this->m_modifiedAudio = true;
			break;
		case Volume:
			if (val.type() != QVariant::Int)
				return false;
			this->m_volume = val.toInt();
			//since changes are applied immediately, there is no need to set the modifed flag.
			break;
		case Status:
			if (val.type() != QVariant::String)
				return false;
			this->m_status = val.toString();
			break;
		default:
			return false;
	}
	
	return true;
}

Qt::ItemFlags HostListItem::getFlags(Column col) const
{

	switch (col)
	{
		case Hostname:
			if (m_editableVideo && m_editableAudio)
				return Qt::ItemIsSelectable | Qt::ItemIsEditable | Qt::ItemIsEnabled;
			else
				return 0;
			break;
		case Video:
			if (m_editableVideo)
				return Qt::ItemIsSelectable | Qt::ItemIsEditable | Qt::ItemIsEnabled;
			else
				return 0;
			break;
		case Audio:
			if (m_editableAudio)
				return Qt::ItemIsSelectable | Qt::ItemIsEditable | Qt::ItemIsEnabled;
			else
				return 0;
			break;
		case AudioSink:
			if (m_audio && m_editableAudio)
				return Qt::ItemIsSelectable | Qt::ItemIsEditable | Qt::ItemIsEnabled;
			else
				return 0;
			break;
		case VideoSink:
			if (m_video && m_editableVideo)
				return Qt::ItemIsSelectable | Qt::ItemIsEditable | Qt::ItemIsEnabled;
			else
				return 0;
			break;
		case AudioStream:
			if (m_audio  && m_editableAudio)
				return Qt::ItemIsSelectable | Qt::ItemIsEditable | Qt::ItemIsEnabled;
			else
				return 0;
			break;
		case Volume:
			if (m_audio)
				return Qt::ItemIsSelectable | Qt::ItemIsEditable | Qt::ItemIsEnabled;
			else
				return 0;
			break;
		case Status:
			return Qt::ItemIsSelectable | Qt::ItemIsEnabled;
			break;
		default:
			return 0;
	}
}

QString HostListItem::getColumnHeader(int i)
{
	switch (i)
	{
		case 0:
			return QString("hostname");
		case 1:
			return QString("Audio");
		case 2:
			return QString("Video");
		case 3:
			return QString("Audio Sink Node");
		case 4:
			return QString("Video Sink Node");
		case 5:
			return QString("Audio Stream");
		case 6:
			return QString("Volume");
		case 7:
			return QString("Status");
		default:
			return QString();
	}
}

QString HostListItem::getHostname()
{
	return m_hostname;
}

bool HostListItem::hasAudio()
{
	return m_audio;
}

bool HostListItem::hasVideo()
{
	return m_video;
}

QString HostListItem::getAudioSink()
{
	return m_audioSink;
}

QString HostListItem::getVideoSink()
{
	return m_videoSink;
}

int HostListItem::getAudioId()
{
	return m_audioId;
}

bool HostListItem::isAudioModified()
{
	return m_modifiedAudio;
}

bool HostListItem::isVideoModified()
{
	return m_modifiedVideo;
}

bool HostListItem::isAudioEditable()
{
	return m_editableAudio;
}

bool HostListItem::isVideoEditable()
{
	return m_editableVideo;
}


void HostListItem::setHostname(QString hostname)
{
	m_hostname = hostname;
}

void HostListItem::enableAudio(bool val)
{
	m_audio = val;
}

void HostListItem::enableVideo(bool val)
{
	m_video = val;
}

void HostListItem::setAudioSink(QString name)
{
	m_audioSink = name;
}

void HostListItem::setVideoSink(QString name)
{
	m_videoSink = name;
}

void HostListItem::setAudioId(int id)
{
	m_audioId = id;
}

void HostListItem::setAudioModified()
{
	m_modifiedAudio = true;
}

void HostListItem::setVideoModified()
{
	m_modifiedVideo = true;
}

void HostListItem::setAudioEditable(bool val)
{
	m_editableAudio = val;
}

void HostListItem::setVideoEditable(bool val)
{
	m_editableVideo = val;
}



