/* This file is part of Noatun

  Copyright 2000-2006 by Charles Samuels <charles@kde.org>
  Copyright 2000 by Nikolas Zimmermann <wildfox@kde.org>
  Copyright 2003-2007 by Stefan Gehn <mETz81@web.de>

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "noatun/frontend.h"
#include "noatun/global.h"
#include "noatun/plugin.h"
#include "noatun/player.h"
#include "noatun/playlist.h"
#include "noatun/pluginloader.h"

#include <kdebug.h>
#include <kaboutdata.h>
#include <kplugininfo.h>
#include <kxmlguiwindow.h>
#include <kxmlguifactory.h>

namespace Noatun
{


class PluginPrivate
{
public:
	Plugin::PluginPlace  plugInto;
	Global              *nInstance;
};


// -----------------------------------------------------------------------------


Plugin::Plugin(const KComponentData &instance, Global *nInstance,
	const char *name) : QObject(nInstance), KXMLGUIClient(),
	d(new PluginPrivate())
{
	Q_ASSERT(instance.isValid());
	Q_ASSERT(nInstance);

	setObjectName(name);
	setComponentData(instance);

	d->plugInto  = PlugNowhere;
	d->nInstance = nInstance;

	connect(nInstance->frontend(), SIGNAL(newFrontend()), SLOT(slotNewFrontend()));
	connect(nInstance->playlist(), SIGNAL(newPlaylist()), SLOT(slotNewPlaylist()));
}


Plugin::~Plugin()
{
	setPlugInto(PlugNowhere);
	delete d;
}


void Plugin::init()
{
}


Plugin::PluginPlace Plugin::plugInto() const
{
	return d->plugInto;
}


void Plugin::setPlugInto(PluginPlace where)
{
	if (factory())
	{
		kDebug(66666) <<
			"KXMLGUIClient of plugin '" << pluginName() <<
			"' is already plugged in into some factory, removing it from there..." << endl;
		factory()->removeClient(this);
	}

	switch(where)
	{
		case PlugInFrontend:
		{
			if (global()->frontend()->mainWindow() != 0L)
			{
				kDebug(66666) <<
					"Adding KXMLGUIClient of plugin '" << pluginName() <<
					"' to Frontend GUI" << endl;

				global()->frontend()->mainWindow()->guiFactory()->addClient(this);
			}
			d->plugInto = PlugInFrontend;
			break;
		}

		case PlugInPlaylist:
		{
			if (global()->playlist()->mainWindow() != 0L)
			{
				kDebug(66666) <<
					"Adding KXMLGUIClient of plugin '" << pluginName() <<
					"' to Playlist GUI" << endl;

				global()->playlist()->mainWindow()->guiFactory()->addClient(this);
			}
			d->plugInto = PlugInPlaylist;
			break;
		}

		default:
		{
			/*kDebug(66666) <<
				"Ensuring Plugin GUI is plugged into nothing!" << endl;*/
			d->plugInto = PlugNowhere;
		}
	}
}


void Plugin::slotNewFrontend()
{
	if (d->plugInto == PlugInFrontend)
	{
		kDebug(66666) << "Checking if we need to replug to Frontend GUI";
		KXmlGuiWindow *win = global()->frontend()->mainWindow();
		if (win != 0L)
		{
			if (factory() != win->factory())
			{
				kDebug(66666) <<
					"Adding KXMLGUIClient of plugin '" << pluginName() <<
					"' to newly loaded Frontend GUI" << endl;
				win->guiFactory()->addClient(this);
			}
		}
	}
}


void Plugin::slotNewPlaylist()
{
	if (d->plugInto == PlugInPlaylist)
	{
		kDebug(66666) << "Checking if we need to replug to Playlist GUI";
		KXmlGuiWindow *win = global()->playlist()->mainWindow();
		if (win != 0L)
		{
			if (factory() != win->factory())
			{
				kDebug(66666) <<
					"Adding KXMLGUIClient of plugin '" << pluginName() <<
					"' to newly loaded Playlist GUI" << endl;
				win->guiFactory()->addClient(this);
			}
		}
	}
}


void Plugin::requestUnload()
{
	kDebug(66666) ;
	emit readyForUnload(this);
}


QString Plugin::pluginName() const
{
	return objectName();
}


KPluginInfo Plugin::pluginInfo() const
{
	return d->nInstance->pluginHandler()->pluginInfo(this);
}


const QStringList Plugin::pluginInterfaces() const
{
	return d->nInstance->pluginHandler()->pluginInterfaces(this);
}


Global *Plugin::global()
{
	return d->nInstance;
}


const Global *Plugin::global() const
{
	return d->nInstance;
}

// the implementation of Plugin::interface is in the source
// file that implements the interface itself because
// those functions are per-interface. in other words,
// see plugininterfaces.cpp

} // namespace Noatun

#include "plugin.moc"
