/* This file is part of Noatun

  Copyright 2000-2006 by Charles Samuels <charles@kde.org>
  Copyright 2000-2002 by Neil Stevens <neil@qualityassistant.com>
  Copyright 2000-2001 by Nikolas Zimmermann <wildfox@kde.org>
  Copyright 2003-2007 by Stefan Gehn <mETz81@web.de>

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include <noatun/pluginloader.h>
#include <noatun/plugin.h>
#include <noatun/global.h>

#include <qtimer.h>
#include <qstack.h>

#include <kapplication.h>
#include <kdebug.h>
#include <kglobal.h>
#include <kmessagebox.h>
#include <kconfig.h>
#include <kservicetypetrader.h>
#include <kparts/componentfactory.h>
#include <kconfiggroup.h>

namespace Noatun
{

class PluginHandler::Private
{
public:
	Private(Global *noatunInstance) : nInstance(noatunInstance), unloadingAll(false)
	{ }

	Global *nInstance;
	// All available plugins, regardless of category, and loaded or not
	QList<KPluginInfo> availablePlugins;
	/**
	 * Dict of all currently loaded plugins, mapping the KPluginInfo to
	 * a plugin
	 */
	typedef QMap<KPluginInfo, Plugin*> LoadedPluginMap;
	LoadedPluginMap loadedPlugins;
	/**
	 * Plugins pending for loading
	 */
	QStack<QString> pluginsToLoad;
	bool unloadingAll;
};


// ---------------------------------------------------------------------------


PluginHandler::PluginHandler(Global *parent)
	: QObject(parent), d(new Private(parent))
{
	kDebug(66666) ;

	//TODO: querying ktrader only once stops us from installing new plugins
	//      at runtime
	d->availablePlugins = KPluginInfo::fromServices(
			KServiceTypeTrader::self()->query("Noatun/Plugin"));
}


PluginHandler::~PluginHandler()
{
	delete d;
	kDebug(66666) ;
}


PluginInfoList PluginHandler::availablePlugins(const QString &category) const
{
	PluginInfoList result;

	// FIXME return empty list instead, but check all uses of this method
	//       before changing this
	if (category.isEmpty())
		return d->availablePlugins;

	foreach(const KPluginInfo &pl, d->availablePlugins)
	{
		if (pl.category() == category)
			result.append(pl);
	}
	return result;
}


PluginInfoList PluginHandler::availablePluginsByInterface(const QString &interface) const
{
	PluginInfoList result;

	if (interface.isEmpty())
		return result;

	foreach(const KPluginInfo &pl, d->availablePlugins)
	{
		if (pl.property("X-Noatun-Interfaces").toStringList().contains(interface) > 0)
			result.append(pl);
	}
	return result;
}


PluginList PluginHandler::loadedPlugins(const QString &category) const
{
	Private::LoadedPluginMap::ConstIterator it;
	PluginList result;

	for (it = d->loadedPlugins.begin(); it != d->loadedPlugins.end(); ++it )
	{
		if (category.isEmpty() || it.key().category() == category)
			result.append(it.value());
	}

	return result;
}


PluginList PluginHandler::loadedPluginsByInterface(const QString &interface) const
{
	Private::LoadedPluginMap::ConstIterator it(d->loadedPlugins.begin());
	Private::LoadedPluginMap::ConstIterator end(d->loadedPlugins.end());
	PluginList result;

	for (; it != end ; ++it)
	{
		if (it.key().property("X-Noatun-Interfaces").toStringList().contains(interface))
			result.append(it.value());
	}

	return result;
}


KPluginInfo PluginHandler::pluginInfo(const Plugin *plugin) const
{
	Private::LoadedPluginMap::ConstIterator it(d->loadedPlugins.begin());
	Private::LoadedPluginMap::ConstIterator end(d->loadedPlugins.end());

	for (; it != end ; ++it)
	{
		if (it.value() == plugin)
			return it.key();
	}
	return KPluginInfo();
}


const QStringList PluginHandler::pluginInterfaces(const KPluginInfo &info) const
{
	if (info.isValid())
		return info.property("X-Noatun-Interfaces").toStringList();
	return QStringList();
}


const QStringList PluginHandler::pluginInterfaces(const Plugin *plugin) const
{
	if (plugin)
	{
		KPluginInfo info = pluginInfo(plugin);
		if (info.isValid())
			return info.property("X-Noatun-Interfaces").toStringList();
	}
	return QStringList();
}



KPluginInfo PluginHandler::pluginInfoFromName(const QString &pluginName) const
{
	PluginInfoList::ConstIterator it;
	for (it = d->availablePlugins.begin(); it != d->availablePlugins.end(); ++it )
	{
		if (it->pluginName() == pluginName)
			return *it;
	}
	return KPluginInfo();
}


bool PluginHandler::loadAll()
{
	QStringList plugs;
	KConfigGroup pluginGroup(KGlobal::config(), "Plugins");

	// load all other plugins
	PluginInfoList::Iterator it(d->availablePlugins.begin());
	PluginInfoList::Iterator end(d->availablePlugins.end());
	for (; it != end; ++it)
	{
		QStringList interfaces = it->property("X-Noatun-Interfaces").toStringList();

		if (!interfaces.contains("playlist") &&
		    !interfaces.contains("userinterface"))
		{
			it->load(pluginGroup);
			if (it->isPluginEnabled())
				plugs.append(it->pluginName());
		}
	}

	loadAutomatic(plugs, "playlist");
	loadAutomatic(plugs, "userinterface");

	return loadPlugin(plugs);
}


void PluginHandler::loadAutomatic(QStringList &thelist, const QString &type)
{
	QString pluginName;

	KConfigGroup pluginGroup(KGlobal::config(), "Plugins");

	PluginInfoList plugins = availablePluginsByInterface(type);

	// first iterate through to see if we have an enabled plugin
	for (PluginInfoList::Iterator i(plugins.begin()); i != plugins.end(); ++i)
	{
		i->load(pluginGroup);
		if (i->isPluginEnabled())
		{
			pluginName = i->pluginName();
			break;
		}
	}

	// ok, see what's enabled by default
	if (pluginName.isEmpty())
	{
		for (PluginInfoList::Iterator i(plugins.begin()); i != plugins.end(); ++i)
		{
			if (i->isPluginEnabledByDefault())
			{
				pluginName = i->pluginName();
				break;
			}
		}
	}

	// ok pick the first available
	if (pluginName.isEmpty())
	{
		if (plugins.count())
		{
			pluginName = plugins.first().pluginName();
			kDebug(66666) << "Picking the first " << type << " plugin available: " << type;
		}
		else
		{
			KMessageBox::error(
					0,
					i18n("There appears to be no '%1' plugins available for use.",type),
					i18n("Missing plugins")
				);
			return;
		}
	}

	kDebug(66666) << "Using " << pluginName << " plugin for " << type;
	thelist.append(pluginName);
}


bool PluginHandler::loadPlugin(const QStringList &modules)
{
	kDebug(66666) << "modules: " << modules;

	QStringList::ConstIterator it;
	for (it = modules.begin(); it != modules.end(); ++it )
		d->pluginsToLoad.push(*it);

	QTimer::singleShot(0, this, SLOT(slotLoadScheduledPlugin()));
	return true;
}


void PluginHandler::loadPlugin(const QString &pluginName)
{
	d->pluginsToLoad.push(pluginName);
	QTimer::singleShot(0, this, SLOT(slotLoadScheduledPlugin()));
}


void PluginHandler::slotLoadScheduledPlugin()
{
	kDebug(66666) ;

	if (d->pluginsToLoad.isEmpty())
	{
		// TODO: distinguish scheduled plugin loading: "at startup" or "while running"
		//if (d->shutdownMode == Private::StartingUp)
		{
			//d->shutdownMode = Private::Running;
			emit allPluginsLoaded();
		}
		return;
	}

	QString key = d->pluginsToLoad.pop();
	kDebug(66666) << "Going to load '" << key << "' ...";
	loadPluginInternal(key);

	QTimer::singleShot(0, this, SLOT(slotLoadScheduledPlugin()));
}


Plugin *PluginHandler::loadPluginInternal(const QString &pluginName)
{
	kDebug(66666) << pluginName;

	KPluginInfo info = pluginInfoFromName(pluginName);
	if (!info.isValid())
	{
		kWarning(66666) <<
			"Unable to find a plugin named '" << pluginName << "'!" << endl;
		return 0L;
	}

	if (d->loadedPlugins.contains(info))
	{
		kWarning(66666) <<
			"Plugin named named '" << pluginName << "' already loaded" << endl;
		return d->loadedPlugins[info];
	}

	int error = 0;
	QList<KSharedPtr<KService> > offers = KServiceTypeTrader::self()->query(
			"Noatun/Plugin",
			QString("[X-KDE-PluginInfo-Name]=='%1'").arg(pluginName)
		);

	Plugin *plugin = 0;
	QStringList slll("Noatun/Plugin");
	for (QList<KSharedPtr<KService> >::Iterator i(offers.begin()); i != offers.end(); ++i)
	{
		KService::Ptr ptr = *i;
		KLibFactory *factory = KLibLoader::self()->factory(ptr->library().toLocal8Bit());
		if (!factory)
			continue;
		PluginFactoryBase *pfb = static_cast<PluginFactoryBase*>(factory);
		if (!pfb)
			continue;
		pfb->setNoatunInstance(d->nInstance);

		plugin = static_cast<Plugin*>(pfb->create(this, ptr->name().toLatin1(), slll));
		if (plugin)
			break;
	}


	if (plugin)
	{
		d->loadedPlugins.insert(info, plugin);
		info.setPluginEnabled(true);

		connect(plugin, SIGNAL(destroyed(QObject *)),
			this, SLOT(slotPluginDestroyed(QObject *)));
		connect(plugin, SIGNAL(readyForUnload(Plugin *)),
			this, SLOT(slotReadyForUnload(Plugin *)));

		kDebug(66666) <<
			"Successfully loaded plugin '" << pluginName << "'" << endl;

		plugin->init();
		emit pluginLoaded(plugin);
	}
	else
	{
		switch(error)
		{
		case KLibLoader::ErrNoServiceFound:
			kDebug(66666) << "No service implementing the given mimetype "
					<< "and fullfilling the given constraint expression can be found." << endl;
			break;

		case KLibLoader::ErrServiceProvidesNoLibrary:
			kDebug(66666) << "the specified service provides no shared library.";
			break;

		case KLibLoader::ErrNoLibrary:
			kDebug(66666) << "the specified library could not be loaded.";
			break;

		case KLibLoader::ErrNoFactory:
			kDebug(66666) << "the library does not export a factory for creating components.";
			break;

		case KLibLoader::ErrNoComponent:
			kDebug(66666) << "the factory does not support creating components of the specified type.";
			break;
		}

		kDebug(66666) << "Loading plugin '" << pluginName <<
			"' failed, KLibLoader reported error: '" << endl <<
			KLibLoader::self()->lastErrorMessage() << "'" << endl;
	}

	return plugin;
}


void PluginHandler::savePluginStatus()
{
	//kDebug(66666) ;
	// Save list of current plugin status (i.e. loaded or not)

	KConfigGroup pluginGroup(KGlobal::config(), "Plugins");

	PluginInfoList::Iterator availIt;
	for (availIt = d->availablePlugins.begin(); availIt != d->availablePlugins.end(); ++availIt)
		availIt->save(pluginGroup);

	pluginGroup.sync();
}


void PluginHandler::unloadAll()
{
	//kDebug(66666) << "BEGIN _____________________________________";
	Private::LoadedPluginMap::ConstIterator it;

	if (d->unloadingAll)
	{
		kWarning(66666) <<
			"Already unloading all plugins, do NOT call me twice!" << endl;
		return;
	}

	savePluginStatus();

	d->unloadingAll = true;
	for (it = d->loadedPlugins.begin(); it != d->loadedPlugins.end();)
	{
		// Plugins could emit their ready for unload signal directly in response to this,
		// which would invalidate the current iterator. Therefore, we copy the iterator
		// and increment it beforehand.
		Private::LoadedPluginMap::ConstIterator current(it);
		++it;
		current.value()->requestUnload();
	}

	#if 0
	QTimer::singleShot(10000, this, SLOT(slotUnloadTimeout()));
	#endif
	//kDebug(66666) << "END ________________________________________________";
}


bool PluginHandler::unloadPlugin(const QStringList &pluginList)
{
	//kDebug(66666) << "Unloading plugins: " << pluginList;
	bool ret = true;
	foreach(const QString &plName, pluginList)
	{
		ret &= unloadPlugin(plName);
	}
	return ret;
}


bool PluginHandler::unloadPlugin(const QString &pluginName)
{
	//kDebug(66666) << "unloading plugin '" << pluginName << "'";
	if(Plugin *plug = plugin(pluginName))
	{
		plug->requestUnload();
		return true;
	}
	else
	{
		return false;
	}
}


void PluginHandler::slotReadyForUnload(Plugin *plugin)
{
	if (!plugin)
	{
		kWarning(66666) << "Calling Plugin is invalid!";
		return;
	}
	//kDebug(66666) << "plugin almost unloaded: " << plugin->pluginName();


	// pluginUnloaded is not exactly the right wording because it's still loaded
	// but if this would be emitted in slotPluginDestroyed() then we couldn't
	// access the plugin object anymore
	emit pluginUnloaded(plugin);
	plugin->deleteLater();
}


#if 0
void PluginHandler::slotUnloadTimeout()
{
	Private::LoadedPluginMap::ConstIterator it;
	QStringList remaining;

	if (!d->unloadingAll)
		return;

	for (it = d->loadedPlugins.begin(); it != d->loadedPlugins.end(); ++it )
	{
		remaining.append(it.key()->pluginName());
	}

	kWarning(66666) << "Some plugins didn't shutdown in time!" << endl
		<< "Remaining plugins: " << remaining.join(QString::fromLatin1(", ")) << endl;

	d->unloadingAll = false;
	slotUnloadAllFinished();
}
#endif

void PluginHandler::slotUnloadAllFinished()
{
	//kDebug(66666) << "Finally we got rid of all loaded plugins";
	emit allPluginsUnloaded();
}


void PluginHandler::slotPluginDestroyed(QObject *plugin)
{
	Private::LoadedPluginMap::Iterator it;
	for (it = d->loadedPlugins.begin(); it != d->loadedPlugins.end(); ++it)
	{
		if (it.value() == plugin)
		{
			/*kDebug(66666) <<
				"Successfully removed '" << it.value()->pluginName() <<
				"' from list of loaded plugins" << endl;*/

			// QMap::key() returns a const object :-( but as KPluginInfo is explicitly shared the
			// following does what we need
			KPluginInfo (it.key()).setPluginEnabled(false);
			d->loadedPlugins.erase(it);
			break;
		}
	}

	if (d->unloadingAll && d->loadedPlugins.isEmpty())
	{
		kDebug(66666) << "LAST PLUGIN UNLOADED :)";
		d->unloadingAll = false;
		QTimer::singleShot(0, this, SLOT(slotUnloadAllFinished()));
	}
}


Plugin* PluginHandler::plugin(const QString &pluginName) const
{
	/*kDebug(66666) <<
		"called for plugin: '" << pluginName << "'" << endl;*/
	KPluginInfo info = pluginInfoFromName(pluginName);
	if (!info.isValid())
		return 0;
	if (d->loadedPlugins.contains(info))
		return d->loadedPlugins[info];
	return 0;
}

} // namespace Noatun
#include "pluginloader.moc"
