/***************************************************************************
	kjfont.cpp
	--------------------------------------
	Font-Handling of KJfol
	Creates pixmaps of strings using the supplied font-pixmap
	--------------------------------------
	Maintainer: Stefan Gehn <sgehn@gmx.net>

 ***************************************************************************/

#include "kjfont.h"
#include "kjloader.h"
#include "kjwidget.h"
#include "kjconfig.h"
#include "helpers.cpp"

#include <kdebug.h>
#include <kglobalsettings.h>

#include <qimage.h>
#include <qpainter.h>

/*******************************************
 * KJFont
 *******************************************/

KJFont::KJFont(const QString &prefix, KJMainWindow *parent)
	: mTextMask(0), mTransparentRGB(0), mParent(parent), sysFontMetrics(0)
{
//	kDebug(66666) << "KJFont::KJFont(const QString &prefix, KJMainWindow *parent)" << prefix.latin1();

	if (prefix=="timefont")
	{
		mString[0]="0123456789: ";
		mString[1]=mString[2]="";
		mNullChar=' ';
	}
	else if ( (prefix=="volumefont") || (prefix=="pitchfont") )
	{
		mString[0]="0123456789% ";
		mString[1]=mString[2]="";
		mNullChar=' ';
	}
	else
	{
		mString[0]="abcdefghijklmnopqrstuvwxyz\"@";
		mString[1]="0123456789;_:()-'!_+\\/[]*&%.=$#";
		mString[2]="?*,                          ";
		mNullChar=' ';
	}

	mText = parent->pixmap(parent->item(prefix+"image")[1]);

	if ( parent->exist(prefix+"size") )
	{
		mWidth = parent->item(prefix+"size")[1].toInt();
		mHeight = parent->item(prefix+"size")[2].toInt();
	}
	else // try to load the font even we are missing important settings
	{	// this still can cause crashes!
		kDebug(66666) << "font height/width missing, this skin might crash noatun!";

		mWidth = mText.width() / strlen(mString[0]);

		if ( (prefix=="timefont") || (prefix=="volumefont") || (prefix=="pitchfont")  )
			mHeight = mText.height();
		else
			mHeight = mText.height()/3;
	}

//	kDebug(66666) << prefix << " h: " << mHeight << " w: " << mWidth;

	// fix for wrong numbers in rc (a skin named steelforged needs that)
	if ( mHeight > mText.height() )
		mHeight = mText.height();

	// Stupid Skin authors tend to forget keys :/
	if ( parent->exist(prefix+"spacing") )
		mSpacing = parent->item(prefix+"spacing")[1].toInt();
	else
		mSpacing = 0; // FIXME: What's default for this in kjfol???

	if ( parent->exist(prefix+"transparent") )
		mTransparent = (bool)parent->item(prefix+"transparent")[1].toInt();
	else
		mTransparent = true; // transparency seems to be default in kjfol

	// define color in font that will be transparent later on
	if ( mTransparent )
	{
		QImage ibackground = mText.convertToImage();
		mTransparentRGB = ibackground.pixel( ibackground.width()-1, ibackground.height()-1 );
//		kDebug(66666) << "color (" << qRed(mTransparentRGB) << "," << qGreen(mTransparentRGB) << "," << qBlue(mTransparentRGB) << ") will be transparent for " << prefix.latin1();
		mTextMask = KJgetMask(ibackground,mTransparentRGB);
	}

	mUseSysFont = parent->prefs()->useSysFont();
	if (mUseSysFont)
		recalcSysFont();
}

void KJFont::recalcSysFont(void)
{
//	kDebug(66666) << "called.";

	mUseSysFont = mParent->prefs()->useSysFont();
	if (!mUseSysFont)
		return;
	sysFont = QFont(mParent->prefs()->sysFontFamily());
	sysFont.setStyleStrategy(QFont::NoAntialias);
	delete sysFontMetrics;
	sysFontMetrics = 0;
	sysFontColor = mParent->prefs()->sysFontColor();

//	kDebug(66666) << "mHeight=" << mHeight;

	int fSize;
	for(fSize = mHeight; fSize>=4; fSize--)
	{
		sysFont.setPixelSize ( fSize );
		sysFontMetrics = new QFontMetrics(sysFont);
//		kDebug(66666) << "wanted fSize=" << fSize << ", metric h=" << sysFontMetrics->height();
		// either found a small enough font or found no proper font at all
		if ( sysFontMetrics->height() <= mHeight || fSize==4 )
		{
//			kDebug(66666) << "stopping @ fSize=" << fSize << ", metric h=" << sysFontMetrics->height();
			return;
		}
		delete sysFontMetrics;
		sysFontMetrics = 0;
	}
}

QPixmap KJFont::draw(const QCString &str, int wide, const QPoint &pos) const
{
	if ( mUseSysFont )
		return drawSysFont(str,wide,pos);
	else
		return drawPixmapFont(str,wide,pos);
}

QPixmap KJFont::drawSysFont(const QCString &s, int wide, const QPoint &pos) const
{
//	kDebug(66666) << "BEGIN, s='" << s << "'";
	QPoint to(pos);
	QString string(s);

	int stringWidth = sysFontMetrics->width( string );
//	kDebug(66666) << "final metrics; w=" << stringWidth << ", h=" << sysFontMetrics->height();

	QPixmap region(
		(stringWidth > wide ? stringWidth : wide),
		mHeight);
	QPainter rp(&region); // region painter

	QBitmap regionMask(
		(stringWidth > wide ? stringWidth : wide),
		mHeight, true); // fully transparent mask
	QPainter mp(&regionMask); // mask painter

//	kDebug(66666) << "region; w=" << region.width() << ", h=" << region.height();

	int freeSpace=0;
	// center string into pixmap if its chars won't fill the whole pixmap
	if ( stringWidth < wide )
	{
		freeSpace = wide - stringWidth;
		mp.fillRect ( to.x(), 0, (freeSpace/2), mHeight, Qt::color0 );
		to += QPoint ( (freeSpace/2), 0 );
//		kDebug(66666) << "centering text, freeSpace=" << freeSpace;
	}

	rp.setFont(sysFont);
	rp.setPen(sysFontColor);
	rp.drawText(to.x(), to.y(), region.width()-freeSpace, mHeight, Qt::AlignLeft|Qt::AlignTop, string);

	mp.setFont(sysFont);
	mp.setPen(Qt::color1);
	mp.drawText(to.x(), to.y(), region.width()-freeSpace, mHeight, Qt::AlignLeft|Qt::AlignTop, string);

	to += QPoint(region.width()-freeSpace,0);
//	kDebug(66666) << "text width=" << region.width()-freeSpace;

	if (freeSpace > 0)
	{
		mp.fillRect ( to.x(), 0, (freeSpace/2), mHeight, Qt::color0 );
		to += QPoint ( (freeSpace/2), 0 );
//		kDebug(66666) << "centering text, freeSpace=" << freeSpace;
	}

	region.setMask( regionMask );
//	kDebug(66666) << "width: " << wide << "|end after drawing: " << to.x();
//	kDebug(66666) << "END" << endl;
	return region;
}

QPixmap KJFont::drawPixmapFont(const QCString &str, int wide, const QPoint &pos) const
{
//	kDebug(66666) << "BEGIN";
	QPoint to(pos);

	QCString string = str.lower();
	QPixmap region(
		(string.length()*mWidth+string.length()*mSpacing > (unsigned int)wide
			? string.length()*mWidth+string.length()*mSpacing : wide),
		mHeight);

	QBitmap regionMask(
			 (string.length()*mWidth+string.length()*mSpacing > (unsigned int)wide
			? string.length()*mWidth+string.length()*mSpacing : wide),
			 mHeight, true); // fully transparent mask
	QPainter mask( &regionMask );

//	kDebug(66666) << "draw: {" << str << "}";

	int freeSpace=0;
	// center string into pixmap if its chars won't fill the whole pixmap
	if ( string.length()*mWidth+string.length()*mSpacing < (unsigned int)wide )
	{
		freeSpace = wide - string.length()*mWidth+string.length()*mSpacing;
		mask.fillRect ( to.x(), 0, (freeSpace/2), mHeight, Qt::color0 );
		to += QPoint ( (freeSpace/2), 0 );
	}

//	kDebug(66666) << "pixmap width=" << region.width();

	// draw every char and add spacing in between these chars if defined
	unsigned int stringLength(string.length());
	for ( unsigned int charPos=0; charPos < stringLength; charPos++ )
	{
		char c = string[charPos]; // the character to be drawn next

		drawCharacter(&region, &regionMask, to, c);
		to += QPoint(mWidth, 0);

		// draw according to "spacing"
		if ( (charPos < string.length()-1) && mSpacing > 0 )
		{	// make the spacing-area transparent
			mask.fillRect ( to.x(), 0, mSpacing, mHeight, Qt::color0 );
			to += QPoint ( mSpacing, 0 );
		}
	}

	if (freeSpace > 0)
	{
		mask.fillRect ( to.x(), 0, (freeSpace/2), mHeight, Qt::color0 );
		to += QPoint ( (freeSpace/2), 0 );
	}

	region.setMask( regionMask );
//	kDebug(66666) << "width: " << wide << "|end after drawing: " << to.x();
	return region;
}

void KJFont::drawCharacter(QPixmap *dev, QBitmap *devMask, const QPoint &to, char c) const
{
	QPoint src=charSource(c);
	int x=src.x();
	int y=src.y();
	int xs=mWidth;
	int ys=mHeight;

	bitBlt(dev, to, &mText, QRect(x,y,xs,ys), Qt::CopyROP);

	// bitBlt mask for transparency
	if ( mTransparent )
	{
		bitBlt(devMask, to, &mTextMask, QRect(x,y,xs,ys), Qt::OrROP);
	}
	else // fill mask
	{
		QPainter tempPainter (devMask);
		tempPainter.fillRect ( to.x(), 0, xs,ys, Qt::color1 );
	}
}

// needed for strchr
#include <string.h>

// searches for top/left coordinate of a given character inside the font-pixmap
QPoint KJFont::charSource(char c) const
{
	for (int i=0; i<3; i++)
	{
		const char *pos = strchr(mString[i], c);

		if (!pos) continue;
		return QPoint(mWidth*((int)(pos-mString[i])), mHeight*i);
	}

	return charSource(mNullChar);
}
