/*
    kjofol_prefs.cpp

    Copyright (c) 2004-2005 by Stefan Gehn             <metz AT gehn DOT net>

    *************************************************************************
    *                                                                       *
    * This program is free software; you can redistribute it and/or modify  *
    * it under the terms of the GNU General Public License as published by  *
    * the Free Software Foundation; either version 2 of the License, or     *
    * (at your option) any later version.                                   *
    *                                                                       *
    *************************************************************************
*/

#include "kjofol_prefs.h"
#include "kjconfig.h"
#include "parser.h"

#include "kjguisettingswidget.h"
#include "kjskinselectorwidget.h"

// system includes
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qlabel.h> // for preview
#include <qlayout.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qslider.h>
#include <qbitmap.h>
#include <qpixmap.h>
#include <qtabwidget.h>
#include <qtextbrowser.h>
#include <qfileinfo.h>
#include <qstringlist.h>
#include <qregexp.h>

#include <kgenericfactory.h>
#include <knuminput.h>
#include <kdebug.h>
#include <kglobal.h>
#include <kio/job.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kmimemagic.h>
#include <knotifyclient.h>
#include <kprocess.h>
#include <kstandarddirs.h>
#include <kglobalsettings.h>
#include <kfontcombo.h>
#include <kcolorcombo.h>
#include <kurlrequester.h>

#include "helpers.cpp"

using namespace Noatun;

KJPrefs::KJPrefs(Noatun::Plugin *plug)
	: PreferencesPage(plug, i18n("Mainwindow (K-Jöfol)"), i18n("K-Jöfol Preferences"), "gear")
{
	(new QVBoxLayout(frame(), 0, 0, "KJPrefs::vboxlayout"))->setAutoAdd(true);
	QTabWidget *tabWidget = new QTabWidget(frame(), "mTabWidget");

	mSkinselectorWidget = new KJSkinselector(tabWidget, "KJPrefs::mSkinselectorWidget");
	mGuiSettingsWidget = new KJGuiSettings(tabWidget, "KJPrefs::mGuiSettingsWidget");
	tabWidget->insertTab(mSkinselectorWidget, i18n("&Skin Selector"));
	tabWidget->insertTab(mGuiSettingsWidget, i18n("O&ther Settings"));

	connect(mSkinselectorWidget->mSkins, SIGNAL(activated(const QString&)), SLOT(slotSkinChanged(const QString&)));
	connect(mSkinselectorWidget->installButton, SIGNAL(clicked()), this, SLOT(installNewSkin()) );
	connect(mSkinselectorWidget->mRemoveButton, SIGNAL(clicked()), this, SLOT(removeSelectedSkin()) );

	connect(mGuiSettingsWidget->visScope, SIGNAL(toggled(bool)), this, SLOT(changed()));
	connect(mGuiSettingsWidget->visAnalyzer, SIGNAL(toggled(bool)), this, SLOT(changed()));
	connect(mGuiSettingsWidget->visNone, SIGNAL(toggled(bool)), this, SLOT(changed()));
	connect(mGuiSettingsWidget->visTimerValue, SIGNAL(valueChanged(int)), this, SLOT(changed()));
	connect(mGuiSettingsWidget->minPitch, SIGNAL(valueChanged(int)), this, SLOT(changed()));
	connect(mGuiSettingsWidget->maxPitch, SIGNAL(valueChanged(int)), this, SLOT(changed()));
	connect(mGuiSettingsWidget->maxPitch, SIGNAL(valueChanged(int)), this, SLOT(changed()));
	connect(mGuiSettingsWidget->displayTooltips, SIGNAL(toggled(bool)), this, SLOT(changed()));
	connect(mGuiSettingsWidget->displaySplash, SIGNAL(toggled(bool)), this, SLOT(changed()));
	connect(mGuiSettingsWidget->useSysFont, SIGNAL(toggled(bool)), this, SLOT(changed()));
	connect(mGuiSettingsWidget->titleScrollSpeed, SIGNAL(valueChanged(int)), this, SLOT(changed()));
	connect(mGuiSettingsWidget->cmbSysFont, SIGNAL(activated(int)), this, SLOT(changed()));
	connect(mGuiSettingsWidget->cmbSysFontColor, SIGNAL(activated(int)), this, SLOT(changed()));
}

void KJPrefs::defaults()
{
	KJConfig::self()->setDefaults();
	loadInternal();
	emit changed(true);
}


void KJPrefs::load()
{
	kDebug(66666) ;
	KJConfig::self()->readConfig();
	loadInternal();
	emit changed(false);
}


void KJPrefs::loadInternal()
{
	KJConfig *cfg = KJConfig::self();
	mGuiSettingsWidget->displayTooltips->setChecked	(cfg->displayTooltips());
	mGuiSettingsWidget->displaySplash->setChecked	(cfg->displaySplashScreen());
	mGuiSettingsWidget->minPitch->setValue				(cfg->minimumPitch());
	mGuiSettingsWidget->maxPitch->setValue				(cfg->maximumPitch());
	mGuiSettingsWidget->visTimerValue->setValue		(cfg->visualizationSpeed());
	mGuiSettingsWidget->useSysFont->setChecked		(cfg->useSysFont());
	mGuiSettingsWidget->cmbSysFont->setCurrentFont	(cfg->sysFontFamily());
	mGuiSettingsWidget->cmbSysFontColor->setColor	(cfg->sysFontColor());

	switch (cfg->titleScrollSpeed())
	{
		case 800:
			mGuiSettingsWidget->titleScrollSpeed->setValue(1);
			break;
		case 400:
			mGuiSettingsWidget->titleScrollSpeed->setValue(2);
			break;
		case 200:
			mGuiSettingsWidget->titleScrollSpeed->setValue(3);
			break;
	}

	switch (cfg->analyzerType())
	{
		case KJConfig::None:
			mGuiSettingsWidget->visNone->setChecked(true);
			mGuiSettingsWidget->visScope->setChecked(false);
			mGuiSettingsWidget->visAnalyzer->setChecked(false);
			break;
		case KJConfig::FFT:
		case KJConfig::StereoFFT:
			mGuiSettingsWidget->visNone->setChecked(false);
			mGuiSettingsWidget->visScope->setChecked(false);
			mGuiSettingsWidget->visAnalyzer->setChecked(true);
			break;
		case KJConfig::Mono:
			mGuiSettingsWidget->visNone->setChecked(false);
			mGuiSettingsWidget->visScope->setChecked(true);
			mGuiSettingsWidget->visAnalyzer->setChecked(false);
			break;
	}


	QStringList skins;
	QStringList skinLocations = KGlobal::dirs()->findDirs("data", "noatun/skins/kjofol");
	// iterate through all paths where Noatun is searching for kjofol-skins
	for (uint i = 0; i < skinLocations.count(); ++i )
	{
		QStringList skinDirs = QDir(skinLocations[i]).entryList();
		// iterate trough all dirs (normally, users can fsck every dir-struct *g*) containing a skin
		for (uint k = 2; k < skinDirs.count(); ++k )
		{
			QDir skinDirCnt = QDir ( skinLocations[i]+skinDirs[k], "*.rc", QDir::Name|QDir::IgnoreCase, QDir::Files );
			// make a list of all .rc-files in a skindir
			QStringList rcFiles = skinDirCnt.entryList();
			// iterate trough all those rc.-files in a skindir
			for (uint j = 0; j < rcFiles.count(); j++ )
			{
				//kDebug(66666) << "found: " << rcFiles[j];
				skins += rcFiles[j];
			}
		}
	}
	skins.sort();

	QString loaded = cfg->skinResource();
	loaded = loaded.mid(loaded.findRev("/")+1);  // remove path
	loaded = loaded.left(loaded.length() - 3); // remove ".rc"

	mSkinselectorWidget->mSkins->clear();

	int index = 0;
	for (QStringList::Iterator i=skins.begin(); i!=skins.end(); ++i)
	{
		*i = (*i).left( (*i).length() - 3 );
		mSkinselectorWidget->mSkins->insertItem(*i);
		if ( (*i) == loaded )
			index = mSkinselectorWidget->mSkins->count()-1; // save index no. to set active item later on
	}

	mSkinselectorWidget->mSkins->setCurrentItem(index);
	showPreview(mSkinselectorWidget->mSkins->currentText());
}


void KJPrefs::save()
{
	kDebug(66666) << "called.";

	KJConfig *cfg = KJConfig::self();

	cfg->setSkinResource(expand(mSkinselectorWidget->mSkins->currentText()));

	cfg->setDisplayTooltips(mGuiSettingsWidget->displayTooltips->isChecked());
	cfg->setDisplaySplashScreen(mGuiSettingsWidget->displaySplash->isChecked());
	cfg->setMinimumPitch(mGuiSettingsWidget->minPitch->value());
	cfg->setMaximumPitch(mGuiSettingsWidget->maxPitch->value());
	cfg->setVisualizationSpeed(mGuiSettingsWidget->visTimerValue->value());
	cfg->setUseSysFont(mGuiSettingsWidget->useSysFont->isChecked());
	cfg->setSysFontFamily(mGuiSettingsWidget->cmbSysFont->currentFont());
	cfg->setSysFontColor(mGuiSettingsWidget->cmbSysFontColor->color());

	switch (mGuiSettingsWidget->titleScrollSpeed->value())
	{
		case 1:
			cfg->setTitleScrollSpeed(800);
			break;
		case 2:
			cfg->setTitleScrollSpeed(400);
			break;
		case 3:
			cfg->setTitleScrollSpeed(200);
			break;
	}

	if (mGuiSettingsWidget->visAnalyzer->isChecked())
		cfg->setAnalyzerType(KJConfig::FFT);
	else if (mGuiSettingsWidget->visScope->isChecked())
		cfg->setAnalyzerType(KJConfig::Mono);
	else
		cfg->setAnalyzerType(KJConfig::None);

	cfg->writeConfig();
	emit changed(false);
	emit saved();
}


void KJPrefs::slotSkinChanged(const QString &skin)
{
	showPreview(skin);
	emit changed(true);
}


void KJPrefs::showPreview(const QString &_skin)
{
	Parser p;
	p.open( expand(_skin) );

	QImage image = p.image(p["BackgroundImage"][1]);
	if (!image.isNull())
	{
		mPixmap.convertFromImage(image);
		mPixmap.setMask(KJgetMask(image));
	}
	else
		mPixmap.resize(0,0);

	mSkinselectorWidget->mPreview->setPixmap(mPixmap);
	mSkinselectorWidget->mAboutText->setText(p.about());
	mSkinselectorWidget->updateGeometry();
}


/**
 * @todo Use kio_zip if possible instead of relying on unzip command
 **/
void KJPrefs::installNewSkin()
{
	bool skinInstalled = false; // flag showing wether a skindir got installed
	KUrl src, dst; // sourcedir and destinationdir for skin-installation

	KUrl srcFile ( mSkinselectorWidget->mSkinRequester->url() );

	//kDebug(66666) << "file to work on: " << srcFile.path().latin1();

	if ( !srcFile.isValid() || srcFile.isEmpty() ) // stop working on broken URLs
	{
		kDebug(66666) << "srcFile is malformed or empty !!!";
		return;
	}

	if ( !srcFile.isLocalFile() )  // TODO: Download file into tmp dir + unpack afterwards
	{
		KMessageBox::sorry(frame(), i18n("Non-Local files are not supported yet") );
		return;
	}

	// Determine file-format trough mimetype (no stupid .ext test)
	KMimeMagicResult * result = KMimeMagic::self()->findFileType( srcFile.path() );

	if ( !result->isValid() )
	{
		kDebug(66666) << "Could not determine filetype of srcFile !!!";
		return;
	}

	if ( result->mimeType() != "application/zip" )
	{
		KMessageBox::error(frame(), i18n("The selected file does not appear to be a valid zip-archive") );
		return;
	}

	// create a dir with name of the skinarchive
	// path to unpack to: pathToTmp/filename.ext/
	QString tmpUnpackPath = locateLocal("tmp", srcFile.fileName()+'/');
	kDebug(66666) << "tmpUnpackPath: " << tmpUnpackPath.latin1();

	// Our extract-process, TODO: wanna have kio_(un)zip instead :)
	KProcess proc;

	// "unzip -d whereToUnpack whatToUnpack"
	proc << "unzip" << "-d" << tmpUnpackPath << srcFile.path();
	kDebug(66666) << "unzip -d " << tmpUnpackPath.latin1() << " " << srcFile.path().latin1();

	// "unzip" spits out errorcodes > 0 only, 0 on success
	if ( proc.execute() != 0 )
	{
		KMessageBox::error(frame(), i18n("Extracting skin-archive failed"));
		// FIXME: Do I have to wait for the job to finish?
		// I'd say no because I don't care about the temp-dir
		// anyway after leaving this method :)
		KIO::del( tmpUnpackPath );
		return;
	}

	QDir tmpCnt = QDir ( tmpUnpackPath );
	tmpCnt.setFilter ( QDir::Dirs );

	QStringList dirList = tmpCnt.entryList();
	// Iterate trough all subdirs of tmpUnpackPath (including "."!)
	for ( unsigned int i = 0; i < dirList.count(); i++ )
	{
		// FIXME: is the following portable?
		if ( dirList[i] == ".." )
			continue;

		QDir tmpSubCnt = QDir( tmpUnpackPath + dirList[i], "*.rc;*.RC;*.Rc;*.rC", QDir::Name|QDir::IgnoreCase, QDir::Files );
		kDebug(66666) << "Searching for *.rc in " << QString(tmpUnpackPath+dirList[i]).latin1();

		// oh, no .rc file in current dir, let's go to next dir in list
		if ( tmpSubCnt.count() == 0 )
			continue;

		src = KUrl::encode_string(tmpUnpackPath+dirList[i]);
		dst = KUrl::encode_string(locateLocal("data","noatun/skins/kjofol/")); // destination to copy skindir into

		if ( dirList[i] == "." ) // zip did not contain a subdir, we have to create one
		{
			// skindir is named like the archive without extension (FIXME: extension is not stripped from name)

			int dotPos = srcFile.fileName().findRev('.');
			if ( dotPos > 0 ) // found a dot -> (hopefully) strip the extension
			{
				dst.addPath( srcFile.fileName().left(dotPos) );
			}
			else // we don't seem to have any extension, just append the archivename
			{
				dst.addPath( srcFile.fileName() );
			}

			kDebug(66666) << "want to create: " << dst.path();

			if ( !dst.isValid() )
			{
				KMessageBox::error(frame(),
					i18n("Installing new skin failed: Destination path is invalid.\n"
						"Please report a bug to the K-Jöfol maintainer"));
				KIO::del(tmpUnpackPath);
				return;
			}
			KIO::mkdir( dst );
		}

		if ( !src.isValid() || !dst.isValid() )
		{
			KMessageBox::error(frame(),
				i18n("Installing new skin failed: Either source or destination path is invalid.\n"
					"Please report a bug to the K-Jöfol maintainer") );
		}
		else
		{
			kDebug(66666) << "src: " << src.path();
			kDebug(66666) << "dst: " << dst.path();
			KIO::Job *job = KIO::copy(src,dst);
			connect(job, SIGNAL(result(KJob*)), this, SLOT(slotResult(KJob*)));
			skinInstalled = true;
		}
	} // END iterate trough dirList

	if ( !skinInstalled )
	{
		KMessageBox::sorry(frame(),
			i18n("No new skin has been installed.\nMake sure the archive contains a valid K-Jöfol skin")
		);
	}
	else
	{
		KMessageBox::information(frame(),
			i18n("The new skin has been successfully installed")
		);
	}

	KIO::del( tmpUnpackPath );
}


void KJPrefs::removeSelectedSkin()
{
	QString question = i18n("Are you sure you want to remove %1?\n"
		"This will delete the files installed by this skin ",
		mSkinselectorWidget->mSkins->currentText() );

	QString loadedSkin = KJConfig::self()->skinResource();
//	kDebug(66666) << "loaded Skin Name: " << QFileInfo(loadedSkin).baseName().latin1();

	if (KMessageBox::questionYesNo(frame(), question, i18n("Confirmation")) != KMessageBox::Yes)
		return;

	bool deletingCurrentSkin = ( mSkinselectorWidget->mSkins->currentText() == QFileInfo(loadedSkin).baseName() );

	// Now find the dir to delete !!!

	QString dirToDelete = QString ("");
	QStringList skinLocations = KGlobal::dirs()->findDirs("data", "noatun/skins/kjofol");

	// iterate through all paths where Noatun is searching for kjofol-skins
	for (uint i = 0; i < skinLocations.count(); ++i )
	{
		QStringList skinDirs = QDir(skinLocations[i]).entryList();

		// iterate trough all dirs containing a skin
		for (uint k = 0; k < skinDirs.count(); ++k )
		{
			QDir skinDirCnt = QDir ( skinLocations[i]+skinDirs[k], "*.rc", QDir::Name|QDir::IgnoreCase, QDir::Files );
			 // make a list of all .rc-files in a skindir
			QStringList rcFiles = skinDirCnt.entryList();

			// iterate trough all those rc.-files in a skindir
			for (uint j = 0; j < rcFiles.count(); j++ )
			{
				if ( rcFiles[j].left(rcFiles[j].length()-3) == mSkinselectorWidget->mSkins->currentText() ) // found skinname.rc :)
				{
					dirToDelete = QString ( skinLocations[i]+skinDirs[k] );
					kDebug(66666) << "FOUND SKIN @ " << dirToDelete.latin1();
				}
			}
		}
	}

	if ( dirToDelete.length() != 0 )
	{
		kDebug(66666) << "Deleting Skindir: " << dirToDelete.latin1();
		KIO::Job *job = KIO::del( dirToDelete, false, true );
		connect ( job, SIGNAL(result(KJob*)), this, SLOT(slotResult(KJob*)) );
	}

	int item = -1;
	// Fallback to kjofol-skin (the default one) if we've deleted the current skin
	if ( deletingCurrentSkin )
	{
		for ( int i = 0; i < mSkinselectorWidget->mSkins->count(); i++ )
		{ // FIXME: no check wether "kjofol" is ever found, well, it HAS to be in the list
			if ( mSkinselectorWidget->mSkins->text(i) == "kjofol" )
				item = i;
		}
	}
	else
		item = mSkinselectorWidget->mSkins->currentItem();

	if ( item != -1 )
		mSkinselectorWidget->mSkins->setCurrentItem( item );

	// update configuration
	if ( deletingCurrentSkin )
		save();
}


void KJPrefs::slotResult(KJob *job )
{
	if ( job->error() )
	{
		job->showErrorDialog(frame());
	}
	else
	{
		// Reload Skinlist
		load();
	}
}


/* =================================================================================== */


// takes name of rc-file without .rc at the end and returns full path to rc-file
QString KJPrefs::expand(const QString &s)
{
//	kDebug(66666) << "expand( "<< s.latin1() << " )";

	QStringList skinLocations = KGlobal::dirs()->findDirs("data", "noatun/skins/kjofol");

	// iterate through all paths where Noatun is searching for kjofol-skins
	for (uint i = 0; i < skinLocations.count(); ++i )
	{
		QStringList skinDirs = QDir(skinLocations[i]).entryList();

		// iterate trough all dirs containing a skin
		for (uint k = 0; k < skinDirs.count(); ++k )
		{
			QDir skinDirCnt = QDir(skinLocations[i]+skinDirs[k], "*.rc",
				QDir::Name | QDir::IgnoreCase, QDir::Files);
			// make a list of all .rc-files in a skindir
			QStringList rcFiles = skinDirCnt.entryList();

			// iterate trough all those rc.-files in a skindir
			for (uint j = 0; j < rcFiles.count(); j++ )
			{
				if ( rcFiles[j].left(rcFiles[j].length()-3) == s ) // found $s.rc :)
				{
					return (skinLocations[i] + skinDirs[k] + '/' + rcFiles[j]);
				}
			}
		}
	}
	return QString();
}

#include "kjofol_prefs.moc"
