/*
  Milk Chocolate userinterface for Noatun

  Copyright 1999 Charles Samuels <charles@kde.org>
*/
#include "userinterface.h"

#include <noatun/controls.h>
#include <noatun/global.h>
#include <noatun/playlist.h>
#include <noatun/player.h>
#include <noatun/stdaction.h>

#include <qgridlayout.h>
#include <qmenu.h>
#include <qpushbutton.h>
#include <qevent.h>

#include <kactioncollection.h>
#include <kstatusbar.h>
#include <kurl.h>
#include <kxmlguifactory.h>


K_EXPORT_COMPONENT_FACTORY(noatun_milkchocolate, PluginFactory<MCPlugin>("noatun_milkchocolate"))


// -----------------------------------------------------------------------------


MCPlugin::MCPlugin(const KComponentData &inst, Global *glob, const char* name)
	: Plugin(inst, glob, name)
{
	ui = new MilkChocolate(this, glob);
	ui->show();
}


Interface *MCPlugin::getInterface(const QString &n)
{
	if (n == "FrontendInterface")
		return this;
	return 0;
}


KXmlGuiWindow *MCPlugin::mainWindow()
{
	return ui;
}


// -----------------------------------------------------------------------------


MilkChocolate::MilkChocolate(Plugin *plugin, Global *nInst)
	: NMainWindow(plugin), nInstance(nInst)
{
	setAcceptDrops(true);
	setCaption(i18n("Noatun"));
	setWindowIcon(KIcon("noatun"));

	static const int buttonSize = 32; // TODO remove when we have a proper gui

	QWidget *wi = new QWidget(this);
	setCentralWidget(wi);

	// these automatically show up in the menubar
	StdAction::playlist(nInstance, actionCollection());
	StdAction::preferences(nInstance, actionCollection());
	StdAction::quit(nInstance, actionCollection());

	// temporary addition to show all working actions so they can be tested
	StdAction::playpause(nInstance, actionCollection());
	StdAction::back(nInstance, actionCollection());
	StdAction::forward(nInstance, actionCollection());
	StdAction::stop(nInstance, actionCollection());
	StdAction::play(nInstance, actionCollection());
	StdAction::pause(nInstance, actionCollection());


	QGridLayout *l = new QGridLayout(wi);
	{ // The buttons
		mBack=new QPushButton(wi);
		mBack->setFixedSize(buttonSize, buttonSize);
		mBack->setIcon(KIcon("noatunback"));
		connect(mBack, SIGNAL(clicked()), nInstance->player(), SLOT(previous()));
		mBack->setToolTip(i18n("Go backward in playlist"));

		mStop=new QPushButton(wi);
		mStop->setFixedSize(buttonSize, buttonSize);
		mStop->setIcon(KIcon("noatunstop"));
		connect(mStop, SIGNAL(clicked()), nInstance->player(), SLOT(stop()));
		mStop->setToolTip( i18n("Stop"));

		mPlay=new QPushButton(wi);
		mPlay->setCheckable(true);
		mPlay->setFixedSize(buttonSize, buttonSize);
		mPlay->setIcon(KIcon("noatunplay"));
		connect(mPlay, SIGNAL(clicked()), nInstance->player(), SLOT(playpause()));
		mPlay->setToolTip( i18n("Play"));

		mForward=new QPushButton(wi);
		mForward->setFixedSize(buttonSize, buttonSize);
		mForward->setIcon(KIcon("noatunforward"));
		connect(mForward, SIGNAL(clicked()), nInstance->player(), SLOT(next()));
		mForward->setToolTip(i18n("Go forward in playlist"));

		mPlaylist=new QPushButton(wi);
		mPlaylist->setCheckable(true);
		mPlaylist->setFixedSize(buttonSize, buttonSize);
		mPlaylist->setIcon(KIcon("noatunplaylist"));
		connect(mPlaylist, SIGNAL(clicked()), nInstance->playlist(), SLOT(toggleList()));
		mPlaylist->setToolTip(i18n("Playlist"));

		/*mLoop=new QPushButton(wi);
		mLoop->setFixedSize(buttonSize, buttonSize);
		mLoop->setIcon(KIcon("noatunloopnone"));
		connect(mLoop, SIGNAL(clicked()), nInstance->player(), SLOT(loop()));
		mLoop->setToolTip( i18n("Change loop style"));*/
	}

	mVolumeSlider = new L33tSlider(Qt::Horizontal, wi);
	mVolumeSlider->setMinimum(0);
	mVolumeSlider->setMaximum(100);
	mVolumeSlider->setPageStep(10);
	mVolumeSlider->setValue(nInstance->player()->volume());

	mSeekSlider = new L33tSlider(Qt::Horizontal, wi);
	mSeekSlider->setMinimum(0);
	mSeekSlider->setMaximum(1); // changed when loading file
	mSeekSlider->setPageStep(15000); // 15 seconds
	mSeekSlider->setSingleStep(2000); // 2 seconds
	mSeekSlider->setValue(0);

	l->addWidget(mBack    , 0, 0);
	l->addWidget(mStop    , 0, 1);
	l->addWidget(mPlay    , 0, 2);
	l->addWidget(mForward , 0, 3);
	l->addWidget(mPlaylist, 0, 4, Qt::AlignLeft);
	//l->addWidget(mLoop    , 0, 5);
	l->addWidget(mVolumeSlider, 1, 0, 1, 5);
	l->addWidget(mSeekSlider  , 2, 0, 2, 5);

	statusBar()->showMessage(i18n("No File Loaded"));
	statusBar()->insertPermanentItem(QString(), 1);
	statusBar()->insertPermanentItem(QString(), 2);

	connect(nInstance->player(), SIGNAL(playing()), SLOT(slotPlaying()));
	connect(nInstance->player(), SIGNAL(stopped()), SLOT(slotStopped()));
	connect(nInstance->player(), SIGNAL(paused()), SLOT(slotPaused()));
	connect(nInstance->player(), SIGNAL(volumeChanged(unsigned int)), SLOT(updateVolume(unsigned int)));
	connect(nInstance->player(), SIGNAL(positionChanged(int)), SLOT(updateSeekSlider(int)));
	connect(nInstance->player(), SIGNAL(lengthChanged(int)), SLOT(updateSeekLen(int)));
	connect(nInstance->player(), SIGNAL(currentItemChanged(const PlaylistItem &)), SLOT(currentItemChanged(const PlaylistItem &)));

	connect(nInstance->playlist(), SIGNAL(listShown()), SLOT(playlistShown()));
	connect(nInstance->playlist(), SIGNAL(listHidden()), SLOT(playlistHidden()));

	connect(mSeekSlider   , SIGNAL(sliderMoved(int)), SLOT(updatePositionText(int)));
	connect(mSeekSlider   , SIGNAL(userChanged(int)), SLOT(setPlayerPosition(int)));

	connect(mVolumeSlider , SIGNAL(sliderMoved(int)), SLOT(setPlayerVolume(int)));
	connect(mVolumeSlider , SIGNAL(userChanged(int)), SLOT(setPlayerVolume(int)));

	// Event Filter for context menu and mousewheel handling
	foreach(QObject *obj, children())
	{
		obj->installEventFilter(this);
	}

	setupGUI();
}

MilkChocolate::~MilkChocolate()
{
	// If cfg dialog is still open, delete it so it saves it's position
//	if(prefDlgExist)
//		delete prefDlg;
}

void MilkChocolate::dragEnterEvent(QDragEnterEvent *event)
{
	// accept uri drops only
	event->setAccepted(KUrl::List::canDecode(event->mimeData()));
}


void MilkChocolate::dropEvent(QDropEvent *event)
{
	KUrl::List uriList = KUrl::List::fromMimeData(event->mimeData());
	if (!uriList.isEmpty())
		nInstance->playlist()->addFile(uriList);
}


void MilkChocolate::slotPlaying()
{
	mPlay->setChecked(true);
	mStop->setEnabled(true);
	mPlay->setIcon(KIcon("noatunpause"));
}


void MilkChocolate::slotStopped()
{
	if (!nInstance->player()->currentItem())
		return;

	mStop->setEnabled(false);
	mPlay->setChecked(false);
	mSeekSlider->setValue(0);
	mPlay->setIcon(KIcon("noatunplay"));
}


void MilkChocolate::slotPaused()
{
	mStop->setEnabled(true);
	mPlay->setChecked(false);
	mPlay->setIcon(KIcon("noatunplay"));
}


void MilkChocolate::playlistShown()
{
	mPlaylist->setChecked(true);
}


void MilkChocolate::playlistHidden()
{
	mPlaylist->setChecked(false);
}


void MilkChocolate::updatePositionText(int msec)
{
	if (nInstance->player()->currentItem())
	{
		statusBar()->changeItem(Noatun::formatDuration(msec), 1);
	}
}


void MilkChocolate::setPlayerPosition(int msec)
{
	//kDebug(66666) << "SETTING NEW POS " << msec;
	nInstance->player()->setPosition(msec);
}


void MilkChocolate::setPlayerVolume(int volume)
{
	//kDebug(66666) << "SETTING NEW VOL " << volume;
	nInstance->player()->setVolume((unsigned int)volume);
}


void MilkChocolate::updateVolume(unsigned int volume)
{
	if (mVolumeSlider->currentlyPressed())
		return;

	mVolumeSlider->setValue(volume);
}


void MilkChocolate::updateSeekSlider(int position)
{
	if (!nInstance->player()->currentItem() || mSeekSlider->currentlyPressed())
		return;

	mSeekSlider->setValue(position);
	statusBar()->changeItem(Noatun::formatDuration(position), 1);
}

void MilkChocolate::updateSeekLen(int msecLen)
{
	mSeekSlider->setRange(0, msecLen);
	statusBar()->changeItem(nInstance->player()->lengthString(), 2);
}

void MilkChocolate::currentItemChanged(const PlaylistItem &newCurrentItem)
{
	kDebug(66666) ;

	bool isValidItem = !newCurrentItem.isNull();

	mPlay->setEnabled(isValidItem);

	mSeekSlider->setValue(0);
	mSeekSlider->setEnabled(isValidItem);
}

bool MilkChocolate::eventFilter(QObject *o, QEvent *e)
{
	if (e->type() == QEvent::Wheel)
	{
		wheelEvent(static_cast<QWheelEvent*>(e));
		return true;
	}
	return QWidget::eventFilter(o, e);
}


void MilkChocolate::wheelEvent(QWheelEvent *e)
{
	nInstance->player()->setVolume(mVolumeSlider->value() + (e->delta() / 100));
	e->setAccepted(true);
}

#include "userinterface.moc"
