/***************************************************************************
                     A simple database-driven playlist for Noatun
                    ---------------------------------------------
    begin                : 27.05.2005
    copyright            : (C) 2005 by Stefan Gehn
    email                : Stefan Gehn <mETz81@web.de>
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "simpledb.h"

#include <qsqldatabase.h>
#include <kdebug.h>
#include <kstandarddirs.h>
#include <kmessagebox.h>
#include <klocale.h>

DBItem::DBItem()
{
	id = -1;
	length = -1;
	next = 0;
}

DBItem::DBItem(const DBItem &o)
{
	id = o.id;
	url = o.url;
	length = o.length;
	properties = o.properties;
	next = o.next;
}

bool DBItem::isNull() const
{
	return (id == -1 && length == -1);
}

// ============================================================================

SimpleDB::SimpleDB(QObject *parent) : QObject(parent, "SimpleDB")
{
	QString dbPath = locateLocal("data", "noatun_simpleplaylist/noatun_simpleplaylist.db", true);

	kDebug(66666) << "dbPath = " << dbPath;

	db = QSqlDatabase::addDatabase("QSQLITE");
	db->setDatabaseName(dbPath);
	if(!db->open())
	{
		KMessageBox::detailedError(0, i18n("Could not open database"),
			db->lastError().text());
	}
	else
	{
		QString setupError = setup();
		if (!setupError.isEmpty())
		{
			KMessageBox::detailedError(0, i18n("Could not setup database tables"),
				setupError);
			db->close();
		}
	}
}

SimpleDB::~SimpleDB()
{
	kDebug(66666) ;

	QSqlQuery v("VACUUM", db);
	v.exec();

	if (db->isOpen())
		db->close();
}

QString SimpleDB::setup() // private
{
	QString dbError;
	if (!createTableIfMissing("items", "id INTEGER PRIMARY KEY, " \
		"url VARCHAR(2048) NOT NULL, length INTEGER DEFAULT -1, next INTEGER"))
	{
		dbError = db->lastError().text();
		return dbError;
	}
	if (!createTableIfMissing("properties", "item_id INTEGER NOT NULL, " \
		"key VARCHAR(32) NOT NULL, value VARCHAR(2048), " \
		"PRIMARY KEY (item_id, key)"))
	{
		dbError = db->lastError().text();
		return dbError;
	}

	if (!createTriggerIfMissing("delete_properties", "DELETE ON items " \
		"BEGIN  DELETE FROM properties WHERE item_id=old.id;  END;"))
	{
		dbError = db->lastError().text();
		return dbError;
	}

	if (!createIndexOn("items", "id"))
	{
		dbError = db->lastError().text();
		return dbError;
	}

	if (!createIndexOn("properties", "item_id"))
	{
		dbError = db->lastError().text();
		return dbError;
	}

	if (!createIndexOn("properties", "key"))
	{
		dbError = db->lastError().text();
		return dbError;
	}

	return QString();
}

bool SimpleDB::createTableIfMissing(const QString &tablename, const QString &fields) // private
{
	if (!db->tables(QSql::Tables).contains(tablename))
	{
		kDebug(66666) << "Creating TABLE " << tablename;

		QString queryString = QString("CREATE TABLE %1 (%2)").arg(tablename, fields);
		QSqlQuery m(QString::null, db);	//krazy:exclude=nullstrassign for old broken gcc
		m.setForwardOnly(true);
		m.prepare(queryString);

		if (!m.exec())
		{
			kDebug(66666) <<
				"Creating table " << tablename << " FAILED, error was '" <<
				db->lastError().text() << "'" << endl;
			return false;
		}
	}
	return true;
}

bool SimpleDB::createTriggerIfMissing(const QString &triggername, const QString &code) // private
{
	QSqlQuery m(QString::null, db);	//krazy:exclude=nullstrassign for old broken gcc
	m.setForwardOnly(true);

	m.prepare("SELECT count() FROM sqlite_master WHERE type='trigger' AND name=:n");
	m.bindValue(":n", triggername);
	if (m.exec() && m.first())
	{
		//kDebug(66666) << "Found TRIGGER " << triggername << " in database.";
		return true;
	}

	kDebug(66666) << "Creating TRIGGER " << triggername;

	m.prepare("CREATE TRIGGER " + triggername + ' ' + code);
	return m.exec();
}

bool SimpleDB::createIndexOn(const QString &tablename, const QString &columnname)
{
	QString indexName = tablename + '_' + columnname;
	indexName.remove(' ');
	indexName.remove(',');

	QSqlQuery q(QString::null, db);	//krazy:exclude=nullstrassign for old broken gcc
	q.prepare("SELECT count() FROM sqlite_master WHERE type='index' AND name=:n");
	q.bindValue(":n", indexName);
	if (q.exec() && q.first())
	{
		//kDebug(66666) << "Found INDEX " << indexName << " in database.";
		return true;
	}

	kDebug(66666) << "Creating INDEX " << indexName;

	q.prepare(QString("CREATE INDEX %1 ON %2 (%3)").arg(indexName, tablename, columnname));
	return q.exec();
}

int SimpleDB::getNextId(const QString &table, const QString &field) const // private
{
	if (!db->isOpen())
		return -1;

	int id = -1;
	QString querystring = QString("SELECT max(%1) FROM %2").arg(field, table);
	QSqlQuery q(QString::null, db);	//krazy:exclude=nullstrassign for old broken gcc
	q.setForwardOnly(true);
	q.prepare(querystring);
	if (q.exec() && q.first())
	{
		id = q.value(0).toInt() + 1;
	}
	else
	{
		kDebug(66666) <<
			"failed for field '" << field << "' in table '"<< table << "'." << endl;
	}
	return id;
}

int SimpleDB::getFirstItemId() const
{
	kDebug(66666) ;

	int id = -1;
	if (db->isOpen())
	{
		QSqlQuery q(QString::null, db);	//krazy:exclude=nullstrassign for old broken gcc
		q.setForwardOnly(true);
		q.prepare("SELECT min(id) FROM items");
		if (q.exec() && q.first())
			id = q.value(0).toInt();
	}
	return id;
}

int SimpleDB::getNextItemId(int currentItemId) const
{
	int id = -1;
	if (db->isOpen())
	{
		QString querystring = QString("SELECT id FROM items WHERE id > %1 LIMIT 1").arg(currentItemId);
		QSqlQuery q(QString::null, db);	//krazy:exclude=nullstrassign for old broken gcc
		q.setForwardOnly(true);
		q.prepare(querystring);
		if (q.exec() && q.first())
			id = q.value(0).toInt();
	}
	return id;
}

int SimpleDB::getLastItemIdInList() const
{
	int id = -1;
	if (db->isOpen())
	{
		QSqlQuery q(QString::null, db);	//krazy:exclude=nullstrassign for old broken gcc
		q.setForwardOnly(true);
		q.prepare("SELECT id FROM items WHERE next=0");
		if (q.exec() && q.first())
			id = q.value(0).toInt();
	}
	return id;
}

bool SimpleDB::insertItem(DBItem &item, int insertAfterId)
{
	if (!db->isOpen())
		return false;

	kDebug(66666) << "item.url: " << item.url << ", insertAfterId = " << insertAfterId;

	bool ta = db->transaction();
	int id = getNextId("items", "id");
	if (id != -1)
	{
		if (insertAfterId == -1)
			insertAfterId = getLastItemIdInList();

		DBItem itemAbove = getItem(insertAfterId, false);

		QSqlQuery query(QString::null, db);	//krazy:exclude=nullstrassign for old broken gcc
		query.setForwardOnly(true);
		query.prepare("INSERT INTO items (id, url, length, next) VALUES (:id, :url, :l, :n)");
		query.bindValue(":id", id);
		query.bindValue(":url", item.url.url());
		query.bindValue(":l", item.length);
		 // next is either the old follower of our above item or -1 if there is no item above us (i.e. insert at top of the playlist)
		query.bindValue(":n", itemAbove.next);

		kDebug(66666) << "inserting new item " << id <<
			" after item " << itemAbove.id <<
			", next will be " << itemAbove.next << endl;

		if (query.exec())
		{
			Noatun::PropertyMap::ConstIterator it;
			for (it = item.properties.begin(); it != item.properties.end(); ++it)
			{
				if (!setProp(id, it.key(), it.data()))
				{
					if (ta)
						db->rollback();
					kDebug(66666) <<
						"INSERTING item with id " << id << " FAILED" << endl;
					return false;
				}
			}

			item.next = itemAbove.next;
			item.id = id;

			if (!itemAbove.isNull())  // update our above item with our own shiny id
				setNext(itemAbove, id);
			if (ta)
				db->commit();
			kDebug(66666) << "INSERTED item with id " << id;

			emit itemInserted(item);
			return true;
		}
	}

	if (ta)
		db->rollback();
	kDebug(66666) <<
		"INSERTING item with id " << id << " FAILED" << endl;
	return false;
}

bool SimpleDB::deleteItem(int id) // public
{
	if (!db->isOpen() || id < 0)
		return false;

	DBItem deletedItem = getItem(id, false);
	if (deletedItem.isNull())
		return false;

	bool ta = db->transaction();

	QSqlQuery q(QString::null, db);	//krazy:exclude=nullstrassign for old broken gcc
	q.setForwardOnly(true);
	q.prepare("DELETE FROM items WHERE id=:id");
	q.bindValue(":id", deletedItem.id);
	if (q.exec() /*&& deleteAllProperties(deletedItem)*/ && replaceNext(deletedItem.id, deletedItem.next))
	{
		if (ta)
			db->commit();
		kDebug(66666) << "DELETED item with id " << deletedItem.id;
		emit itemDeleted(deletedItem.id);
		return true;
	}
	if (ta)
		db->rollback();

	kDebug(66666) <<
		"DELETING item with id " << deletedItem.id << " FAILED" << endl;

	return false;
}

bool SimpleDB::moveItem(int itemId, int nearId, bool before) // public
{
	if (!db->isOpen() || itemId < 0)
		return false;

	kDebug(66666) << "moving item " << itemId <<
		(before ? " before " : " after ") << nearId << endl;

	DBItem movedItem = getItem(itemId, false);
	if (movedItem.isNull())
		return false;

	int oldNext = movedItem.next;

	// inform CURRENT item above us that we will be gone now
	replaceNext(movedItem.id, movedItem.next);

	// get NEW item which we will be moved after/before
	DBItem itemAbove;
	if (before)
		itemAbove = getItemBefore(nearId);
	else
		itemAbove = getItem(nearId, false);
	int newNext = (before ? nearId : itemAbove.next); // our new next item

	// inform NEW item above us about us being the next item now
	// does nothing if there is no item above us
	if (!setNext(itemAbove, movedItem.id))
		return false;

	// update ourselves with the old next-item referenced by above item
	if (!setNext(movedItem, newNext))
		return false;

	emit itemUpdatedPosition(movedItem.id, oldNext, newNext);
	return true;
}

bool SimpleDB::replaceNext(int oldNext, int newNext) // private
{
	kDebug(66666) <<
		"replacing all next = " << oldNext << " with next = " << newNext << endl;

	QSqlQuery q(QString::null, db);	//krazy:exclude=nullstrassign for old broken gcc
	q.setForwardOnly(true);
	q.prepare("UPDATE items set next=:n WHERE next=:o");
	q.bindValue(":n", newNext);
	q.bindValue(":o", oldNext);
	return q.exec();
}

bool SimpleDB::setNext(DBItem &item, int newNextId) // private
{
	if (item.isNull())
		return true;

	if (updateField(item.id, "next", QVariant(newNextId)))
	{
		kDebug(66666) << "next for " << item.id << " is now " << newNextId;
		item.next = newNextId;
		return true;
	}
	return false;
}

bool SimpleDB::setUrl(int id, const KUrl &url) // public
{
	if (updateField(id, "url", QVariant(url.url())));
	{
		emit itemUpdated(id);
		return true;
	}
	return false;
}

bool SimpleDB::setLength(int id, int length) // public
{
	if (updateField(id, "length", QVariant(length)));
	{
		emit itemUpdatedLength(id, length);
		return true;
	}
	return false;
}

bool SimpleDB::updateField(int id, const QString &field, const QVariant &val) // private
{
	if (!db->isOpen() || id < 0)
		return false;

	QString querystring = QString("UPDATE items SET %1=:val WHERE id=:id").arg(field);
	QSqlQuery q(QString::null, db);	//krazy:exclude=nullstrassign for old broken gcc
	q.setForwardOnly(true);
	q.prepare(querystring);
	q.bindValue(":val", val);
	q.bindValue(":id", id);
	if (!q.exec())
	{
		kDebug(66666) <<
			"query: " << q.executedQuery() <<
			" | error: " << db->lastError().text() << endl;
		return false;
	}
	return true;
}

bool SimpleDB::setProperties(int id, const Noatun::PropertyMap &properties) // public
{
	if (!db->isOpen() || id < 0)
		return false;

	kDebug(66666) << "for item with id " << id;

	bool ta = db->transaction();
	Noatun::PropertyMap::ConstIterator it;
	for (it = properties.begin(); it != properties.end(); ++it)
	{
		if (!setProp(id, it.key(), it.data()))
		{
			if (ta)
				db->rollback();
			return false;
		}
	}
	if (ta)
		db->commit();

	emit itemUpdated(id);
	return true;
}

bool SimpleDB::setProperty(int id, const QString &key, const QString &val) // public
{
	if (setProp(id, key, val))
	{
		emit itemUpdated(id);
		return true;
	}
	return false;
}

bool SimpleDB::setProp(int id, const QString &key, const QString &val) // private
{
	if (!db->isOpen() || id < 0)
		return false;

	QSqlQuery q(QString::null, db);	//krazy:exclude=nullstrassign for old broken gcc
	q.setForwardOnly(true);
	q.prepare("INSERT OR REPLACE INTO properties (item_id, key, value) VALUES (:item_id, :key, :val)");
	q.bindValue(":item_id", id);
	q.bindValue(":key", key);
	q.bindValue(":val", val);
	if (!q.exec())
	{
		kDebug(66666) <<
			"query: " << q.executedQuery() <<
			" | error: " << db->lastError().text() << endl;
		return false;
	}
	return true;
}

bool SimpleDB::deleteProperty(int id, const QString &key) // public
{
	if (!db->isOpen() || id < 0)
		return false;

	QSqlQuery q(QString::null, db);	//krazy:exclude=nullstrassign for old broken gcc
	q.setForwardOnly(true);
	q.prepare("DELETE FROM properties WHERE item_id=:id AND key=:key");
	q.bindValue(":id", id);
	q.bindValue(":key", key);
	if (!q.exec())
	{
		kDebug(66666) <<
			"query: " << q.executedQuery() <<
			" | error: " << db->lastError().text() << endl;
		return false;
	}

	emit itemUpdated(id);
	return true;
}

Noatun::PropertyMap SimpleDB::getProperties(int id) const // private
{
	Noatun::PropertyMap ret;
	if (!db->isOpen() || id < 0)
		return ret;

	//kDebug(66666) << "for item with id " << id;

	QSqlQuery q(QString::null, db);	//krazy:exclude=nullstrassign for old broken gcc
	q.setForwardOnly(true);
	q.prepare("SELECT key, value FROM properties WHERE item_id=:id");
	q.bindValue(":id", id);
	if (q.exec())
	{
		while (q.next())
			ret.insert(q.value(0).toString(), q.value(1).toString());
	}
	else
	{
		kDebug(66666) <<
			"query: " << q.executedQuery() <<
			" | error: " << db->lastError().text() << endl;
	}
	return ret;
}

QMap<int, DBItem> SimpleDB::getOrderedItems() const
{
	QMap<int, DBItem> ret;
	if (!db->isOpen())
		return ret;

	kDebug(66666) ;

	QSqlQuery q(QString::null, db);	//krazy:exclude=nullstrassign for old broken gcc
	q.setForwardOnly(true);
	q.prepare("SELECT id, url, length, next FROM items");
	if (q.exec())
	{
		while (q.next())
		{
			DBItem item;
			item.id = q.value(0).toInt();
			item.url = KUrl(q.value(1).toString());
			item.length = q.value(2).toInt();
			item.next = q.value(3).toInt();
			item.properties = getProperties(item.id);
			ret.insert(item.next, item);
		}
	}
	else
	{
		kDebug(66666) <<
			"query: " << q.executedQuery() <<
			" | error: " << db->lastError().text() << endl;
	}
	return ret;
}

DBItem SimpleDB::getItem(int id, bool withProps) const // public
{
	return getItemByField("id", id, withProps);
}

DBItem SimpleDB::getItemBefore(int id, bool withProps) const // public
{
	return getItemByField("next", id, withProps);
}

DBItem SimpleDB::getItemByField(const QString &field, int id, bool withProps) const // private
{
	DBItem ret;
	if (!db->isOpen() || id < 0)
		return ret;

	QString queryString = QString("SELECT url, length, next FROM items WHERE %1=:id").arg(field);
	QSqlQuery q(QString::null, db);	//krazy:exclude=nullstrassign for old broken gcc
	q.setForwardOnly(true);
	q.prepare(queryString);
	q.bindValue(":id", id);
	if (q.exec() && q.first())
	{
		ret.id = id;
		ret.url = KUrl(q.value(0).toString());
		ret.length = q.value(1).toInt();
		ret.next = q.value(2).toInt();
		if (withProps)
			ret.properties = getProperties(id);
	}
	else
	{
		kDebug(66666) <<
			"query: " << q.executedQuery() <<
			", error: " << db->lastError().text() << endl;
	}
	return ret;
}

#include "simpledb.moc"
