/* -------------------------------------------------------------

   toplevel.cpp (part of The KDE Dictionary Client)

   Copyright (C) 2000-2001 Christian Gebauer <gebauer@kde.org>
   (C) by Matthias Hölzer 1998

   This file is distributed under the Artistic License.
   See LICENSE for details.

   -------------------------------------------------------------

   TopLevel   The toplevel widget of Kdict.

 ------------------------------------------------------------- */

#include <qclipboard.h>
#include <Q3CString>
#include <QSplitter>

#include <kstandardshortcut.h>
#include <kstandardaction.h>
#include <kapplication.h>
#include <kstatusbar.h>
#include <klocale.h>
#include <kwin.h>
#include <kedittoolbar.h>
#include <kdebug.h>
#include <KGlobal>
#include <KActionCollection>
#include <KToggleAction>
#include <KToolBarPopupAction>

#include "actions.h"
#include "dict.h"
#include "options.h"
#include "queryview.h"
#include "matchview.h"
#include "sets.h"
#include "toplevel.h"
#include "kdictadaptor.h"

// cut a QString and add "..."
QString getShortString(QString str,unsigned int length)
{
  if (str.length()>length) {
    str.truncate(length-3);
    str.append("...");
  }
  return str;
}


DictInterface *interface;
GlobalData *global;


TopLevel::TopLevel(QWidget* parent)
  : KMainWindow(parent, Qt::WType_TopLevel),
    optDlg(0L), setsDlg(0L), stopRef(0)
{
  (void)new KdictAdaptor(this);
   QDBusConnection::sessionBus().registerObject("/Kdict", this);
  kapp->setMainWidget(this);

  global = new GlobalData();
  global->topLevel = this;
  global->read();
  interface = new DictInterface();
  connect(interface,SIGNAL(infoReady()),SLOT(stratDbChanged()));
  connect(interface,SIGNAL(started(const QString&)),SLOT(clientStarted(const QString&)));
  connect(interface,SIGNAL(stopped(const QString&)),SLOT(clientStopped(const QString&)));

  queryView = new QueryView(this);
  connect(queryView,SIGNAL(defineRequested(const QString&)),SLOT(define(const QString&)));
  connect(queryView,SIGNAL(matchRequested(const QString&)),SLOT(match(const QString&)));
  connect(queryView,SIGNAL(clipboardRequested()),SLOT(defineClipboard()));
  connect(queryView,SIGNAL(enableCopy(bool)),SLOT(enableCopy(bool)));
  connect(queryView,SIGNAL(enablePrintSave()),SLOT(enablePrintSave()));
  connect(queryView,SIGNAL(renderingStarted()),SLOT(renderingStarted()));
  connect(queryView,SIGNAL(renderingStopped()),SLOT(renderingStopped()));
  connect(queryView,SIGNAL(newCaption(const QString&)),SLOT(newCaption(const QString&)));

  matchView = new MatchView();
  connect(matchView,SIGNAL(defineRequested(const QString&)),SLOT(define(const QString&)));
  connect(matchView,SIGNAL(matchRequested(const QString&)),SLOT(match(const QString&)));
  connect(matchView,SIGNAL(clipboardRequested()),SLOT(matchClipboard()));
  connect(matchView,SIGNAL(windowClosed()),SLOT(toggleMatchListShow()));
  connect(&resetStatusbarTimer,SIGNAL(timeout()),SLOT(resetStatusbar()));

  setupStatusBar();
  setupActions();
  recreateGUI();
  buildHistMenu();

  if (global->showMatchList)
  {    // show splitter, html view & match list
    splitter = new QSplitter(Qt::Horizontal,this);
    splitter->setOpaqueResize( KGlobalSettings::opaqueResize() );
    queryView->reparent(splitter,0,queryView->pos(),true);
    matchView->reparent(splitter,0,matchView->pos(),true);
    setCentralWidget(splitter);
    splitter->setResizeMode(matchView,QSplitter::KeepSize);
    adjustMatchViewSize();
  }
  else
  {                       // show only html view
      setCentralWidget(queryView);
      matchView->hide();
  }

  //apply settings
  resize(600,390);
  applyMainWindowSettings(KGlobal::config().data(),"toplevel_options");
  stratDbChanged();              // fill combos, build menus

  actQueryCombo->setFocus();        // place cursor in combobox
}


TopLevel::~TopLevel()
{
}


void TopLevel::normalStartup()
{
  if (global->defineClipboard)
    defineClipboard();
}

// ******* D-Bus Interface ********************************************************

void TopLevel::quit()
{
  kDebug(5004) << "*D-Bus call* TopLevel::quit()" << endl;
  kapp->closeAllWindows();
}


void TopLevel::makeActiveWindow()
{
  kDebug(5004) << "*D-Bus call* TopLevel::makeActiveWindow()" << endl;
  raiseWindow();
}


void TopLevel::definePhrase(QString phrase)
{
  kDebug(5004) << "*D-Bus call* TopLevel::definePhrase()" << endl;
  define(phrase);
  raiseWindow();
}


void TopLevel::matchPhrase(QString phrase)
{
  kDebug(5004) << "*D-Bus call* TopLevel::matchPhrase()" << endl;
  match(phrase);
  raiseWindow();
}


void TopLevel::defineClipboardContent()
{
  kDebug(5004) << "*D-Bus call* TopLevel::defineClipboardContent()" << endl;
  defineClipboard();
  raiseWindow();
}


void TopLevel::matchClipboardContent()
{
  kDebug(5004) << "*D-Bus call* TopLevel::matchClipboardContent()" << endl;
  matchClipboard();
  raiseWindow();
}


QStringList TopLevel::getDatabases()
{
  kDebug(5004) << "*D-Bus call* TopLevel::getDatabases()" << endl;
  return global->databases;
}


QString TopLevel::currentDatabase()
{
  kDebug(5004) << "*D-Bus call* TopLevel::currentDatabase()" << endl;
  return global->databases[global->currentDatabase];
}


QStringList TopLevel::getStrategies()
{
  kDebug(5004) << "*D-Bus call* TopLevel::getStrategies()" << endl;
  return global->strategies;
}


QString TopLevel::currentStrategy()
{
  kDebug(5004) << "*D-Bus call* TopLevel::currentStrategy()" << endl;
  return global->strategies[global->currentStrategy];
}


bool TopLevel::setDatabase(QString db)
{
  kDebug(5004) << "*D-Bus call* TopLevel::setDatabase()" << endl;

  int newCurrent = global->databases.findIndex(db);
  if (newCurrent == -1)
    return false;
  else {
    global->currentDatabase = newCurrent;
    actDbCombo->setCurrentItem(global->currentDatabase);
    return true;
  }
}


bool TopLevel::setStrategy(QString strategy)
{
  kDebug(5004) << "*D-Bus call* TopLevel::setStrategy()" << endl;

  return matchView->selectStrategy(strategy);
}


bool TopLevel::historyGoBack()
{
  kDebug(5004) << "*D-Bus call* TopLevel::historyGoBack()" << endl;

  if (!queryView->browseBackPossible())
    return false;
  else {
    queryView->browseBack();
    return true;
  }
}


bool TopLevel::historyGoForward()
{
  kDebug(5004) << "*D-Bus call* TopLevel::historyGoForward()" << endl;

  if (!queryView->browseForwardPossible())
    return false;
  else {
    queryView->browseForward();
    return true;
  }
}

// *******************************************************************************

void TopLevel::define(const QString &query)
{
  kDebug(5004) << "TopLevel::define()" << endl;
  actQueryCombo->setEditText(query);
  doDefine();
}


void TopLevel::defineClipboard()
{
  kDebug(5004) << "TopLevel::defineClipboard()" << endl;
  QString text = kapp->clipboard()->text(QClipboard::Selection);
  if (text.isEmpty()) {
    text = kapp->clipboard()->text(QClipboard::Clipboard);
  }
  define(text);
}


void TopLevel::match(const QString &query)
{
  kDebug(5004) << "TopLevel::match()" << endl;
  actQueryCombo->setEditText(query);
  doMatch();
}


void TopLevel::matchClipboard()
{
  kDebug(5004) << "TopLevel::matchClipboard()" << endl;
  QString text = kapp->clipboard()->text(QClipboard::Selection);
  if (text.isEmpty()) {
    text = kapp->clipboard()->text(QClipboard::Clipboard);
  }
  match(text);
}


bool TopLevel::queryClose()
{
  kDebug(5004) << "TopLevel::queryClose()" << endl;

  saveMainWindowSettings(KGlobal::config().data(),"toplevel_options");
  saveMatchViewSize();
  global->queryComboCompletionMode = actQueryCombo->completionMode();

  global->write();

  return true;
}


void TopLevel::setupActions()
{
  // file menu...
  actSave = KStandardAction::save(queryView, SLOT(saveQuery()), actionCollection());
  actionCollection()->addAction("file_save", actSave);
  actSave->setText(i18n("&Save As..."));
  actSave->setEnabled(false);
  actPrint = KStandardAction::print(queryView, SLOT(printQuery()), actionCollection());
  actionCollection()->addAction("file_print", actPrint);
  actPrint->setEnabled(false);
  actStartQuery = actionCollection()->addAction("start_query");
  actStartQuery->setText(i18n("St&art Query"));
  actStartQuery->setIcon(KIcon("reload"));
  connect(actStartQuery, SIGNAL(triggered()), SLOT(doDefine()));

  actStopQuery = actionCollection()->addAction("stop_query");
  actStopQuery->setText(i18n("St&op Query"));
  actStopQuery->setIcon(KIcon("stop"));
  connect(actStopQuery, SIGNAL(triggered()), SLOT(stopClients()));
  actStopQuery->setEnabled(false);
  KStandardAction::quit(kapp, SLOT(closeAllWindows()), actionCollection());

  // edit menu...
  actCopy = KStandardAction::copy(queryView, SLOT(copySelection()), actionCollection());
  actCopy->setEnabled(false);
  KStandardAction::selectAll(queryView, SLOT(selectAll()), actionCollection());

  QAction *defineClipboardAction = actionCollection()->addAction("define_clipboard");
  defineClipboardAction->setText(i18n("&Define Clipboard Content"));
  defineClipboardAction->setIcon(KIcon("define_clip"));
  connect(defineClipboardAction, SIGNAL(triggered()), SLOT(defineClipboard()));

  QAction *matchClipboardAction = actionCollection()->addAction("match_clipboard");
  matchClipboardAction->setText(i18n("&Match Clipboard Content"));
  connect(matchClipboardAction, SIGNAL(triggered()), SLOT(matchClipboard()));

  KStandardAction::find(queryView, SLOT(showFindDialog()), actionCollection());

  // history menu...
  actBack = new KToolBarPopupAction(KIcon("back"), i18n("&Back"), actionCollection());
  actionCollection()->addAction("browse_back", actBack);
  connect(actBack, SIGNAL(triggered()), queryView, SLOT(browseBack()));
  actBack->setShortcuts(KStandardShortcut::shortcut(KStandardShortcut::Back));
  actBack->setDelayed(true);
  actBack->setStickyMenu(false);
  actBack->setEnabled(false);
  actForward = new KToolBarPopupAction(KIcon("forward"), i18n("&Forward"), actionCollection());
  actionCollection()->addAction("browse_forward", actForward);
  connect(actForward, SIGNAL(triggered()), queryView, SLOT(browseForward()));
  actForward->setShortcuts(KStandardShortcut::shortcut(KStandardShortcut::Forward));
  actForward->setDelayed(true);
  actForward->setStickyMenu(false);
  actForward->setEnabled(false);

  QAction *clearQueryAction = actionCollection()->addAction("clear_history");
  clearQueryAction->setText(i18n("&Clear History"));
  connect(clearQueryAction, SIGNAL(triggered()), SLOT(clearQueryHistory()));

  // server menu...
  QAction *getCapabilitiesAction = actionCollection()->addAction("get_capabilities");
  getCapabilitiesAction->setText(i18n("&Get Capabilities"));
  connect(getCapabilitiesAction, SIGNAL(triggered()), interface, SLOT(updateServer()));

  QAction *editSetsAction = actionCollection()->addAction("edit_sets");
  editSetsAction->setText(i18n("Edit &Database Sets..."));
  editSetsAction->setIcon(KIcon("edit"));
  connect(editSetsAction, SIGNAL(triggered()), SLOT(showSetsDialog()));

  QAction *summaryAction = actionCollection()->addAction("db_summary");
  summaryAction->setText(i18n("&Summary"));
  connect(summaryAction, SIGNAL(triggered()), interface, SLOT(showDatabases()));

  QAction *strategyInfoAction = actionCollection()->addAction("strategy_info");
  strategyInfoAction->setText(i18n("S&trategy Information"));
  connect(strategyInfoAction, SIGNAL(triggered()), interface, SLOT(showStrategies()));

  QAction *serverInfoAction = actionCollection()->addAction("server_info");
  serverInfoAction->setText(i18n("&Server Information"));
  connect(serverInfoAction, SIGNAL(triggered()), interface, SLOT(showInfo()));

  // settings menu...
  createStandardStatusBarAction();
  setStandardToolBarMenuEnabled(true);

  actShowMatchList = new KToggleAction(i18n("Show &Match List") , actionCollection());
  actionCollection()->addAction("show_match", actShowMatchList);
  connect(actShowMatchList, SIGNAL(triggered()), SLOT(toggleMatchListShow()));
  actShowMatchList->setCheckedState(KGuiItem(i18n("Hide &Match List")));
  actShowMatchList->setChecked(global->showMatchList);
//FIXME
#if 0
  KStandardAction::keyBindings(guiFactory(), SLOT(configureShortcuts()),
actionCollection());
#endif
  KStandardAction::configureToolbars(this, SLOT(slotConfToolbar()), actionCollection());
  KStandardAction::preferences(this, SLOT(showOptionsDialog()), actionCollection());

  // toolbar...
  QAction *queryEraseAction = actionCollection()->addAction("clear_query");
  serverInfoAction->setText(i18n("Clear Input Field"));
  serverInfoAction->setIcon(KIcon("query_erase"));
  connect(serverInfoAction, SIGNAL(triggered()), SLOT(clearInput()));

  actQueryLabel = new DictLabelAction(i18n("&Look for:"), actionCollection());
  actionCollection()->addAction("look_label", actQueryLabel);
  actQueryCombo = new DictComboAction(i18n("Query"), actionCollection(), true,true);
  actionCollection()->addAction("query_combo", actQueryCombo);
  connect(actQueryCombo,SIGNAL(activated(const QString &)), SLOT(define(const QString&)));
  actQueryCombo->setCompletionMode(global->queryComboCompletionMode);
  actDbLabel = new DictLabelAction(i18n("&in"), actionCollection());
  actionCollection()->addAction("in_label", actDbLabel);
  actDbCombo = new DictComboAction(i18n("Databases"), actionCollection(), false,false);
  actionCollection()->addAction("db_combo", actDbCombo);
  connect(actDbCombo,SIGNAL(activated(int)),SLOT(databaseSelected(int)));
  actDefineBtn = new DictButtonAction(i18n("&Define"), actionCollection());
  connect(actDefineBtn, SIGNAL(triggered()), SLOT(doDefine()));
  actionCollection()->addAction("define_btn", actDefineBtn);
  actMatchBtn = new DictButtonAction(i18n("&Match"), actionCollection());
  connect(actMatchBtn, SIGNAL(triggered()), SLOT(doMatch()));
  actionCollection()->addAction("match_btn", actMatchBtn);

  queryView->setActions(actBack,actForward,actQueryCombo);
}


void TopLevel::setupStatusBar()
{
  statusBar()->insertItem(i18n(" Ready "),0,2);
  statusBar()->setItemAlignment(0,Qt::AlignLeft | Qt::AlignVCenter);

  QString serverInfo;
  if (global->authEnabled)
    serverInfo = QString(" %1@%2:%3 ").arg(getShortString(global->user,50))
                                    .arg(getShortString(global->server,50))
                                    .arg(global->port);
  else
    serverInfo = QString(" %1:%3 ").arg(getShortString(global->server,50))
                                 .arg(global->port);
  statusBar()->insertItem(serverInfo, 1,3);
  statusBar()->setItemAlignment(1,Qt::AlignLeft | Qt::AlignVCenter);
}


void TopLevel::recreateGUI()
{
  kDebug(5004) << "TopLevel::recreateGUI()" << endl;
  createGUI("kdictui.rc");
  actQueryCombo->setList(global->queryHistory);
  actQueryCombo->clearEdit();
  actQueryLabel->setBuddy(actQueryCombo->widget());

  actDbCombo->setList(global->databases);
  actDbCombo->setCurrentItem(global->currentDatabase);
  actDbLabel->setBuddy(actDbCombo->widget());
  int bwidth;
  if (actDefineBtn->widthHint() > actMatchBtn->widthHint())
    bwidth = actDefineBtn->widthHint();
  else
    bwidth = actMatchBtn->widthHint();
  actDefineBtn->setWidth(bwidth);
  actMatchBtn->setWidth(bwidth);
}


// add text in the query-combobox to the history
void TopLevel::addCurrentInputToHistory()
{
  QString text(actQueryCombo->currentText());

  // maintain queryHistory
  global->queryHistory.remove(text);                     // no double entries
  global->queryHistory.prepend(text);                   // prepend new item
  while (global->queryHistory.count()>global->maxHistEntrys)    // shorten list
    global->queryHistory.remove(global->queryHistory.fromLast());

  actQueryCombo->setList(global->queryHistory);
  actQueryCombo->setCurrentItem(0);
  buildHistMenu();
}



// erase text in query-combobox
void TopLevel::clearInput()
{
  actQueryCombo->clearEdit();
  actQueryCombo->setFocus();      // place cursor in combobox
}


// define text in the combobox
void TopLevel::doDefine()
{
  QString text(actQueryCombo->currentText());

  if (!text.isEmpty())
  {
    addCurrentInputToHistory();
    actQueryCombo->selectAll();
    interface->define(text);
  }
}


void TopLevel::doMatch()
{
  QString text(actQueryCombo->currentText());

  if (!text.isEmpty())
  {
    addCurrentInputToHistory();
    actQueryCombo->selectAll();

    if (!global->showMatchList)
    {
      toggleMatchListShow();
    }

    matchView->match(text);
    setCaption(getShortString(text.simplified(),70));
  }
}


void TopLevel::stopClients()
{
  interface->stop();
  queryView->stop();
}


// rebuild history menu on demand
void TopLevel::buildHistMenu()
{
  unplugActionList("history_items");

  historyActionList.clear();

  unsigned int i = 0;
  while ((i<10)&&(i<global->queryHistory.count())) {
    QAction *action = new KAction(getShortString(global->queryHistory[i],70), actionCollection());
    connect(action, SIGNAL(triggered()), SLOT(queryHistMenu()));
    historyActionList.append(action);
    i++;
  }

  plugActionList("history_items", historyActionList);
}


// process a query via the history menu
void TopLevel::queryHistMenu()
{
  Q3CString name = sender()->name();
  if (!name.isEmpty())
    define(QString::fromUtf8(name));
}


void TopLevel::clearQueryHistory()
{
  global->queryHistory.clear();
  actQueryCombo->clear();
  buildHistMenu();
}


// fill combos, rebuild menus
void TopLevel::stratDbChanged()
{
  actDbCombo->setList(global->databases);
  actDbCombo->setCurrentItem(global->currentDatabase);
  matchView->updateStrategyCombo();

  unplugActionList("db_detail");

  dbActionList.clear();

  for (unsigned int i=0;i<global->serverDatabases.count();i++)
  {
    QAction *action = new KAction(global->serverDatabases[i], actionCollection());
    connect(action, SIGNAL(triggered()), SLOT(dbInfoMenuClicked()));
    dbActionList.append(action);
  }

  plugActionList("db_detail", dbActionList);
}


void TopLevel::dbInfoMenuClicked()
{
  Q3CString name = sender()->name();
  if (!name.isEmpty())
    interface->showDbInfo(name);
}


void TopLevel::databaseSelected(int num)
{
  global->currentDatabase = num;
}


void TopLevel::enableCopy(bool selected)
{
  actCopy->setEnabled(selected);
}


void TopLevel::enablePrintSave()
{
  actSave->setEnabled(true);
  actPrint->setEnabled(true);
}


void TopLevel::clientStarted(const QString &message)
{
  statusBar()->changeItem(message,0);
  resetStatusbarTimer.stop();
  stopRef++;
  actStopQuery->setEnabled(stopRef>0);       // enable stop-icon
  kapp->setOverrideCursor(Qt::WaitCursor);
}


void TopLevel::clientStopped(const QString &message)
{
  statusBar()->changeItem(message,0);
  resetStatusbarTimer.start(4000);
  if (stopRef > 0)
    stopRef--;
  actStopQuery->setEnabled(stopRef>0);      // disable stop-icon
  kapp->restoreOverrideCursor();
}


void TopLevel::resetStatusbar()
{
  resetStatusbarTimer.stop();
  statusBar()->changeItem(i18n(" Ready "),0);
}


void TopLevel::renderingStarted()
{
  stopRef++;
  actStopQuery->setEnabled(stopRef>0);      // disable stop-icon
  kapp->setOverrideCursor(Qt::WaitCursor);
}


void TopLevel::renderingStopped()
{
  if (stopRef > 0)
    stopRef--;
  actStopQuery->setEnabled(stopRef>0);      // disable stop-icon
  kapp->restoreOverrideCursor();
}


void TopLevel::newCaption(const QString &s)
{
  setCaption(s);
}

void TopLevel::toggleMatchListShow()
{
  saveMatchViewSize();
  if (global->showMatchList) // list is visible -> hide it
  {
    global->showMatchList = false;
    queryView->reparent(this,0,queryView->pos(),true);
    matchView->reparent(this,0,matchView->pos(),true);
    matchView->hide();
    delete splitter;
    setCentralWidget(queryView);
  }
  else                       // list is not visible -> show it
  {
    global->showMatchList = true;
    splitter = new QSplitter(Qt::Horizontal,this);
    splitter->setOpaqueResize( KGlobalSettings::opaqueResize() );
    setCentralWidget(splitter);
    splitter->show();
    queryView->reparent(splitter,0,queryView->pos(),true);
    matchView->reparent(splitter,0,matchView->pos(),true);
    splitter->setResizeMode(matchView,QSplitter::KeepSize);
    adjustMatchViewSize();
  }

  actShowMatchList->setChecked(global->showMatchList);
}


void TopLevel::saveMatchViewSize()
{
  if (global->showMatchList)
  {
      global->splitterSizes = splitter->sizes();
  }
}


void TopLevel::adjustMatchViewSize()
{
  if (global->splitterSizes.count()==2)
  {
    splitter->setSizes(global->splitterSizes);
  }
}


void TopLevel::slotConfToolbar()
{
  saveMainWindowSettings(KGlobal::config().data(),"toplevel_options");
  KEditToolbar dlg(actionCollection(), "kdictui.rc");
  connect(&dlg,SIGNAL( newToolbarConfig() ), this, SLOT( slotNewToolbarConfig() ));
  dlg.exec();
}


void TopLevel::slotNewToolbarConfig()
{
  recreateGUI();
  applyMainWindowSettings(KGlobal::config().data(),"toplevel_options");
  buildHistMenu();   // actionlists must be inserted
  stratDbChanged();
}


void TopLevel::showSetsDialog()
{
  if (!setsDlg) {
    setsDlg = new DbSetsDialog(this);
    connect(setsDlg,SIGNAL(setsChanged()),this,SLOT(setsChanged()));
    connect(setsDlg,SIGNAL(dialogClosed()),this,SLOT(hideSetsDialog()));
    setsDlg->show();
  } else {
#ifdef Q_WS_X11
    KWin::activateWindow(setsDlg->winId());
#endif
  }
}


void TopLevel::hideSetsDialog()
{
  if (setsDlg) {
    setsDlg->delayedDestruct();
    setsDlg = 0L;
  }
}


void TopLevel::setsChanged()
{
  actDbCombo->setList(global->databases);
  actDbCombo->setCurrentItem(global->currentDatabase);
}


void TopLevel::showOptionsDialog()
{
  if (!optDlg) {
    optDlg = new OptionsDialog(this);
    connect(optDlg,SIGNAL(optionsChanged()),this,SLOT(optionsChanged()));
    connect(optDlg,SIGNAL(finished()),this,SLOT(hideOptionsDialog()));
    optDlg->show();
  }
#ifdef Q_WS_X11 
  else {
    KWin::activateWindow(optDlg->winId());
  }
#endif  
}


void TopLevel::hideOptionsDialog()
{
  if (optDlg) {
    optDlg->delayedDestruct();
    optDlg=0;
  }
}


void TopLevel::optionsChanged()
{
  QString serverInfo;
  if (global->authEnabled)
    serverInfo = QString(" %1@%2:%3 ").arg(getShortString(global->user,50))
                                    .arg(getShortString(global->server,50))
                                    .arg(global->port);
  else
    serverInfo = QString(" %1:%3 ").arg(getShortString(global->server,50))
                                    .arg(global->port);
  statusBar()->changeItem(serverInfo,1);
  interface->serverChanged();                  // inform client
  queryView->optionsChanged();              // inform html-view
}

void TopLevel::raiseWindow()
{
  // Bypass focus stealing prevention
  kapp->updateUserTimestamp();
#ifdef Q_WS_X11
  KWin::WindowInfo info = KWin::windowInfo(winId(), 0);

  if (  !info.isOnCurrentDesktop() )
  {
    KWin::setOnDesktop( winId(), KWin::currentDesktop() );
  }

  KWin::activateWindow(winId());
#endif
}


//--------------------------------

#include "toplevel.moc"
