/*
 * Copyright (C) 2004, Mart Kelder (mart.kde@hccnet.nl)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "kio_subjects.h"

#include "kio.h"
#include "kio_single_subject.h"
#include "kio_proto.h"
#include "mailsubject.h"

#include <kio/global.h>
#include <kio/scheduler.h>
#include <kdebug.h>

#include <QList>
#include <QString>

KIO_Subjects::KIO_Subjects( QObject * parent )
	: QObject( parent ),
	_protocol( 0 ),
	_slave( 0 ),
	_valid( true )
{
	_jobs = new QList<KIO_Single_Subject*>;
	_kurl = new KUrl;
	_metadata = new KIO::MetaData;
}

KIO_Subjects::~KIO_Subjects( )
{
	while( !_jobs->isEmpty() )
		delete _jobs->takeFirst();
	delete _jobs;
	delete _kurl;
	delete _metadata;
	_protocol = 0;
}

void KIO_Subjects::doReadSubjects( KKioDrop *drop )
{
	QList<KKioDrop::FileInfo>::ConstIterator it;
	QList<KKioDrop::FileInfo>::ConstIterator end_it = drop->_mailurls->end();

	_kio = drop;
	_protocol = _kio->_protocol;
	*_kurl = *_kio->_kurl;
	*_metadata = *_kio->_metadata;

	if( _jobs->count() > 0 )
		kWarning() << i18n( "Already a slave pending." ) << endl;

	while( !_jobs->isEmpty() )
		delete _jobs->takeFirst();

	//Open connection
	getConnection( );

	//Open jobs for easy item in the list
	for( it = _kio->_mailurls->begin(); it != end_it; it++ )
		startJob( (*it).name, (*it).size );

	//close connection for trivial situations (empty list)
	disConnect( true );

	//passing number of subjects for progress bar.
	_kio->emitReadSubjectsTotalSteps( _jobs->count() );
}

void KIO_Subjects::getConnection( )
{
	KUrl kurl = *_kurl;
	KIO::MetaData metadata = *_metadata;

	if( _slave )
	{
		KIO::Scheduler::disconnectSlave( _slave );
		_slave = 0;
	}

	if( _protocol->connectionBased( ) )
	{
		_protocol->readSubjectConnectKUrl( kurl, metadata );

		if( kurl.port() == 0 )
			kurl.setPort( _protocol->defaultPort( _kio->_ssl ) );

		if( ! ( _slave = KIO::Scheduler::getConnectedSlave( kurl, metadata ) ) )
		{
			kWarning() << i18n( "Not able to open a kio-slave for %1.", _protocol->configName() );
			_kio->emitShowPassivePopup( i18n( "Not able to open a kio-slave for %1.", _protocol->configName() ) );
			_valid = false;
			_kio->emitValidChanged();
			_slave = 0;
			_kio->emitReadSubjectsReady( false );
			return;
		}
	}
}

void KIO_Subjects::startJob( const QString &name, const long size )
{
	KUrl kurl = *_kurl;
	KIO::MetaData metadata = *_metadata;
	KIO_Single_Subject *subject;

	kurl = name;

	_protocol->readSubjectKUrl( kurl, metadata );

	if( kurl.port() == 0 )
		kurl.setPort( _protocol->defaultPort( _kio->_ssl ) );

	subject = new KIO_Single_Subject( this, kurl, metadata, _protocol, _slave, name, size );

	connect( subject, SIGNAL( readSubject( KornMailSubject* ) ), this, SLOT( slotReadSubject( KornMailSubject* ) ) );
	connect( subject, SIGNAL( finished( KIO_Single_Subject* ) ), this, SLOT( slotFinished( KIO_Single_Subject* ) ) );

	_jobs->append( subject );
}

void KIO_Subjects::disConnect( bool result )
{
	if( _jobs->isEmpty() )
	{
		if( _slave )
		{
			KIO::Scheduler::disconnectSlave( _slave );
			_slave = 0;
		}
		_kio->emitReadSubjectsReady( result );
	}
}

void KIO_Subjects::canceled( )
{
	while( !_jobs->isEmpty() )
		delete _jobs->takeFirst();

	//_slave died in cancelJob with is by called from the destructor of KIO_Single_Subject,
	//withs is by called by _jobs->clear because autoRemove equals true.
	_slave = 0;
	disConnect( false );
}

void KIO_Subjects::slotReadSubject( KornMailSubject* subject )
{
	_valid = true;
	_kio->emitValidChanged();
	subject->setMailDrop( _kio );
	_kio->emitReadSubjectsRead( subject );
}

void KIO_Subjects::slotFinished( KIO_Single_Subject* item )
{
	item->deleteLater();
	_jobs->removeAll( item );

	_kio->emitReadSubjectsProgress( _jobs->count( ) );

	disConnect( true ); //Only works when all jobs are finished.
}

#include "kio_subjects.moc"
