/*
    This file is part of kdepim.

    Copyright (c) 2004 Reinhold Kainhofer <reinhold@kainhofer.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "API_Blogger.h"
#include "xmlrpcjob.h"
#include <kdebug.h>
//Added by qt3to4:
#include <QList>

using namespace KBlog;

QString APIBlogger::getFunctionName( blogFunctions type )
{
  switch ( type ) {
    case bloggerGetUserInfo:    return "blogger.getUserInfo";
    case bloggerGetUsersBlogs:  return "blogger.getUsersBlogs";
    case bloggerGetRecentPosts: return "blogger.getRecentPosts";
    case bloggerNewPost:        return "blogger.newPost";
    case bloggerEditPost:       return "blogger.editPost";
    case bloggerDeletePost:     return "blogger.deletePost";
    case bloggerGetPost:        return "blogger.getPost";
    case bloggerGetTemplate:    return "blogger.getTemplate";
    case bloggerSetTemplate:    return "blogger.setTemplate";
    default: return QString();
  }
}




KIO::Job *APIBlogger::createUserInfoJob()
{
  kDebug() << "read user info..." << endl;
  QList<QVariant> args( defaultArgs() );
  return KIO::xmlrpcCall( mServerURL, getFunctionName( bloggerGetUserInfo ), args, false );
}

KIO::Job *APIBlogger::createListFoldersJob()
{
  // TODO: Check if we're already authenticated. If not, do it!
//   if ( isValid() ) {
    kDebug() << "Fetch List of Blogs..." << endl;
    QList<QVariant> args( defaultArgs() );
    return KIO::xmlrpcCall( mServerURL, getFunctionName( bloggerGetUsersBlogs ), args, false );
//   } else {
//     warningNotInitialized();
//     return 0;
//   }
}

KIO::TransferJob *APIBlogger::createListItemsJob( const KUrl &url )
{
  // TODO: Check if we're already authenticated. If not, do it!
//   if ( isValid() ) {
    kDebug() << "Fetch List of Posts..." << endl;
    QList<QVariant> args( defaultArgs( url.url() ) );
    args << QVariant( mDownloadCount );
    return KIO::xmlrpcCall( mServerURL, getFunctionName( bloggerGetRecentPosts ), args, false );
//   } else {
//     warningNotInitialized();
//     return 0;
//   }
}

KIO::TransferJob *APIBlogger::createDownloadJob( const KUrl &url )
{
//   if ( isValid() ){
    kDebug() << "Fetch Posting with url " << url.url() << endl;
    QList<QVariant> args( defaultArgs( url.url() ) );
    return KIO::xmlrpcCall( mServerURL, getFunctionName( bloggerGetPost ), args, false );
//   } else {
//     warningNotInitialized();
//     return 0;
//   }
}

KIO::TransferJob *APIBlogger::createUploadJob( const KUrl &url, KBlog::BlogPosting *posting )
{
  if ( !posting ) {
    kDebug() << "APIBlogger::createUploadJob: posting=0" << endl;
    return 0;
  }
//   if ( isValid() ){
    kDebug() << "Uploading Posting with url " << url.url() << endl;
    QList<QVariant> args( defaultArgs( posting->postID() ) );
    args << QVariant( posting->content() );
    args << QVariant( /*publish=*/true, 0 );
    return KIO::xmlrpcCall( mServerURL, getFunctionName( bloggerEditPost ), args, false );
//   } else {
//     warningNotInitialized();
//     return 0;
//   }
}

KIO::TransferJob *APIBlogger::createUploadNewJob( KBlog::BlogPosting *posting )
{
  if ( !posting ) {
    kDebug() << "APIBlogger::createUploadNewJob: posting=0" << endl;
    return 0;
  }
//   if ( isValid() ){
    kDebug() << "Creating new Posting with blogid " << posting->blogID() << " at url " << mServerURL << endl;
    QList<QVariant> args( defaultArgs( posting->blogID() ) );
    args << QVariant( posting->content() );
    args << QVariant( /*publish=*/true, 0 );
    return KIO::xmlrpcCall( mServerURL, getFunctionName( bloggerNewPost ), args, false );
//   } else {
//     warningNotInitialized();
//     return 0;
//   }
}

KIO::Job *APIBlogger::createRemoveJob( const KUrl &/*url*/, const QString &postid )
{
kDebug() << "APIBlogger::createRemoveJob: postid=" << postid << endl;
//   if ( isValid() ){
    QList<QVariant> args( defaultArgs( postid ) );
    args << QVariant( /*publish=*/true, 0 );
    return KIO::xmlrpcCall( mServerURL, getFunctionName( bloggerDeletePost ), args, false );
//   } else {
//     warningNotInitialized();
//     return 0;
//   }
}




bool APIBlogger::interpretUserInfoJob( KIO::Job *job )
{
  // TODO: Implement user authentication
//   isValid = true;
  KIO::XmlrpcJob *trfjob = dynamic_cast<KIO::XmlrpcJob*>(job);
  if ( job->error() || !trfjob ) {
    // TODO: Error handling
    return false;
  } else if ( trfjob ) {
    QList<QVariant> message( trfjob->response() );

    kDebug () << "TOP: " << message[ 0 ].typeName() << endl;
    const QList<QVariant> posts = message;
    QList<QVariant>::ConstIterator it = posts.begin();
    QList<QVariant>::ConstIterator end = posts.end();
    for ( ; it != end; ++it ) {
      kDebug () << "MIDDLE: " << ( *it ).typeName() << endl;
      const QMap<QString, QVariant> postInfo = ( *it ).toMap();
      const QString nickname = postInfo[ "nickname" ].toString();
      const QString userid = postInfo[ "userid" ].toString();
      const QString email = postInfo[ "email" ].toString();
      kDebug() << "Post " << nickname << " " << userid << " " << email << endl;
      // FIXME: How about a BlogUserInfo class???
      emit userInfoRetrieved( nickname, userid, email );
    }
    return true;
  }
  return false;
}

void APIBlogger::interpretListFoldersJob( KIO::Job *job )
{
kDebug() << "APIBlogger::interpretListFoldersJob" << endl;
  KIO::XmlrpcJob *trfjob = dynamic_cast<KIO::XmlrpcJob*>(job);
  if ( job->error() || !trfjob ) {
    // TODO: Error handling
  } else {
kDebug() << "APIBlogger::interpretListFoldersJob, no error!" << endl;
    QList<QVariant> message( trfjob->response() );
    kDebug () << "TOP: " << message[ 0 ].typeName() << endl;

    const QList<QVariant> posts = message[ 0 ].toList();
    QList<QVariant>::ConstIterator it = posts.begin();
    QList<QVariant>::ConstIterator end = posts.end();
    for ( ; it != end; ++it ) {
      kDebug () << "MIDDLE: " << ( *it ).typeName() << endl;
      const QMap<QString, QVariant> postInfo = ( *it ).toMap();

      const QString id( postInfo[ "blogid" ].toString() );
      const QString name( postInfo[ "blogName" ].toString() );
      const QString url( postInfo[ "url" ].toString() );

      // Use the Blog ID instead of the URL. The ID already indicates the correct blog, and the
      // URL for all calls will be the XML-RPC interface, anyway.
      if ( !id.isEmpty() && !name.isEmpty() ) {
        emit folderInfoRetrieved( id, name );
kDebug()<< "Emitting folderInfoRetrieved( id=" << id << ", name=" << name << "); " << endl;
      }
    }
  }
}

bool APIBlogger::interpretListItemsJob( KIO::Job *job )
{
  return interpretDownloadItemsJob( job );
}

bool APIBlogger::interpretDownloadItemsJob( KIO::Job *job )
{
  kDebug(5800)<<"APIBlogger::interpretDownloadItemJob"<<endl;
  KIO::XmlrpcJob *trfjob = dynamic_cast<KIO::XmlrpcJob*>(job);
  bool success = false;
  if ( job->error() || !trfjob ) {
    // TODO: Error handling
    success = false;
  } else {
    //array of structs containing ISO.8601 dateCreated, String userid, String postid, String content;
    // TODO: Time zone for the dateCreated!
    QList<QVariant> message( trfjob->response() );
    kDebug () << "TOP: " << message[ 0 ].typeName() << endl;

    const QList<QVariant> postReceived = message[ 0 ].toList();
    QList<QVariant>::ConstIterator it = postReceived.begin();
    QList<QVariant>::ConstIterator end = postReceived.end();
    success = true;
    for ( ; it != end; ++it ) {
      BlogPosting posting;
      kDebug () << "MIDDLE: " << ( *it ).typeName() << endl;
      const QMap<QString, QVariant> postInfo = ( *it ).toMap();
      if ( readPostingFromMap( &posting, postInfo ) ) {
        KCal::Journal *j = journalFromPosting( &posting );
//         dumpBlog( &posting );
        kDebug() << "Emitting itemOnServer( posting.postID()="<<posting.postID() << "); " << endl;
        emit itemOnServer( KUrl( posting.postID() ) );
        kDebug() << "Emitting itemDownloaded( j=" << j << ", uid=" << j->uid()
                  << ", postID=" << posting.postID() << ", fpr="
                  << posting.fingerprint() << "); " << endl;
        emit itemDownloaded( j, j->uid(), KUrl( posting.postID() ),
                             posting.fingerprint(), posting.postID() );
      } else {
        kDebug() << "readPostingFromMap failed! " << endl;
        success = false;
        // TODO: Error handling
      }
    }
  }
  return success;
}


bool APIBlogger::readPostingFromMap( BlogPosting *post, const QMap<QString, QVariant> &postInfo )
{
  // FIXME:
  if ( !post ) return false;
  QStringList mapkeys = postInfo.keys();
  kDebug() << endl << "Keys: " << mapkeys.join(", ") << endl << endl;

  QString fp = QString() ;
  
  QDateTime dt( postInfo[ "dateCreated" ].toDateTime() );
  if ( dt.isValid() && !dt.isNull() ) {
    post->setCreationDateTime( dt );
    QString fp = dt.toString( Qt::ISODate );
  }
  dt = postInfo[ "postDate" ].toDateTime();
  if ( dt.isValid() && !dt.isNull() ) {
    post->setDateTime( dt );
    fp = dt.toString( Qt::ISODate );
  }
  dt = postInfo[ "lastModified" ].toDateTime();
  if ( dt.isValid() && !dt.isNull() ) {
    post->setModificationDateTime( dt );
    fp = dt.toString( Qt::ISODate );
  }
  post->setFingerprint( fp );

  post->setUserID( postInfo[ "userid" ].toString() );
  post->setPostID( postInfo[ "postid" ].toString() );

  QString title( postInfo[ "title" ].toString() );
  QString description( postInfo[ "description" ].toString() );
  QString contents( postInfo[ "content" ].toString() );
  QString category;

    // TODO: Extract title and cats from the old-style blogger api without extensions
/*
  if ( (title.isEmpty() || description.isEmpty() ) && !contents.isEmpty()  ) {
    // we don't have both title and description, so use the content (ie. it's an
    // old-style blogger api, not the extended drupal api.

    kDebug() << "No title and description given, so it's an old-style "
                 "Blogger API without extensions" << endl;
    QString catTagOpen = mTemplate.categoryTagOpen();
    QString catTagClose = mTemplate.categoryTagClose();
    QString titleTagOpen = mTemplate.titleTagOpen();
    QString titleTagClose = mTemplate.titleTagClose();

    int catStart = contents.find( catTagOpen, 0, false ) + catTagOpen.length();
    int catEnd = contents.find( catTagClose, 0, false );
kDebug() << "  catTagOpen = " << catTagOpen << ", catTagClose = " << catTagClose << ", start - end : " << catStart <<" - " << catEnd << endl;
    if ( catEnd > catStart ) {
      category = contents.mid( catStart, catEnd - catStart );
      kDebug() << "Found a category \"" << category << "\"" << endl;
      contents = contents.remove( catStart - catTagOpen.length(),
              catEnd - catStart + catTagClose.length() + catTagOpen.length() );
    }
    int titleStart = contents.find( titleTagOpen, 0, false ) + titleTagOpen.length();
    int titleEnd = contents.find( titleTagClose, 0, false );
kDebug() << "  titleTagOpen = " << titleTagOpen << ", titleTagClose = " << titleTagClose << ", start - end : " << titleStart <<" - " << titleEnd << endl;
    kDebug() << "Title start and end: " << titleStart << ", " << titleEnd << endl;
    if ( titleEnd > titleStart ) {
      title = contents.mid( titleStart, titleEnd - titleStart );
      contents = contents.remove( titleStart - titleTagOpen.length(),
              titleEnd - titleStart + titleTagClose.length() + titleTagOpen.length() );
    }
    kDebug() << endl << endl << endl << "After treatment of the special tags, we have a content of: "<< endl << contents << endl;
  }
*/

  post->setTitle( title );
  post->setContent( contents );
  if ( !category.isEmpty() )
    post->setCategory( category );
  return true;
}



