#! /usr/bin/env perl

# Copyright (C) 2006 by Adriaan de Groot
#
# This file is released under the terms of the GNU General Public License
# (GPL) version 2.

use strict;
use Getopt::Long;
use File::Basename;
use File::Path;
use Cwd 'abs_path';
use Template;

my $Prog = 'nudox';
my
$VERSION = '0.9';

my $help = '';
my $version = '';
my $qtdocdir = '';
my $qtdoctag = '';
my $style = 'capacity';
my $doxdir = abs_path(dirname($0));
my $force_tags = '';

exit 1 if ( not GetOptions(
  'force-tags' => \$force_tags,
  'style=s' => \$style,
  'doxdatadir=s' => \$doxdir,
  'qtdocdir=s' => \$qtdocdir,
  'qtdoctag=s' => \$qtdoctag,
  'help' => \$help,
  'version' => \$version
   ) );

&Version() if $version;
&Help() if $help;
exit 0 if $version or $help;

my $srcdir = $ARGV[0];
my $module = basename( $srcdir );

-d $srcdir or die "! Source '$srcdir' is not a directory.";
$doxdir and $module and $srcdir or die "! Insufficient arguments";

# This maps DOXYGEN_* (not DOXYGEN_SET_*) assignments
# to valid Doxyfile settings.
my %doxysettingsmap = (
  "NAME" => "PROJECT_NAME",
  "VERSION" => "PROJECT_NUMBER",
  "ENABLE" => "# ENABLED"
  ) ;
# These are the settings extracted from the top-level doxyfile
# and applied to all the lower-level ones.
my @doxytoplevelsettings = ( "PROJECT_NAME", "PROJECT_NUMBER" );


&createdoxdir();
-d $module or die "! Target '$module' is not a directory.";
unlink( "$module/$module.tag" ) if $force_tags;

&lookforqttags();
&generatedoxyfile();
&completedoxyfile();
my @subdirs = &findsubdirs();
&createsubdirsfile();
&createhtmlmenu();
&createlocaldoxyfiles();
&rundoxygen();

###
#
# Create the directory that we will write all the APIDOX into.
#
sub createdoxdir() {
  mkpath("$module");
  die "! Target directory '$module' can not be created." unless -d $module;
}

###
#
# Search for QT tagfile
#
# As documented:
#  Return 0 = TAG exists and is a file and QTDOCDIR is set
#  Return 1 = TAG doesn't exist
#
# Funtion looktorqttags() does the work, the other bits are
# helpers for various messages.

sub tagsnotfound() {
  print "! Qt documentation can not be found.\n";
  die "! Use --qtdocdir and --qtdoctag.\n";
}

sub tagsexist() {
  print "# Using Qt tag file '$qtdoctag'\n";
  system("cp \"$qtdoctag\" \"$module\"/qt.tag");
  writetags();
}

sub tagsneeded() {
  print "# Generating Qt tag file from '$qtdocdir'\n";
  system("doxytag -t \"$module\"/qt.tag \"$qtdocdir\"");
  writetags();
}

sub writetags() {
  open TAGFILE, ">$module/tagfiles" or die "! Can not write tagfiles\n";
  print TAGFILE "\"qt.tag=$qtdocdir\"";
  close TAGFILE;
}

# Look for the QT tag file and exit(1) if not found.
sub lookforqttags() {
  if ( -f $qtdoctag ) {
    print "# Checking Qt tags file '$qtdoctag'\n";
    if (!open TAGF, "<$qtdoctag") {
      print "! Can not read given file '$qtdoctag'\n";
      tagsnotfound();
    }

    my $firstline = <TAGF>;
    if (! $firstline =~ /tagfile/ ) {
      print "! File '$qtdoctag' does not seem to be a tag file\n";
      tagsnotfound();
    }

    if ( $qtdocdir =~ /^$/ ) {
      print "! No documentation directory given\n";
      tagsnotfound();
    }

    if ( not -d $qtdocdir ) {
      print "# Qt documentation directory is not a directory,\n";
      print "# but I'll use it anyway because of the tag file.\n";
    }

    tagsexist();
    return 0;
  }

  if ( -d $qtdocdir and -f "$qtdocdir/index.html" and -f "$qtdocdir/qstring.html" ) {
    print "# Qt documentation will be generated from '$qtdocdir'\n";
    tagsneeded();
    return 0;
  }

  print "! No documentation directory given\n";
  tagsnotfound();
}


###
#
# Create an initial doxygen file by calling on it to generate
# the file (while suppressing output). Pack some custom settings
# onto the end immediately -- we do not want to generate LaTeX,
# for instance.
#
sub generatedoxyfile() {
  print "# Generating initial Doxygen file ...\n";
  system("doxygen -g \"$module/Doxyfile.in\" > /dev/null 2>&1");
  open( DOXYFILE, ">>$module/Doxyfile.in") or die "! Can not open Doxyfile.in\n";

  print DOXYFILE "
RECURSIVE = YES
GENERATE_LATEX = NO
GENERATE_HTML = NO # Turned on later
QUIET = YES
";
  close( DOXYFILE );
}


###
#
# Modify the doxyfile to generate the HTML files; this does not
# actually do any of the generation nor fill in source directories.
#
sub completedoxyfile() {
  open( DOXYFILE, ">>$module/Doxyfile.in" ) or die "! Can not open Doxyfile.in\n";
  print DOXYFILE "
GENERATE_HTML = YES
HTML_OUTPUT = \".\"

EXTRACT_ALL = NO

DISABLE_INDEX = YES

ALPHABETICAL_INDEX = YES
IGNORE_PREFIX = K Q

WARNINGS = YES
WARN_IF_UNDOCUMENTED = YES
WARN_IF_DOC_ERROR = YES
WARN_NO_PARAMDOC = YES
";
}


###
#
# Find those subdirs that should generate APIDOX. These are
# (in KDE4) those directories that have a Mainpage.dox.
#
sub findsubdirs() {
  open OUTDIRS, ">$module/subdirs.in" or
    die "! Can not write subdirs.in list.\n";
  open SUBDIRS, "find $srcdir -name Mainpage.dox |" or
    die "! Can not start reading subdirectories for Mainpage.dox\n";

  my @subdirlist = ();
  while (<SUBDIRS>) {
    chomp;
    s+^$srcdir++;
    s+/Mainpage.dox$++;
    next if /^$/;
    next if /^Mainpage.dox/;
    # Skip KDE3 directories
    next if -f "$srcdir/$_/Makefile.am";
    push @subdirlist, $_;
  }

  close SUBDIRS;

  print OUTDIRS sort(@subdirlist);
  close OUTDIRS;

  return sort(@subdirlist);
}


###
#
# We need the list of subdirs in a file for later use by the shellscript.
#
sub createsubdirsfile() {
  open( SUBDIRS, ">$module/subdirs" ) or
    die "! Can not create subdir list\n";
  print SUBDIRS join("\n",@subdirs);
  close( SUBDIRS );
}


###
#
# Scan a Mainpage.dox for DOXYGEN_* assignments.
#
sub extract_doxysettings {
  my $filename = $_[0];
  my %v = ();
  open( MAINPAGE, "<$filename" ) or die "Can not open '$filename'.";
  while ( <MAINPAGE> ) {
    next unless /^\/\/ DOXYGEN_/;
    chomp;
    s/^.. DOXYGEN_//;
    my $name = $_;
    my $value = $_;
    $name =~ s/[\s=].*//;
    $value =~ s/^[^\s=]*\s*=\s*//;

    # We have the convention that DOXYGEN_SET_* writes raw
    # Doxygen settings to the file, while DOXYGEN_* for other
    # values are "special" and get mapped around.
    if ($name =~ /SET_/) {
      $name =~ s/SET_//;
    } else {
      # This ought to be a special one.
      next unless exists $doxysettingsmap{$name};
      $name = $doxysettingsmap{$name};
    }

    # Start accumulating possible multi-line value
    my $tvalue = $value;
    while ($value =~ /\\$/) {
      $value = <MAINPAGE>;
      chomp $value;
      die unless $value =~ /^\/\//;
      $value =~ s/^\/\/\s*//;
      $tvalue .= "\n\t$value";
    }

    $v{$name} = $tvalue;
  }
  close( MAINPAGE );
  return %v;
}

###
#
# Write the HTML menu code for all the subdirs.
#
sub createhtmlmenu() {
  open( SUBDIRHTML, ">$module/subdirs.html" ) or die "! Can not create subdir HTML\n";
  my @l = ();
  my $depth = 0;
  for my $dir (@subdirs) {
    my @dirs = split( "/", $dir );
    my $i = 0;
    while ($l[$i] eq $dirs[$i]) {
      $i++;
    }
    if ($i>$depth) {
      my $j=$depth;
      while ($j<$i) {
        print SUBDIRHTML "<ul style=\"padding-left:1em;\" >";
        $j++;
      }
    }
    if ($i<$depth) {
      my $j=$i;
      while ($j<$depth) {
        print SUBDIRHTML "</ul>";
        $j++;
      }
    }
    print SUBDIRHTML "<li><a href=\"[%top%]/";
    print SUBDIRHTML $dir;
    print SUBDIRHTML "/index.html\">";
    print SUBDIRHTML $dirs[$i];
    print SUBDIRHTML "</a></li>\n";

    @l = @dirs;
    $depth = $i;
  }
  if (0<$depth) {
    while (0<$depth) {
      print SUBDIRHTML "</ul>";
      $depth--;
    }
  }
  close( SUBDIRHTML );
}



###
#
# Create the header and footer files for each subdirectory,
# also create Doxyfile.local for each subdirectory to read
# those header and footers.
#
sub createlocaldoxyfiles() {
  my $tt = Template->new({
    OUTPUT_PATH => $module,
    RELATIVE => 1,
    ABSOLUTE => 1
    }) || die "! ", Template->error(), "\n";


  my %doxy_toplevel = extract_doxysettings("$srcdir/Mainpage.dox");


  my $hdr = "header.html";
  my $ftr = "footer.html";

  for my $dir (@subdirs,".") {
    # Skip directories that are still KDE3 structure (used in playground)
    next if -f "$srcdir/$dir/Makefile.am";

    my $subdir_name = basename($dir);

    print SUBDIRS "$dir\n";
    mkpath("$module/$dir");

    open( OUTF , ">$module/$dir/Doxyfile.local" ) or die "Can not write Doxygen settings in '$dir'.";

    print OUTF "INPUT = \"$srcdir/$dir/\"\n";
    print OUTF "OUTPUT_DIRECTORY = \"$module/$dir/\"\n";
    print OUTF "RECURSIVE = NO\n" if $dir eq ".";
    print OUTF "GENERATE_TAGFILE = \"$module/$dir/$subdir_name.tag\"";

    my %doxy_local = extract_doxysettings("$srcdir/$dir/Mainpage.dox");
    print OUTF "# Local settings\n";
    foreach my $k (keys(%doxy_local)) {
      print OUTF $k . " = " . $doxy_local{$k} . "\n";
    }
    print OUTF "# Toplevel settings\n";
    foreach my $k (@doxytoplevelsettings) {
      if (exists($doxy_toplevel{$k}) and not exists($doxy_local{$k})) {
        print OUTF $k . " = " . $doxy_toplevel{$k} . "\n";
      }
    }

    print OUTF "# Calculated settings\n";
    my @subsubs = ();
    for my $d (@subdirs) {
      next unless $d =~ /^$dir\//;
      push @subsubs, "$srcdir/$d/";
    }

    print OUTF "EXCLUDE = ", join( " \\\n", @subsubs ), "\n" if @subsubs;

    my $breadcrumb_path = "";
    my @l = ();
    my @up = ();
    for my $i (split("/","$module/$dir")) {
      next unless $i;
      push @l,$i;
      push @up,"..";
    }
    pop @up;
    my $toplevel = join( "/", @up );

    for my $i (@l) {
      my $item=$i;
      $item="<a href=\"" . join("/",@up) . "\">" . $i . "</a>" if @up;
      $breadcrumb_path .= "<li>$item</li>\n";
      pop @up;
    }
    $breadcrumb_path="" if $dir eq ".";
    $toplevel="." if $dir eq ".";

    my $date = gmtime;
    my $module_name = $doxy_toplevel{'PROJECT_NAME'} or $module;

    my $vars = {
      date => $date,
      top => $toplevel,
      site => "http://www.kde.org",
      module => $module,
      module_name => $module_name,
      subdir => $subdir_name,
      breadcrumbs => $breadcrumb_path
    };

    print "# Generating template for '$dir'\n";
    my $flavour = "";
    $flavour = "toplevel-" if $dir eq ".";

    if ($tt->process( "$doxdir/$style-$flavour$hdr", $vars, "$dir/$hdr" )) {
      print OUTF "HTML_HEADER=\"$module/$dir/$hdr\"\n";
    } else {
      die "! ", $tt->error(), "\n";
    }

    if ($tt->process( "$doxdir/$style-$flavour$ftr", $vars, "$dir/$ftr" )) {
      print OUTF "HTML_FOOTER=\"$module/$dir/$ftr\"\n"
    } else {
      die "! ", $tt->error(), "\n";
    }

    close( OUTF );
  }
}


###
#
# Actually run Doxygen on all of the directories we have.
#
sub rundoxygen() {
  # Create empty file
  open( OUTF, ">$module/Doxygen.log" ) or die "! Can not open doxygen logfile.\n";
  close( OUTF );

  # Generate all of them
  for my $dir (@subdirs,".") {
    print "# Generating APIDOX in $dir\n";
    system( "cat \"$module/Doxyfile.in\" \"$module/$dir/Doxyfile.local\" > \"$module/$dir/Doxyfile\"" );
    system( "doxygen \"$module/$dir/Doxyfile\" >> \"$module/Doxygen.log\" 2>&1" );
  }
}

sub Help() {
  &Version();

  print "# nudox [options] src-dir
#
# --without-tags     Do not generate tags file for module.
#                    Useful if the tags are already there.
# --doxdatadir=dir   Use dir as the source of global Doxygen files.
# --installdir=dir   Use dir as target for installation.
# --qtdocdir=dir     Dir (or URL) where Qt documentation is.
# --qtdoctag=tagfile Qt documentation tag file.
# --style=name       Use HTML style <name> (default none).
#
# Instead of --qtdocdir and --qtdoctag you may use the environment
# variables QTDOCDIR and QTDOCTAG. If the tag file exists, you can
# use --qtdocdir to point to a URL; otherwise it must be the
# Qt HTML documentation directory.
#
";
}

sub Version() {
  print "$Prog, version $VERSION";
  exit 0 if $version;
}
