/*
 * Some auxiliar functions for cursor operations on a GtkTextView widget
 */
#include "support.h"

#include "tutor.h"
#include "cursor.h"

extern GtkWidget *window_tutor_;

struct
{
  gboolean blink;
  gboolean is_initializing;
} cursor;

/*******************************************************************************
 * Interface functions
 */
gboolean
cursor_get_blink ()
{
  return (cursor.blink);
}

void
cursor_set_blink (gboolean status)
{
  cursor.blink = status;
}

gboolean
cursor_get_is_initializing ()
{
  return (cursor.is_initializing);
}

void
cursor_set_is_initializing (gboolean status)
{
  cursor.is_initializing = status;
}

/*******************************************************************************
 * Advance the cursor n positions on the tutor text view
 */
gint
cursor_advance (gint n)
{
  gint i;
  gboolean cursor_out_screen;
  GtkWidget *wg;
  GtkTextBuffer *buf;
  GtkTextIter new_start;
  GtkTextIter old_start;
  GtkTextIter end;
  GtkAdjustment *scroll;
  GtkTextMark *mark;

  wg = lookup_widget (window_tutor_, "text_tutor");
  buf = gtk_text_view_get_buffer (GTK_TEXT_VIEW (wg));

  gtk_text_buffer_get_iter_at_mark (buf, &new_start,
				    gtk_text_buffer_get_insert (buf));
  old_start = new_start;

  if (n > 0)
    for (i = 0; i < n && gtk_text_iter_forward_cursor_position (&new_start);
	 i++);
  else
    for (i = 0; i > n && gtk_text_iter_backward_cursor_position (&new_start);
	 i--);

  end = new_start;
  gtk_text_iter_forward_char (&end);
  gtk_text_buffer_apply_tag_by_name (buf, "cursor_blink", &new_start, &end);

  end = old_start;
  gtk_text_iter_forward_char (&end);
  gtk_text_buffer_remove_tag_by_name (buf, "cursor_blink", &old_start, &end);

  gtk_text_buffer_place_cursor (buf, &new_start);

  if (i == n)
    {
      mark = gtk_text_buffer_create_mark (buf, "aux", &new_start, FALSE);
      cursor_out_screen =
	gtk_text_view_move_mark_onscreen (GTK_TEXT_VIEW (wg), mark);
      if (cursor_out_screen)
	{
	  gtk_text_view_scroll_mark_onscreen (GTK_TEXT_VIEW (wg),
					      gtk_text_buffer_get_insert
					      (buf));
	  wg = lookup_widget (window_tutor_, "scrolledwindow_tutor_main");
	  scroll =
	    gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (wg));
	  gtk_adjustment_set_value (scroll,
				    gtk_adjustment_get_value (scroll) +
				    3 * 23 * (n > 0 ? 1 : -1));
	}
    }
  return (i);
}


/*******************************************************************************
 * Paint the background of current char (at cursor) with color
 */
void
cursor_paint_char (gchar * color_tag_name)
{
  GtkWidget *wg;
  GtkTextBuffer *buf;
  GtkTextIter start;
  GtkTextIter end;

  wg = lookup_widget (window_tutor_, "text_tutor");
  buf = gtk_text_view_get_buffer (GTK_TEXT_VIEW (wg));
  gtk_text_buffer_get_iter_at_mark (buf, &start,
				    gtk_text_buffer_get_insert (buf));
  end = start;
  gtk_text_iter_forward_char (&end);

  gtk_text_buffer_apply_tag_by_name (buf, color_tag_name, &start, &end);
}


/*******************************************************************************
 * Get the character under the cursor
 */
gunichar
cursor_get_char ()
{
  gunichar chr;
  gchar *tmp_str;
  GtkWidget *wg;
  GtkTextBuffer *buf;
  GtkTextIter start;
  GtkTextIter end;

  wg = lookup_widget (window_tutor_, "text_tutor");
  buf = gtk_text_view_get_buffer (GTK_TEXT_VIEW (wg));
  gtk_text_buffer_get_iter_at_mark (buf, &start,
				    gtk_text_buffer_get_insert (buf));
  gtk_text_buffer_get_iter_at_mark (buf, &end,
				    gtk_text_buffer_get_insert (buf));
  gtk_text_iter_forward_char (&end);
  tmp_str = gtk_text_buffer_get_text (buf, &start, &end, FALSE);
  chr = g_utf8_get_char (tmp_str);
  g_free (tmp_str);
  return (chr);
}

/*******************************************************************************
 * Initialize the cursor so that it blinks
 */
gboolean
cursor_init (gpointer data)
{
  cursor.blink = TRUE;
  cursor.is_initializing = FALSE;
  cursor_off (NULL);
  return FALSE;
}

/**********************************************************************
 * Turns on the cursor and makes it blink if 'cursor_blink' is TRUE
 */
gboolean
cursor_on (gpointer data)
{
  if (cursor.blink == FALSE)
    return FALSE;

  if (window_tutor_ == NULL)
    {
      cursor.blink = FALSE;
      return FALSE;
    }

  g_timeout_add (TIME_CURSOR_OFF, &cursor_off, NULL);

  cursor_switch_on ();

  return FALSE;
}

/**********************************************************************
 * Turns off the cursor and makes it blink if 'cursor_blink' is TRUE
 */
gboolean
cursor_off (gpointer data)
{
  if (cursor.blink == FALSE)
    return FALSE;

  if (window_tutor_ == NULL)
    {
      cursor.blink = FALSE;
      return FALSE;
    }

  g_timeout_add (TIME_CURSOR_ON, &cursor_on, NULL);

  cursor_switch_off ();

  return FALSE;
}

/**********************************************************************
 * Turns on the cursor immediately
 */
void
cursor_switch_on ()
{
  GtkTextView *wg_text;
  GtkTextBuffer *buf;
  GtkTextIter start;
  GtkTextIter end;

  wg_text = GTK_TEXT_VIEW (lookup_widget (window_tutor_, "text_tutor"));
  buf = gtk_text_view_get_buffer (wg_text);

  gtk_text_buffer_get_iter_at_mark (buf, &start,
				    gtk_text_buffer_get_insert (buf));
  gtk_text_buffer_get_iter_at_mark (buf, &end,
				    gtk_text_buffer_get_insert (buf));
  gtk_text_iter_forward_char (&end);
  gtk_text_buffer_apply_tag_by_name (buf, "cursor_blink", &start, &end);
}

/**********************************************************************
 * Turns off the cursor immediately
 */
void
cursor_switch_off ()
{
  GtkTextView *wg_text;
  GtkTextBuffer *buf;
  GtkTextIter start;
  GtkTextIter end;

  wg_text = GTK_TEXT_VIEW (lookup_widget (window_tutor_, "text_tutor"));
  buf = gtk_text_view_get_buffer (wg_text);

  gtk_text_buffer_get_iter_at_mark (buf, &start,
				    gtk_text_buffer_get_insert (buf));
  gtk_text_buffer_get_iter_at_mark (buf, &end,
				    gtk_text_buffer_get_insert (buf));
  gtk_text_iter_forward_char (&end);
  gtk_text_buffer_remove_tag_by_name (buf, "cursor_blink", &start, &end);
}
