#include "CKlearAppTXTStreamer.h"

CKlearAppTXTStreamer::CKlearAppTXTStreamer( CKlearAppConfig *config, CKlearAppTXTDecoder *decoder )
{
  const QString __FUNC__ = "CKlearAppTXTStreamer::CKlearAppTXTStreamer( CKlearAppConfig *config )";

  try{
      this->pid = 0;
      this->fd = -1;
      this->KlearConfig = config;
      this->TXTDecoder = decoder;
      setRunning( true );
  }catch( CKlearAppErrorException &e ){  // all system exceptions
        e.addShowException( __LOC__, i18n( "Could not setup StreamReader\n" ) );
  }catch( std::exception &x ){  // all system exceptions
        CKlearAppErrorException e;
        e.addShowException( __LOC__, x.what() );
   }
}

CKlearAppTXTStreamer::~CKlearAppTXTStreamer()
{
  const QString __FUNC__ = "CKlearAppTXTStreamer::~CKlearAppTXTStreamer()";

  this->tearDownDemux();
  delete this->KlearConfig;
  delete this->TXTDecoder;
}

void CKlearAppTXTStreamer::tearDownDemux()
{
  const QString __FUNC__ = "CKlearAppTXTStreamer::tearDownDemux()";

  ioctl( this->fd, DMX_STOP, 0 );
  close( this->fd );
}

int CKlearAppTXTStreamer::getPid()
{
  const QString __FUNC__ = "CKlearAppTXTStreamer::getPID()";

  return this->pid;
}

void CKlearAppTXTStreamer::setPid( int pid )
{
  const QString __FUNC__ = "CKlearAppTXTStreamer::setPID( int PID )";

  this->pid = pid;
}

void CKlearAppTXTStreamer::setRunning( bool status )
{
   this->isRunning = status;
}

void CKlearAppTXTStreamer::run()
{
  const QString __FUNC__ = "CKlearAppTXTStreamer::run()";

  std::cout << "Reading in Content" << std::endl;

  try{
      this->readContent();
  }catch( CKlearAppErrorException &e ){  // all system exceptions
        e.addShowException( __LOC__, i18n( "Could not read TXT content\n" ) );
  }catch( std::exception &x ){  // all system exceptions
        CKlearAppErrorException e;
        e.addShowException( __LOC__, x.what() );
  }
}

void CKlearAppTXTStreamer::readContent()
{
  const QString __FUNC__ = "CKlearAppTXTStreamer::readContent()";

  try{
      setSectionDemux();
      this->readSectionBuffer();
      setSectionDemux();
      this->readSectionBuffer();
      setPesDemux();
      this->readPesBuffer();
  }catch( CKlearAppErrorException &e ){  // all system exceptions
        e.addErrorMessage( __LOC__, i18n( "Could not read sector loop\n" ) );
       throw;
  }catch( std::exception &x ){  // all system exceptions
        CKlearAppErrorException e( __LOC__, x.what() );
        throw e;
  }
}

void CKlearAppTXTStreamer::setSectionDemux()
{
  const QString __FUNC__ = "CKlearAppTXTStreamer::setSectionDemux()";

  if( ( this->fd = open( "/dev/dvb/" + this->KlearConfig->getDVBAdapter() + "/" + this->KlearConfig->getDVBDemux(), O_RDWR ) ) < 0 )
  {
      std::cerr << "PAT/PMT TXT: Could not open demux device for reading! Continue without TXT" << std::endl;
      CKlearAppErrorException e( __LOC__,  i18n( "Could not open demux device for reading!\nWill continue without TXT." ) );
      throw e;
  }

  if( ioctl( this->fd, DMX_SET_BUFFER_SIZE, SECT_BUF_SIZE ) < 0 )
  {
    std::cerr << "DMX_SET_BUFFER_SIZE failed" << std::endl;
    this->tearDownDemux();
    CKlearAppErrorException e( __LOC__,  i18n( "DMX_SET_BUFFER_SIZE for Section failed" ) );
    throw e;
  }

  struct dmx_sct_filter_params flt;
  memset( &flt, 0, sizeof( struct dmx_sct_filter_params ) );
  flt.pid = getPid();
  flt.filter.filter[ 0 ] = 0;
  flt.filter.mask[ 0 ] = 0;
  flt.timeout = 0;
  flt.flags = DMX_IMMEDIATE_START; // |= DMX_CHECK_CRC;

  if ( ioctl( this->fd, DMX_SET_FILTER, &flt ) < 0 )
  {
    std::cerr << "DMX_SET_FILTER failed" << std::endl;
    this->tearDownDemux();
    CKlearAppErrorException e( __LOC__,  i18n( "DMX_SET_FILTER for Section failed" ) );
    throw e;
  }
}

void CKlearAppTXTStreamer::setPesDemux()
{
  const QString __FUNC__ = "CKlearAppTXTStreamer::setPesDemux()";

  if( ( this->fd = open( "/dev/dvb/" + this->KlearConfig->getDVBAdapter() + "/" + this->KlearConfig->getDVBDemux(), O_RDWR ) ) < 0 )
  {
    std::cerr << "PMT TXT: Could not open demux device for reading! Continue without TXT" << std::endl;
    CKlearAppErrorException e( __LOC__,  i18n( "Could not open demux device for reading!\nWill continue without TXT." ) );
    throw e;
  }

  if( ioctl( this->fd, DMX_SET_BUFFER_SIZE, PES_BUF_SIZE ) < 0 )
  {
    std::cerr << "DMX_SET_BUFFER_SIZE failed" << std::endl;
    this->tearDownDemux();
    CKlearAppErrorException e( __LOC__,  i18n( "DMX_SET_BUFFER_SIZE for PES failed" ) );
    throw e;
  }

  struct dmx_pes_filter_params flt;
  memset( &flt, 0, sizeof(struct dmx_pes_filter_params ) );
  flt.pid = getPid();
  flt.input  = DMX_IN_FRONTEND;
  flt.output = DMX_OUT_TAP;
  flt.pes_type = DMX_PES_OTHER;
  flt.flags = DMX_IMMEDIATE_START;

  if ( ioctl ( this->fd, DMX_SET_PES_FILTER, &flt ) < 0 ) {
    std::cerr << "DMX_SET_FILTER failed" << std::endl;
    this->tearDownDemux();
    CKlearAppErrorException e( __LOC__,  i18n( "DMX_SET_FILTER for PES failed" ) );
    throw e;
  }
}

long CKlearAppTXTStreamer::readSectionBuffer()
{
  const QString __FUNC__ = "CKlearAppTXTStreamer::readSectionBuffer()";

  int readBuffer;
  int sectionLength;

  // give parsed buffer data to TXTDecoder
  try{
    readBuffer = read( this->fd, this->buf, 3 );  // read section header
  }catch( std::exception &x ){  // all system exceptions
    CKlearAppErrorException e( __LOC__, x.what() );
    throw e;
  }

  if ( readBuffer <= 0 ) return readBuffer;  // error or strange, abort

  sectionLength = ( ( buf[1] & 0x0F ) << 8 ) + buf[ 2 ]; // get section size
  if ( sectionLength > ( sizeof( this->buf ) -3 ) ) return -1;   // something odd?

  try{
    readBuffer = read( this->fd, this->buf + 3, sectionLength ); // read 1 section
  }catch( std::exception &x ){  // all system exceptions
    CKlearAppErrorException e( __LOC__, x.what() );
    throw e;
  }

  if ( readBuffer >= 0 )
    readBuffer += 3;  // we already read header bytes

  // give parsed buffer data to TXTDecoder
  try{
    if ( getPid() == 0 )
      setPid( this->TXTDecoder->decodeSectionPAT( this->buf ) );
    else
      setPid( this->TXTDecoder->decodeSectionPMT( this->buf ) );
  }catch( CKlearAppErrorException &e ){  // all system exceptions
    e.addErrorMessage( __LOC__, i18n( "Could not decode buffer\n" ) );
    throw;
  }catch( std::exception &x ){  // all system exceptions
    CKlearAppErrorException e( __LOC__, x.what() );
    throw e;
  }

  return readBuffer;
}

void CKlearAppTXTStreamer::readPesBuffer()
{
  const QString __FUNC__ = "CKlearAppTXTStreamer::readPesBuffer()";

  long len;
  while( isRunning == true )  // to stop reading, set running to false
  {
    try{
      len = readSectorLoop(); // modifies this->buf with demux data
    }catch( CKlearAppErrorException &e ){  // all system exceptions
      e.addErrorMessage( __LOC__, i18n("Could not read sector loop\n") );
      throw;
    }catch( std::exception &x ){  // all system exceptions
      CKlearAppErrorException e( __LOC__, x.what() );
      throw e;
    }

    if ( len < 0 )
    { // -- error or eof?
      if ( errno == ETIMEDOUT )
        break;        // Timout, abort
      else
        continue;
    }

    if ( len == 0 )
      continue;  // dmxmode = no eof!

    // give parsed buffer data to TXTDecoder
    try{
        this->TXTDecoder->decodePESTXT( this->buf, len );
    }catch( CKlearAppErrorException &e ){  // all system exceptions
      e.addErrorMessage( __LOC__, i18n( "Could not decode buffer\n" ) );
      throw;
    }catch( std::exception &x ){  // all system exceptions
      CKlearAppErrorException e( __LOC__, x.what() );
      throw e;
    }
  }
}

long CKlearAppTXTStreamer::readSectorLoop()
{
  const QString __FUNC__ = "CKlearAppTXTStreamer::readSectorLoop()";

  long n1, n2, l;
  u_long skipped_bytes = 0;
  u_long sync = 0xFFFFFFFF;

  buf[0] = 0x00;
  buf[1] = 0x00;
  buf[2] = 0x01;

  while( true )
  {
    u_char c;

    try{
      n1 = read( this->fd, this->buf + 3, 1 );
    }catch( std::exception &x ){  // all system exceptions
      CKlearAppErrorException e( __LOC__, x.what() );
      throw e;
    }

    if( n1 <= 0 )
      return n1;

    c = buf[ 3 ];

    sync = ( sync << 8 ) | c;
    if( ( sync & 0xFFFFFF00 ) == 0x00000100 )
    {
      if( c == 0xB9 )  // MPEG_program_end
      {
        skipped_bytes -= 3;
        return 4;
      }

      if( c == 0xBA )  // pack_header_start
      {
        n1 = readPackHeader( this->fd, buf + 4 );
        skipped_bytes -= 3;
        return ( n1 > 0 ) ? n1 + 4 : n1;
      }

      if( c == 0xBB )  break;  // system_header_start  (same as PES packet)
      if( c >= 0xBC )  break;  // PES packet
    }
    ( skipped_bytes )++;  // sync skip counter
  }

  skipped_bytes -= 3;

  try{
    n1 = read( this->fd, buf + 4, 2 );
  }catch( std::exception &x ){  // all system exceptions
    CKlearAppErrorException e( __LOC__, x.what() );
    throw e;
  }

  if ( n1 == 2 )
  {
    l = ( buf[ 4 ] << 8 ) + buf[ 5 ];  // PES packet size...
    n1 = 6;  // 4 + 2 bytes read

    if ( l > 0 )
    {
      try{
        n2 = read( this->fd, buf + 6, ( unsigned int ) l );
      }catch( std::exception &x ){  // all system exceptions
        CKlearAppErrorException e( __LOC__, x.what() );
        throw e;
      }
      n1 = ( n2 < 0 ) ? n2 : n1 + n2;
    }
  }
  return n1;
}

long CKlearAppTXTStreamer::readPackHeader( int fd, u_char *buf )
{
  const QString __FUNC__ = "CKlearAppTXTStreamer::readPackHeader( int fd, u_char *buf )";
  long  n1, n2;
  int  len;

  try{
    n1 = read( fd, buf, 10 );
  }catch( std::exception &x ){  // all system exceptions
    CKlearAppErrorException e( __LOC__, x.what() );
    throw e;
  }

  if ( n1 <= 0 ) return n1;
  if ( n1 != 10 ) return -1;

  len = getBits( buf + 9, 0, 5, 3 ); // -- get pack_stuffing_length

  try{
    n2 = read( fd, buf + 10, len );
  }catch( std::exception &x ){  // all system exceptions
    CKlearAppErrorException e( __LOC__, x.what() );
    throw e;
  }

  if ( n2 < 0 )
    return n2;

  return n1 + n2;
}
