/* -*- c++ -*-
 *
 * mlconfig.cpp
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@kmldonkey.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include <qlayout.h>
#include <qgroupbox.h>
#include <qcheckbox.h>

#include <kdebug.h>
#include <klocale.h>

#include "donkeyprotocol.h"
#include "network.h"

#include "mlconfig.h"
#include "kmldonkey.h"
#include "optioneditor.h"



NetworkPage::NetworkPage(QWidget* parent, const char* name)
    : QWidget(parent, name)
{
    QGridLayout* l1 = new QGridLayout(this, 1, 1, 11, 6);
    QGroupBox* box = new QGroupBox(i18n("Enabled Networks"), this);
    box->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
    box->setColumnLayout(0, Qt::Vertical);
    box->layout()->setSpacing(6);
    box->layout()->setMargin(11);
    QVBoxLayout* boxl = new QVBoxLayout(box->layout());
    boxl->setAlignment(Qt::AlignTop);
    
    // Add network checkboxes
    QIntDictIterator<Network> nwit( KMLDonkey::App->donkey->availableNetworks() );
    for ( ; nwit.current(); ++nwit ) {
	Network* net = nwit.current();
	QCheckBox* c = new QCheckBox(net->networkName(), box);
	c->setChecked(net->networkEnabled());
	boxl->addWidget(c);
	connect(c, SIGNAL(toggled(bool)), SLOT(becomeDirty(bool)));
	nwBox.insert(net->networkNo(), c);
    }

    l1->addWidget(box, 1, 1);
    QSpacerItem* spacer = new QSpacerItem( 0, 0, QSizePolicy::Minimum, QSizePolicy::Minimum );
    l1->addItem( spacer, 1, 0 );
    spacer = new QSpacerItem( 0, 0, QSizePolicy::Minimum, QSizePolicy::Minimum );
    l1->addItem( spacer, 1, 2 );
    spacer = new QSpacerItem( 0, 0, QSizePolicy::Minimum, QSizePolicy::Minimum );
    l1->addMultiCell( spacer, 0, 0, 0, 2 );
    spacer = new QSpacerItem( 0, 0, QSizePolicy::Minimum, QSizePolicy::Minimum );
    l1->addMultiCell( spacer, 2, 2, 0, 2 );
}

void NetworkPage::becomeDirty(bool)
{
    emit pageDirty();
}

void NetworkPage::applyOptions(DonkeyProtocol* target)
{
    QIntDictIterator<QCheckBox> it(nwBox);
    for (; it.current(); ++it) {
	Network* net = target->findNetworkNo(it.currentKey());
	if (!net || net->networkEnabled() == it.current()->isChecked())
	    continue;
	target->enableNetwork(it.currentKey(), it.current()->isChecked());
    }
}



MLDonkeyConfigDialog::MLDonkeyConfigDialog(QWidget* parent, const char* name)
    : KDialogBase(TreeList, i18n("MLDonkey Configuration"),
		  Default|Ok|Apply|Cancel, Ok, parent, name)
{
    setRootIsDecorated(true);
    enableButton(Apply, false);
    enableButton(Default, true);
    
    // Add regular option sections
    OptionEditorWidget* page;
    const QValueList<DonkeyOption>& list = KMLDonkey::App->donkey->sectionOptions();
    QValueList<DonkeyOption>::ConstIterator it;
    for (it = list.begin(); it != list.end(); ++it) {
	if (!KMLDonkey::App->advancedConfig && (*it).optionAdvanced())
	    continue;
	page = pages[(*it).optionSection()];
	if (!page) {
	    QWidget* p = (QWidget*)addVBoxPage((*it).optionSection(), (*it).optionSection());
	    page = new OptionEditorWidget(p);
	    connect(page, SIGNAL(listIsDirty()), SLOT(setDirtyDialog()));
	    pages.insert((*it).optionSection(), page);
	}
	page->addOption(*it);
    }
    
    // Add the root network page
    QString networksName = i18n("mldonkey config: network options page", "Networks");
    m_networkPage = new NetworkPage((QWidget*)addVBoxPage(networksName, networksName));
    connect(m_networkPage, SIGNAL(pageDirty()), SLOT(setDirtyDialog()));
    
    // Add the network sub-pages
    const QValueList<DonkeyOption>& plist = KMLDonkey::App->donkey->pluginOptions();
    for (it = plist.begin(); it != plist.end(); ++it) {
	if (!KMLDonkey::App->advancedConfig && (*it).optionAdvanced())
	    continue;
	QStringList section;
	section << networksName << (*it).optionSection();
	page = pages[(*it).optionSection()];
	if (!page) {
	    QWidget* p = (QWidget*)addVBoxPage(section, (*it).optionSection());
	    page = new OptionEditorWidget(p);
	    connect(page, SIGNAL(listIsDirty()), SLOT(setDirtyDialog()));
	    pages.insert((*it).optionSection(), page);
	}
	page->addOption(*it);
    }

    setInitialSize(configDialogSize("MLConfigDialog"));
    connect(this, SIGNAL(finished()), SLOT(closeDialog()));
}

void MLDonkeyConfigDialog::setDirtyDialog()
{
    enableButton(Apply, true);
}

void MLDonkeyConfigDialog::slotApply()
{
    QDictIterator<OptionEditorWidget> it(pages);
    for (; it.current(); ++it)
	it.current()->applyChangedOptions(KMLDonkey::App->donkey);
    m_networkPage->applyOptions(KMLDonkey::App->donkey);
    enableButton(Apply, false);
}

void MLDonkeyConfigDialog::slotOk()
{
    slotApply();
    KDialogBase::slotOk();
}

void MLDonkeyConfigDialog::slotDefault()
{
    kdDebug() << "MLDonkeyConfigDialog::slotDefault() needs to be implemented." << endl;
}

void MLDonkeyConfigDialog::closeDialog()
{
    saveDialogSize("MLConfigDialog");
    delayedDestruct();
}


#include "mlconfig.moc"
